/**
 * @file IEvolutionGeniviDbusHfpSiriSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Send Request Interface for HfpSiri.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS send request for HfpSiri.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_HFP_SIRI_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_HFP_SIRI_SEND_REQUEST_IF_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <string>

#include "asf/core/Types.h"

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusHfpSiriCallbackIf;

/**
 * Send request interface class for HfpSiri.
 */
class IEvolutionGeniviDbusHfpSiriSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusHfpSiriSendRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusHfpSiriCallbackIf* callbackIf, bool enableProxy) = 0;

   /**
    * Handle signal for created device.
    *
    * @param[in] objPath: object path of created device
    */
   virtual void deviceCreated(const ::std::string& objPath) = 0;

   /**
    * Handle signal for removed device.
    *
    * @param[in] objPath: object path of removed device
    */
   virtual void deviceRemoved(const ::std::string& objPath) = 0;

   /**
    * Send GetProperties Request.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendGetPropertiesRequest(const ::std::string& objPath) = 0;

   /**
    * Send Enable Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] vendor_id: vendor id of device
    * @param[in] product_id: product id of device
    * @param[in] version: version of device
    * @param[in] support_type: support type of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendEnableRequest(const ::std::string& objPath, uint16 vendor_id, uint16 product_id, uint16 version, uint16 support_type) = 0;

   /**
    * Send SetNR Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] status: NR status of the device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSetNRRequest(const ::std::string& objPath, uint8 status) = 0;

   /**
    * Send SetProperty Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] property: property name to be updated
    * @param[in] value: property value to be updated
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSetPropertyRequest(const ::std::string& objPath, const ::std::string& property, const bool value) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_HFP_SIRI_SEND_REQUEST_IF_H_
