/**
 * @file IEvolutionGeniviDbusHfpHandsfreeSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Send Request Interface for HfpHandsfree.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS send request for HfpHandsfree.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_HFP_HANDSFREE_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_HFP_HANDSFREE_SEND_REQUEST_IF_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <string>
#include <vector>

#include "asf/core/Types.h"

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusHfpHandsfreeCallbackIf;

/**
 * Send request interface class for HfpHandsfree.
 */
class IEvolutionGeniviDbusHfpHandsfreeSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusHfpHandsfreeSendRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusHfpHandsfreeCallbackIf* callbackIf, bool enableProxy) = 0;

   /**
    * Handle signal for created device.
    *
    * @param[in] objPath: object path of created device
    */
   virtual void deviceCreated(const ::std::string& objPath) = 0;

   /**
    * Handle signal for removed device.
    *
    * @param[in] objPath: object path of removed device
    * @param[in] destroyDirectly: flag to trigger direct proxy destruction or via work item request
    */
   virtual void deviceRemoved(const ::std::string& objPath, const bool destroyDirectly = false) = 0;

   /**
    * Send GetProperties Request.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendGetPropertiesRequest(const ::std::string& objPath) = 0;

   /**
    * Send SetProperty Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] value: value
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSetPropertyRequest(const ::std::string& objPath, const ::std::string& property, const bool value) = 0;

   virtual act_t sendSetPropertyRequest(const ::std::string& objPath, const ::std::string& property, const ::std::string& value) = 0;

   /**
    * Send RequestPhoneNumber Request.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendRequestPhoneNumberRequest(const ::std::string& objPath) = 0;

   /**
    * Send GetSubscriber Request.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendGetSubscriberRequest(const ::std::string& objPath) = 0;

   /**
    * Send SendCustomATCommand Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] data: data
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSendCustomATCommandRequest(const ::std::string& objPath, const ::std::vector< uint8 >& data) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_HFP_HANDSFREE_SEND_REQUEST_IF_H_
