/**
 * @file IEvolutionGeniviDbusAgentTestStubIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Test Stub Interface for Agent.
 *
 * @copyright (C) 2016 - 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS test stub for Agent.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_AGENT_TEST_STUB_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_AGENT_TEST_STUB_IF_H_

#include "CcDbusIfTypes.h"
#include <boost/shared_ptr.hpp>

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusAgentCallbackIf;

/**
 * Test stub interface class for Agent.
 */
class IEvolutionGeniviDbusAgentTestStubIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusAgentTestStubIf() {}

   /**
    * Create stub.
    *
    * @param[in] callback: callback handler
    * @param[in] port: port
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void createStub(IEvolutionGeniviDbusAgentCallbackIf& callback, const ::std::string& port, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Destroy stub.
    *
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void destroyStub(const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Destroy all stubs.
    */
   virtual void destroyAllStubs(void) = 0;

   /**
    * Send ReleaseResponse message.
    *
    * @param[in] act: asynchronous completion token
    */
   virtual void sendReleaseResponse(const act_t act) = 0;

   /**
    * Send ReleaseError message.
    *
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendReleaseError(const std::string& errorName, const std::string& errorMessage, const act_t act) = 0;

   /**
    * Send RequestPinCodeResponse message.
    *
    * @param[in] pincode: pin for an authentication
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestPinCodeResponse(const ::std::string& pincode, const act_t act) = 0;

   /**
    * Send RequestPinCodeError message.
    *
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestPinCodeError(const ::std::string& errorName, const ::std::string& errorMessage, const act_t act) = 0;

   /**
    * Send DisplayPinCodeResponse message.
    *
    * @param[in] act: asynchronous completion token
    */
   virtual void sendDisplayPinCodeResponse(const act_t act) = 0;

   /**
    * Send DisplayPinCodeError message.
    *
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendDisplayPinCodeError(const ::std::string& errorName, const ::std::string& errorMessage, const act_t act) = 0;

   /**
    * Send RequestPasskeyResponse message.
    *
    * @param[in] passkey: passkey for the authentication
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestPasskeyResponse(uint32 passkey, const act_t act) = 0;

   /**
    * Send RequestPasskeyError message.
    *
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestPasskeyError(const ::std::string& errorName, const ::std::string& errorMessage, const act_t act) = 0;

   /**
    * Send RequestConfirmationResponse message.
    *
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestConfirmationResponse(const act_t act) = 0;

   /**
    * Send RequestConfirmationError message.
    *
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestConfirmationError(const ::std::string& errorName, const ::std::string& errorMessage, const act_t act) = 0;

   /**
    * Send CancelResponse message.
    *
    * @param[in] act: asynchronous completion token
    */
   virtual void sendCancelResponse(const act_t act) = 0;

   /**
    * Send CancelRequestResponse message.
    *
    * @param[in] act: asynchronous completion token
    */
   virtual void sendCancelRequestResponse(const act_t act) = 0;

   /**
    * Send AuthorizeServiceResponse message.
    *
    * @param[in] act: asynchronous completion token
    */
   virtual void sendAuthorizeServiceResponse(const act_t act) = 0;

   /**
    * Send AuthorizeServiceError message.
    *
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendAuthorizeServiceError(const ::std::string& errorName, const ::std::string& errorMessage, const act_t act) = 0;

   /**
    * Send RequestAuthorizationResponse message.
    *
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestAuthorizationResponse(const act_t act) = 0;

   /**
    * Send RequestAuthorizationError message.
    *
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestAuthorizationError(const ::std::string& errorName, const ::std::string& errorMessage, const act_t act) = 0;

   /**
    * Send RequestLinkkeyResponse message.
    *
    * @param[in] linkkey: linkey data
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestLinkkeyResponse(const ::std::vector< uint8 >& linkkey, const act_t act) = 0;

   /**
    * Send RequestLinkkeyError message.
    *
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestLinkkeyError(const ::std::string& errorName, const ::std::string& errorMessage, const act_t act) = 0;

   /**
    * Send DisplayPasskeyResponse message.
    *
    * @param[in] act: asynchronous completion token
    */
   virtual void sendDisplayPasskeyResponse(const act_t act) = 0;

   /**
    * Send DisplayPasskeyError message.
    *
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendDisplayPasskeyError(const ::std::string& errorName, const ::std::string& errorMessage, const act_t act) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_AGENT_TEST_STUB_IF_H_
