/**
 * @file IEvolutionGeniviDbusAgentSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Send Request Interface for Agent.
 *
 * @copyright (C) 2016 - 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS send request for Agent.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_AGENT_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_AGENT_SEND_REQUEST_IF_H_

#include "CcDbusIfTypes.h"

namespace ccdbusif {

// class forward declarations
class IDestroyAllProxies;
class IEvolutionGeniviDbusAgentCallbackIf;

/**
 *  Send request interface class for Agent.
 */
class IEvolutionGeniviDbusAgentSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusAgentSendRequestIf() {}

   /**
    * Get destroy interface.
    *
    * @return = destroy interface
    */
   virtual IDestroyAllProxies* getDestroyer(void) = 0;

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableStub: flag to enable/disable stub
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusAgentCallbackIf* callbackIf, const bool enableStub, const unsigned int callbackId, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Create stub interface.
    *
    * @param[in] callbackId: callback id
    * @param[in] createDirectly: flag to trigger direct stub creation or via work item request
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void createStubIf(const unsigned int callbackId, const bool createDirectly = false, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Destroy stub interface.
    *
    * @param[in] callbackId: callback id
    * @param[in] destroyDirectly: flag to trigger direct stub destruction or via work item request
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void destroyStubIf(const unsigned int callbackId, const bool destroyDirectly = false, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Send ReleaseResponse message.
    *
    * @param[in] callbackId: callback id
    * @param[in] act: asynchronous completion token
    */
   virtual void sendReleaseResponse(const unsigned int callbackId, const act_t act) = 0;

   /**
    * Send ReleaseError message.
    *
    * @param[in] callbackId: callback id
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendReleaseError(const unsigned int callbackId, const std::string& errorName, const std::string& errorMessage, const act_t act) = 0;

   /**
    * Send RequestPinCodeResponse message.
    *
    * @param[in] callbackId: callback id
    * @param[in] pincode: pin for an authentication
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestPinCodeResponse(const unsigned int callbackId, const ::std::string& pincode, const act_t act) = 0;

   /**
    * Send RequestPinCodeError message.
    *
    * @param[in] callbackId: callback id
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestPinCodeError(const unsigned int callbackId, const ::std::string& errorName, const ::std::string& errorMessage, const act_t act) = 0;

   /**
    * Send DisplayPinCodeResponse message.
    *
    * @param[in] callbackId: callback id
    * @param[in] act: asynchronous completion token
    */
   virtual void sendDisplayPinCodeResponse(const unsigned int callbackId, const act_t act) = 0;

   /**
    * Send DisplayPinCodeError message.
    *
    * @param[in] callbackId: callback id
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendDisplayPinCodeError(const unsigned int callbackId, const ::std::string& errorName, const ::std::string& errorMessage, const act_t act) = 0;

   /**
    * Send RequestPasskeyResponse message.
    *
    * @param[in] callbackId: callback id
    * @param[in] passkey: passkey for the authentication
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestPasskeyResponse(const unsigned int callbackId, uint32 passkey, const act_t act) = 0;

   /**
    * Send RequestPasskeyError message.
    *
    * @param[in] callbackId: callback id
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestPasskeyError(const unsigned int callbackId, const ::std::string& errorName, const ::std::string& errorMessage, const act_t act) = 0;

   /**
    * Send RequestConfirmationResponse message.
    *
    * @param[in] callbackId: callback id
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestConfirmationResponse(const unsigned int callbackId, const act_t act) = 0;

   /**
    * Send RequestConfirmationError message.
    *
    * @param[in] callbackId: callback id
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestConfirmationError(const unsigned int callbackId, const ::std::string& errorName, const ::std::string& errorMessage, const act_t act) = 0;

   /**
    * Send CancelResponse message.
    *
    * @param[in] callbackId: callback id
    * @param[in] act: asynchronous completion token
    */
   virtual void sendCancelResponse(const unsigned int callbackId, const act_t act) = 0;

   /**
    * Send CancelRequestResponse message.
    *
    * @param[in] callbackId: callback id
    * @param[in] act: asynchronous completion token
    */
   virtual void sendCancelRequestResponse(const unsigned int callbackId, const act_t act) = 0;

   /**
    * Send AuthorizeServiceResponse message.
    *
    * @param[in] callbackId: callback id
    * @param[in] act: asynchronous completion token
    */
   virtual void sendAuthorizeServiceResponse(const unsigned int callbackId, const act_t act) = 0;

   /**
    * Send AuthorizeServiceError message.
    *
    * @param[in] callbackId: callback id
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendAuthorizeServiceError(const unsigned int callbackId, const ::std::string& errorName, const ::std::string& errorMessage, const act_t act) = 0;

   /**
    * Send RequestAuthorizationResponse message.
    *
    * @param[in] callbackId: callback id
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestAuthorizationResponse(const unsigned int callbackId, const act_t act) = 0;

   /**
    * Send RequestAuthorizationError message.
    *
    * @param[in] callbackId: callback id
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestAuthorizationError(const unsigned int callbackId, const ::std::string& errorName, const ::std::string& errorMessage, const act_t act) = 0;

   /**
    * Send RequestLinkkeyResponse message.
    *
    * @param[in] callbackId: callback id
    * @param[in] linkkey: linkey data
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestLinkkeyResponse(const unsigned int callbackId, const ::std::vector< uint8 >& linkkey, const act_t act) = 0;

   /**
    * Send RequestLinkkeyError message.
    *
    * @param[in] callbackId: callback id
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendRequestLinkkeyError(const unsigned int callbackId, const ::std::string& errorName, const ::std::string& errorMessage, const act_t act) = 0;

   /**
    * Send DisplayPasskeyResponse message.
    *
    * @param[in] callbackId: callback id
    * @param[in] act: asynchronous completion token
    */
   virtual void sendDisplayPasskeyResponse(const unsigned int callbackId, const act_t act) = 0;

   /**
    * Send DisplayPasskeyError message.
    *
    * @param[in] callbackId: callback id
    * @param[in] errorName: error name
    * @param[in] errorMessage: error message
    * @param[in] act: asynchronous completion token
    */
   virtual void sendDisplayPasskeyError(const unsigned int callbackId, const ::std::string& errorName, const ::std::string& errorMessage, const act_t act) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_AGENT_SEND_REQUEST_IF_H_
