/**
 * @file IEvolutionGeniviDbusAgentManagerSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Send Request Interface for AgentManager.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS send request for AgentManager.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_AGENT_MANAGER_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_AGENT_MANAGER_SEND_REQUEST_IF_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <string>

#include "asf/core/Types.h"

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusAgentManagerCallbackIf;

/**
 * Send request interface class for AgentManager.
 */
class IEvolutionGeniviDbusAgentManagerSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusAgentManagerSendRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusAgentManagerCallbackIf* callbackIf, bool enableProxy) = 0;

   /**
    * Send RegisterAgent Request.
    * This registers an agent handler.
       An application is not required to register an agent as a default agent is
       available,and for most case, using the default agent is the preferred
       action. Only an application such as a pairing wizard should register its
       own agent. An application can only register one agent as multiple agents
       per application is not supported.
    *
    * @param[in] agent: The object path defines the path of the agent that will
    *                    be called when user input is needed. Every application
    *                    can register its own agent and for all actions triggered
    *                    by that application its agent is used.
    *
    * @param[in] capability: The display and keyboard capability parameter which
    *                        reflects the input and output capabilities of the
    *                        agent. If an empty string is used it will fallback
    *                        to "DisplayYesNo.
    *            capability: possible input:
    *                         DISPLAYONLY: DisplayOnly
    *                         DISPLAYYESNO: DisplayYesNo
    *                         KEYBOARDONLY: KeyboardOnly
    *                         NOINPUTNOOUTPUT: NoInputNoOutput
    *                         KEYBOARDDISPLAY: KeyboardDisplay
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendRegisterAgentRequest(const ::std::string& agent, const ::std::string& capability) = 0;

   /**
    * Send UnregisterAgent Request.
    * This unregisters an agent. The object path parameter must match the value
    * provided when the agent was registered
    *
    * @param[in] agent:The object path of the registered agent
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendUnregisterAgentRequest(const ::std::string& agent) = 0;

   /**
    * Send RequestDefaultAgent Request.
    * This request is to make the application agent the default agent.
    * The application is required to register an agent. Special permission
    * might be required to become the default agent.
    *
    * @param[in] agent:The object path of the registered agent
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendRequestDefaultAgentRequest(const ::std::string& agent) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_AGENT_MANAGER_SEND_REQUEST_IF_H_
