/**
 * @file IEvolutionGeniviDbusAdapterSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Send Request Interface for Adapter.
 *
 * @copyright (C) 2016 - 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS send request for Adapter.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_ADAPTER_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_ADAPTER_SEND_REQUEST_IF_H_

#include "CcDbusIfTypes.h"

namespace ccdbusif {

// class forward declarations
class IDestroyAllProxies;
class IEvolutionGeniviDbusAdapterCallbackIf;

/**
 * Send request interface class for Adapter.
 */
class IEvolutionGeniviDbusAdapterSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusAdapterSendRequestIf() {}

   /**
    * Get destroy interface.
    *
    * @return = destroy interface
    */
   virtual IDestroyAllProxies* getDestroyer(void) = 0;

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusAdapterCallbackIf* callbackIf, const bool enableProxy, const unsigned int callbackId, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Create proxy interface.
    *
    * @param[in] callbackId: callback id
    * @param[in] createDirectly: flag to trigger direct proxy creation or via work item request
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void createProxyIf(const unsigned int callbackId, const bool createDirectly = false, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Destroy proxy interface.
    *
    * @param[in] callbackId: callback id
    * @param[in] destroyDirectly: flag to trigger direct proxy destruction or via work item request
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void destroyProxyIf(const unsigned int callbackId, const bool destroyDirectly = false, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Send StartDiscovery Request.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendStartDiscoveryRequest(const unsigned int callbackId) = 0;

   /**
    * Send StopDiscovery Request.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendStopDiscoveryRequest(const unsigned int callbackId) = 0;

   /**
    * Send RemoveDevice Request.
    *
    * @param[in] callbackId: callback id
    * @param[in] device: device object path
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendRemoveDeviceRequest(const unsigned int callbackId, const ::std::string& device) = 0;

   /**
    * Send CreateDevice Request.
    *
    * @param[in] callbackId: callback id
    * @param[in] address: device address
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendCreateDeviceRequest(const unsigned int callbackId, const ::std::string& address) = 0;

   /**
    * Send SendHCICmd Request.
    *
    * @param[in] callbackId: callback id
    * @param[in] data: HCI command data
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSendHCICmdRequest(const unsigned int callbackId, const ::std::vector< uint8 >& data) = 0;

   /**
    * Send Address Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAddressGet(const unsigned int callbackId) = 0;

   /**
    * Send Class Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendClassGet(const unsigned int callbackId) = 0;

   /**
    * Send Alias Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAliasGet(const unsigned int callbackId) = 0;

   /**
    * Send Alias Set.
    *
    * @param[in] callbackId: callback id
    * @param[in] alias: alias
    */
   virtual void sendAliasSet(const unsigned int callbackId, const ::std::string& alias) = 0;

   /**
    * Send Powered Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendPoweredGet(const unsigned int callbackId) = 0;

   /**
    * Send Powered Set.
    *
    * @param[in] callbackId: callback id
    * @param[in] powered: powered on/off flag
    */
   virtual void sendPoweredSet(const unsigned int callbackId, bool powered) = 0;

   /**
    * Send Discoverable Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDiscoverableGet(const unsigned int callbackId) = 0;

   /**
    * Send Discoverable Set.
    *
    * @param[in] callbackId: callback id
    * @param[in] discoverable: discoverable on/off flag
    */
   virtual void sendDiscoverableSet(const unsigned int callbackId, bool discoverable) = 0;

   /**
    * Send DiscoverableTimeout Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDiscoverableTimeoutGet(const unsigned int callbackId) = 0;

   /**
    * Send DiscoverableTimeout Set.
    *
    * @param[in] callbackId: callback id
    * @param[in] discoverableTimeout: discoverable timeout
    */
   virtual void sendDiscoverableTimeoutSet(const unsigned int callbackId, uint32 discoverableTimeout) = 0;

   /**
    * Send Pairable Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendPairableGet(const unsigned int callbackId) = 0;

   /**
    * Send Pairable Set.
    *
    * @param[in] callbackId: callback id
    * @param[in] pairable: pairable on/off flag
    */
   virtual void sendPairableSet(const unsigned int callbackId, bool pairable) = 0;

   /**
    * Send PairableTimeout Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendPairableTimeoutGet(const unsigned int callbackId) = 0;

   /**
    * Send PairableTimeout Set.
    *
    * @param[in] callbackId: callback id
    * @param[in] pairableTimeout: pairable timeout
    */
   virtual void sendPairableTimeoutSet(const unsigned int callbackId, uint32 pairableTimeout) = 0;

   /**
    * Send Discovering Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDiscoveringGet(const unsigned int callbackId) = 0;

   /**
    * Send UUIDs Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendUUIDsGet(const unsigned int callbackId) = 0;

   /**
    * Send UUIDs Set.
    *
    * @param[in] callbackId: callback id
    * @param[in] uUIDs: UUIDs
    */
   virtual void sendUUIDsSet(const unsigned int callbackId, const ::std::vector< ::std::string >& uUIDs) = 0;

   /**
    * Send WBSMode Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendWBSModeGet(const unsigned int callbackId) = 0;

   /**
    * Send WBSMode Set.
    *
    * @param[in] callbackId: callback id
    * @param[in] wBSMode: WBS mode
    */
   virtual void sendWBSModeSet(const unsigned int callbackId, bool wBSMode) = 0;

   /**
    * Send Connectable Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendConnectableGet(const unsigned int callbackId) = 0;

   /**
    * Send Connectable Set.
    *
    * @param[in] callbackId: callback id
    * @param[in] connectable: connectable on/off flag
    */
   virtual void sendConnectableSet(const unsigned int callbackId, bool connectable) = 0;

   /**
    * Send HCIMode Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendHCIModeGet(const unsigned int callbackId) = 0;

   /**
    * Send HCIMode Set.
    *
    * @param[in] callbackId: callback id
    * @param[in] hCIMode: HCI mode
    */
   virtual void sendHCIModeSet(const unsigned int callbackId, bool hCIMode) = 0;

   /**
    * Send Version Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendVersionGet(const unsigned int callbackId) = 0;

   /**
    * Send TraceLevel Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendTraceLevelGet(const unsigned int callbackId) = 0;

   /**
    * Send TraceLevel Set.
    *
    * @param[in] callbackId: callback id
    * @param[in] traceLevel: trace level setting
    */
   virtual void sendTraceLevelSet(const unsigned int callbackId, const ::std::vector< ::std::string >& traceLevel) = 0;

   /**
    * Send TraceCategory Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendTraceCategoryGet(const unsigned int callbackId) = 0;

   /**
    * Send TraceCategory Set.
    *
    * @param[in] callbackId: callback id
    * @param[in] traceCategory: trace category setting
    */
   virtual void sendTraceCategorySet(const unsigned int callbackId, const ::std::vector< ::std::string >& traceCategory) = 0;

   /**
    * Send AvailableSPPServers Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAvailableSPPServersGet(const unsigned int callbackId) = 0;

   /**
    * Send AvailableSPPServers Set.
    *
    * @param[in] callbackId: callback id
    * @param[in] availableSPPServers: available SPP server setting
    */
   virtual void sendAvailableSPPServersSet(const unsigned int callbackId, const ::std::vector< ::std::string >& availableSPPServers) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_ADAPTER_SEND_REQUEST_IF_H_
