/**
 * @file EvolutionGeniviDbusUuids.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Properties for UUIDs.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS Properties for UUIDs.
 */

#ifndef _EVOLUTION_GENIVI_DBUS_UUIDS_H_
#define _EVOLUTION_GENIVI_DBUS_UUIDS_H_

#include "DbusProperties.h"
#include "EvolutionGeniviDbusTypes.h"

namespace ccdbusif {
namespace evolution {

/**
 * UUIDs properties class for Evolution Genivi interface.
 */
class EvolutionGeniviDbusUuids : public DbusProperties
{
public:
   EvolutionGeniviDbusUuids();
   virtual ~EvolutionGeniviDbusUuids();

   UUID getUuid2Enum(const ::std::string& value) const;
   const ::std::string& getUuid2String(const UUID value) const;
   const ::std::string& getUuid2StringTrace(const UUID value) const;
   bool isValidUuid(const ::std::string& value, const bool ignoreCaseSensitive = true) const;
   bool isShortUuid(UUID& uuid, const ::std::string& value) const;

private:
   const ::std::string _uuid32; /**< 32 bit UUID */

   // see https://www.bluetooth.com/specifications/assigned-numbers/service-discovery
   const ::std::string _uuidBase; /**< BASE_UUID */
   // protocols
   /*
   SDP 0x0001 Bluetooth Core Specification
   UDP 0x0002 [NO USE BY PROFILES]
   RFCOMM 0x0003 RFCOMM with TS 07.10
   TCP 0x0004 [NO USE BY PROFILES]
   TCS-BIN 0x0005 Telephony Control Specification / TCS Binary [DEPRECATED]
   TCS-AT 0x0006 [NO USE BY PROFILES]
   ATT 0x0007 Attribute Protocol
   OBEX 0x0008 IrDA Interoperability
   IP 0x0009 [NO USE BY PROFILES]
   FTP 0x000A [NO USE BY PROFILES]
   HTTP 0x000C [NO USE BY PROFILES]
   WSP 0x000E [NO USE BY PROFILES]
   BNEP 0x000F Bluetooth Network Encapsulation Protocol (BNEP)
   UPNP 0x0010 Extended Service Discovery Profile (ESDP) [DEPRECATED]
   HIDP 0x0011 Human Interface Device Profile (HID)
   HardcopyControlChannel 0x0012 Hardcopy Cable Replacement Profile (HCRP)
   HardcopyDataChannel 0x0014 See Hardcopy Cable Replacement Profile (HCRP)
   HardcopyNotification 0x0016 Hardcopy Cable Replacement Profile (HCRP)
   AVCTP 0x0017 Audio/Video Control Transport Protocol (AVCTP)
   AVDTP 0x0019 Audio/Video Distribution Transport Protocol (AVDTP)
   CMTP 0x001B Common ISDN Access Profile (CIP) [DEPRECATED]
   MCAPControlChannel 0x001E Multi-Channel Adaptation Protocol (MCAP)
   MCAPDataChannel 0x001F Multi-Channel Adaptation Protocol (MCAP)
   L2CAP 0x0100 Bluetooth Core Specification
    */
   const ::std::string _uuidSDP; /**< Bluetooth Core Specification */
   const ::std::string _uuidUDP; /**< [NO USE BY PROFILES] */
   const ::std::string _uuidRFCOMM; /**< RFCOMM with TS 07.10 */
   const ::std::string _uuidTCP; /**< [NO USE BY PROFILES] */
   const ::std::string _uuidTCS_BIN; /**< Telephony Control Specification / TCS Binary [DEPRECATED] */
   const ::std::string _uuidTCS_AT; /**< [NO USE BY PROFILES] */
   const ::std::string _uuidATT; /**< Attribute Protocol */
   const ::std::string _uuidOBEX; /**< IrDA Interoperability */
   const ::std::string _uuidIP; /**< [NO USE BY PROFILES] */
   const ::std::string _uuidFTP; /**< [NO USE BY PROFILES] */
   const ::std::string _uuidHTTP; /**< [NO USE BY PROFILES] */
   const ::std::string _uuidWSP; /**< [NO USE BY PROFILES] */
   const ::std::string _uuidBNEP; /**< Bluetooth Network Encapsulation Protocol (BNEP) */
   const ::std::string _uuidUPNP; /**< Extended Service Discovery Profile (ESDP) [DEPRECATED] */
   const ::std::string _uuidHIDP; /**< Human Interface Device Profile (HID) */
   const ::std::string _uuidHardcopyControlChannel; /**< Hardcopy Cable Replacement Profile (HCRP) */
   const ::std::string _uuidHardcopyDataChannel; /**< See Hardcopy Cable Replacement Profile (HCRP) */
   const ::std::string _uuidHardcopyNotification; /**< Hardcopy Cable Replacement Profile (HCRP) */
   const ::std::string _uuidAVCTP; /**< Audio/Video Control Transport Protocol (AVCTP) */
   const ::std::string _uuidAVDTP; /**< Audio/Video Distribution Transport Protocol (AVDTP) */
   const ::std::string _uuidCMTP; /**< Common ISDN Access Profile (CIP) [DEPRECATED] */
   const ::std::string _uuidMCAPControlChannel; /**< Multi-Channel Adaptation Protocol (MCAP) */
   const ::std::string _uuidMCAPDataChannel; /**< Multi-Channel Adaptation Protocol (MCAP) */
   const ::std::string _uuidL2CAP; /**< Bluetooth Core Specification */
   // services
   /*
   ServiceDiscoveryServerServiceClassID 0x1000 Bluetooth Core Specification
   BrowseGroupDescriptorServiceClassID 0x1001 Bluetooth Core Specification
   SerialPort 0x1101 Serial Port Profile (SPP)
   LANAccessUsingPPP 0x1102 LAN Access Profile [DEPRECATED]
   DialupNetworking 0x1103 Dial-up Networking Profile (DUN)
   IrMCSync 0x1104 Synchronization Profile (SYNC)
   OBEXObjectPush 0x1105 Object Push Profile (OPP)
   OBEXFileTransfer 0x1106 File Transfer Profile (FTP)
   IrMCSyncCommand 0x1107 Synchronization Profile (SYNC)
   Headset 0x1108 Headset Profile (HSP)
   CordlessTelephony 0x1109 Cordless Telephony Profile (CTP) [DEPRECATED]
   AudioSource 0x110A Advanced Audio Distribution Profile (A2DP)
   AudioSink 0x110B Advanced Audio Distribution Profile (A2DP)
   A/V_RemoteControlTarget 0x110C Audio/Video Remote Control Profile (AVRCP)
   AdvancedAudioDistribution 0x110D Advanced Audio Distribution Profile (A2DP)
   A/V_RemoteControl 0x110E Audio/Video Remote Control Profile (AVRCP)
   A/V_RemoteControlController 0x110F Audio/Video Remote Control Profile (AVRCP)
   Intercom 0x1110 Intercom Profile (ICP) [DEPRECATED]
   Fax 0x1111 Fax Profile (FAX) [DEPRECATED]
   Headset - Audio Gateway (AG) 0x1112 Headset Profile (HSP)
   WAP 0x1113 Interoperability Requirements for Bluetooth technology as a WAP, Bluetooth SIG [DEPRECATED]
   WAP_CLIENT 0x1114 Interoperability Requirements for Bluetooth technology as a WAP, Bluetooth SIG [DEPRECATED]
   PANU 0x1115 Personal Area Networking Profile (PAN)
   NAP 0x1116 Personal Area Networking Profile (PAN)
   GN 0x1117 Personal Area Networking Profile (PAN)
   DirectPrinting 0x1118 Basic Printing Profile (BPP)
   ReferencePrinting 0x1119 See Basic Printing Profile (BPP)
   Basic Imaging Profile 0x111A Basic Imaging Profile (BIP)
   ImagingResponder 0x111B Basic Imaging Profile (BIP)
   ImagingAutomaticArchive 0x111C Basic Imaging Profile (BIP)
   ImagingReferencedObjects 0x111D Basic Imaging Profile (BIP)
   Handsfree 0x111E Hands-Free Profile (HFP)
   HandsfreeAudioGateway 0x111F Hands-free Profile (HFP)
   DirectPrintingReferenceObjectsService 0x1120 Basic Printing Profile (BPP)
   ReflectedUI 0x1121 Basic Printing Profile (BPP)
   BasicPrinting 0x1122 Basic Printing Profile (BPP)
   PrintingStatus 0x1123 Basic Printing Profile (BPP)
   HumanInterfaceDeviceService 0x1124 Human Interface Device (HID)
   HardcopyCableReplacement 0x1125 Hardcopy Cable Replacement Profile (HCRP)
   HCR_Print 0x1126 Hardcopy Cable Replacement Profile (HCRP)
   HCR_Scan 0x1127 Hardcopy Cable Replacement Profile (HCRP)
   Common_ISDN_Access 0x1128 Common ISDN Access Profile (CIP) [DEPRECATED]
   SIM_Access 0x112D SIM Access Profile (SAP)
   Phonebook Access - PCE 0x112E Phonebook Access Profile (PBAP)
   Phonebook Access - PSE 0x112F Phonebook Access Profile (PBAP)
   Phonebook Access 0x1130 Phonebook Access Profile (PBAP)
   Headset - HS 0x1131 Headset Profile (HSP)
   Message Access Server 0x1132 Message Access Profile (MAP)
   Message Notification Server 0x1133 Message Access Profile (MAP)
   Message Access Profile 0x1134 Message Access Profile (MAP)
   GNSS 0x1135 Global Navigation Satellite System Profile (GNSS)
   GNSS_Server 0x1136 Global Navigation Satellite System Profile (GNSS)
   3D Display 0x1137 3D Synchronization Profile (3DSP)
   3D Glasses 0x1138 3D Synchronization Profile (3DSP)
   3D Synchronization 0x1139 3D Synchronization Profile (3DSP)
   MPS Profile UUID 0x113A Multi-Profile Specification (MPS)
   MPS SC UUID 0x113B Multi-Profile Specification (MPS)
   CTN Access Service 0x113C Calendar, Task, and Notes (CTN) Profile
   CTN Notification Service 0x113D Calendar, Tasks, and Notes (CTN) Profile
   CTN Profile 0x113E Calendar, Tasks, and Notes (CTN) Profile
   PnPInformation 0x1200 Device Identification (DID)
   GenericNetworking 0x1201 N/A
   GenericFileTransfer 0x1202 N/A
   GenericAudio 0x1203 N/A
   GenericTelephony 0x1204 N/A
   UPNP_Service 0x1205 Enhanced Service Discovery Profile (ESDP) [DEPRECATED]
   UPNP_IP_Service 0x1206 Enhanced Service Discovery Profile (ESDP) [DEPRECATED]
   ESDP_UPNP_IP_PAN 0x1300 Enhanced Service Discovery Profile (ESDP) [DEPRECATED]
   ESDP_UPNP_IP_LAP 0x1301 Enhanced Service Discovery Profile (ESDP)[DEPRECATED]
   ESDP_UPNP_L2CAP 0x1302 Enhanced Service Discovery Profile (ESDP)[DEPRECATED]
   VideoSource 0x1303 Video Distribution Profile (VDP)
   VideoSink 0x1304 Video Distribution Profile (VDP)
   VideoDistribution 0x1305 Video Distribution Profile (VDP)
   HDP 0x1400 Health Device Profile
   HDP Source 0x1401 Health Device Profile (HDP)
   HDP Sink 0x1402 Health Device Profile (HDP)
    */
   const ::std::string _uuidServiceDiscoveryServerServiceClassID; /**< Bluetooth Core Specification */
   const ::std::string _uuidBrowseGroupDescriptorServiceClassID; /**< Bluetooth Core Specification */
   const ::std::string _uuidSerialPort; /**< Serial Port Profile (SPP) */
   const ::std::string _uuidLANAccessUsingPPP; /**< LAN Access Profile [DEPRECATED] */
   const ::std::string _uuidDialupNetworking; /**< Dial-up Networking Profile (DUN) */
   const ::std::string _uuidIrMCSync; /**< Synchronization Profile (SYNC) */
   const ::std::string _uuidOBEXObjectPush; /**< Object Push Profile (OPP) */
   const ::std::string _uuidOBEXFileTransfer; /**< File Transfer Profile (FTP) */
   const ::std::string _uuidIrMCSyncCommand; /**< Synchronization Profile (SYNC) */
   const ::std::string _uuidHeadset; /**< Headset Profile (HSP) */
   const ::std::string _uuidCordlessTelephony; /**< Cordless Telephony Profile (CTP) [DEPRECATED] */
   const ::std::string _uuidAudioSource; /**< Advanced Audio Distribution Profile (A2DP) */
   const ::std::string _uuidAudioSink; /**< Advanced Audio Distribution Profile (A2DP) */
   const ::std::string _uuidA_V_RemoteControlTarget; /**< Audio/Video Remote Control Profile (AVRCP) */
   const ::std::string _uuidAdvancedAudioDistribution; /**< Advanced Audio Distribution Profile (A2DP) */
   const ::std::string _uuidA_V_RemoteControl; /**< Audio/Video Remote Control Profile (AVRCP) */
   const ::std::string _uuidA_V_RemoteControlController; /**< Audio/Video Remote Control Profile (AVRCP) */
   const ::std::string _uuidIntercom; /**< Intercom Profile (ICP) [DEPRECATED] */
   const ::std::string _uuidFax; /**< Fax Profile (FAX) [DEPRECATED] */
   const ::std::string _uuidHeadset_Audio_Gateway; /**< Headset Profile (HSP) */
   const ::std::string _uuidWAP; /**< Interoperability Requirements for Bluetooth technology as a WAP, Bluetooth SIG [DEPRECATED] */
   const ::std::string _uuidWAP_CLIENT; /**< Interoperability Requirements for Bluetooth technology as a WAP, Bluetooth SIG [DEPRECATED] */
   const ::std::string _uuidPANU; /**< Personal Area Networking Profile (PAN) - PAN User */
   const ::std::string _uuidNAP; /**< Personal Area Networking Profile (PAN) - Network Access Point */
   const ::std::string _uuidGN; /**< Personal Area Networking Profile (PAN) - Group Ad-hoc Network */
   const ::std::string _uuidDirectPrinting; /**< Basic Printing Profile (BPP) */
   const ::std::string _uuidReferencePrinting; /**< See Basic Printing Profile (BPP) */
   const ::std::string _uuidBasic_Imaging_Profile; /**< Basic Imaging Profile (BIP) */
   const ::std::string _uuidImagingResponder; /**< Basic Imaging Profile (BIP) */
   const ::std::string _uuidImagingAutomaticArchive; /**< Basic Imaging Profile (BIP) */
   const ::std::string _uuidImagingReferencedObjects; /**< Basic Imaging Profile (BIP) */
   const ::std::string _uuidHandsfree; /**< Hands-Free Profile (HFP) */
   const ::std::string _uuidHandsfreeAudioGateway; /**< Hands-free Profile (HFP) */
   const ::std::string _uuidDirectPrintingReferenceObjectsService; /**< Basic Printing Profile (BPP) */
   const ::std::string _uuidReflectedUI; /**< Basic Printing Profile (BPP) */
   const ::std::string _uuidBasicPrinting; /**< Basic Printing Profile (BPP) */
   const ::std::string _uuidPrintingStatus; /**< Basic Printing Profile (BPP) */
   const ::std::string _uuidHumanInterfaceDeviceService; /**< Human Interface Device (HID) */
   const ::std::string _uuidHardcopyCableReplacement; /**< Hardcopy Cable Replacement Profile (HCRP) */
   const ::std::string _uuidHCR_Print; /**< Hardcopy Cable Replacement Profile (HCRP) */
   const ::std::string _uuidHCR_Scan; /**< Hardcopy Cable Replacement Profile (HCRP) */
   const ::std::string _uuidCommon_ISDN_Access; /**< Common ISDN Access Profile (CIP) [DEPRECATED] */
   const ::std::string _uuidSIM_Access; /**< SIM Access Profile (SAP) */
   const ::std::string _uuidPhonebook_Access_PCE; /**< Phonebook Access Profile (PBAP) */
   const ::std::string _uuidPhonebook_Access_PSE; /**< Phonebook Access Profile (PBAP) */
   const ::std::string _uuidPhonebook_Access; /**< Phonebook Access Profile (PBAP) */
   const ::std::string _uuidHeadset_HS; /**< Headset Profile (HSP) */
   const ::std::string _uuidMessage_Access_Server; /**< Message Access Profile (MAP) */
   const ::std::string _uuidMessage_Notification_Server; /**< Message Access Profile (MAP) */
   const ::std::string _uuidMessage_Access_Profile; /**< Message Access Profile (MAP) */
   const ::std::string _uuidGNSS; /**< Global Navigation Satellite System Profile (GNSS) */
   const ::std::string _uuidGNSS_Server; /**< Global Navigation Satellite System Profile (GNSS) */
   const ::std::string _uuid3D_Display; /**< 3D Synchronization Profile (3DSP) */
   const ::std::string _uuid3D_Glasses; /**< 3D Synchronization Profile (3DSP) */
   const ::std::string _uuid3D_Synchronization; /**< 3D Synchronization Profile (3DSP) */
   const ::std::string _uuidMPS_Profile; /**< Multi-Profile Specification (MPS) */
   const ::std::string _uuidMPS_SC; /**< Multi-Profile Specification (MPS) */
   const ::std::string _uuidCTN_Access_Service; /**< Calendar, Task, and Notes (CTN) Profile */
   const ::std::string _uuidCTN_Notification_Service; /**< Calendar, Tasks, and Notes (CTN) Profile */
   const ::std::string _uuidCTN_Profile; /**< Calendar, Tasks, and Notes (CTN) Profile */
   const ::std::string _uuidPnPInformation; /**< Device Identification (DID) */
   const ::std::string _uuidGenericNetworking; /**< N/A */
   const ::std::string _uuidGenericFileTransfer; /**< N/A */
   const ::std::string _uuidGenericAudio; /**< N/A */
   const ::std::string _uuidGenericTelephony; /**< N/A */
   const ::std::string _uuidUPNP_Service; /**< Enhanced Service Discovery Profile (ESDP) [DEPRECATED] */
   const ::std::string _uuidUPNP_IP_Service; /**< Enhanced Service Discovery Profile (ESDP) [DEPRECATED] */
   const ::std::string _uuidESDP_UPNP_IP_PAN; /**< Enhanced Service Discovery Profile (ESDP) [DEPRECATED] */
   const ::std::string _uuidESDP_UPNP_IP_LAP; /**< Enhanced Service Discovery Profile (ESDP)[DEPRECATED] */
   const ::std::string _uuidESDP_UPNP_L2CAP; /**< Enhanced Service Discovery Profile (ESDP)[DEPRECATED] */
   const ::std::string _uuidVideoSource; /**< Video Distribution Profile (VDP) */
   const ::std::string _uuidVideoSink; /**< Video Distribution Profile (VDP) */
   const ::std::string _uuidVideoDistribution; /**< Video Distribution Profile (VDP) */
   const ::std::string _uuidHDP; /**< Health Device Profile */
   const ::std::string _uuidHDP_Source; /**< Health Device Profile (HDP) */
   const ::std::string _uuidHDP_Sink; /**< Health Device Profile (HDP) */
   // SyncML
   const ::std::string _uuidSyncMLService; /**< SyncML */

   // trace values
   const ::std::string _traceBASE; /**< trace value for BASE */
   const ::std::string _traceSDP; /**< trace value for SDP */
   const ::std::string _traceUDP; /**< trace value for UDP */
   const ::std::string _traceRFCOMM; /**< trace value for RFCOMM */
   const ::std::string _traceTCP; /**< trace value for TCP */
   const ::std::string _traceTCS_BIN; /**< trace value for TCS_BIN */
   const ::std::string _traceTCS_AT; /**< trace value for TCS_AT */
   const ::std::string _traceATT; /**< trace value for ATT */
   const ::std::string _traceOBEX; /**< trace value for OBEX */
   const ::std::string _traceIP; /**< trace value for IP */
   const ::std::string _traceFTP; /**< trace value for FTP */
   const ::std::string _traceHTTP; /**< trace value for HTTP */
   const ::std::string _traceWSP; /**< trace value for WSP */
   const ::std::string _traceBNEP; /**< trace value for BNEP */
   const ::std::string _traceUPNP; /**< trace value for UPNP */
   const ::std::string _traceHIDP; /**< trace value for HIDP */
   const ::std::string _traceHardcopyControlChannel; /**< trace value for HardcopyControlChannel */
   const ::std::string _traceHardcopyDataChannel; /**< trace value for HardcopyDataChannel */
   const ::std::string _traceHardcopyNotification; /**< trace value for HardcopyNotification */
   const ::std::string _traceAVCTP; /**< trace value for AVCTP */
   const ::std::string _traceAVDTP; /**< trace value for AVDTP */
   const ::std::string _traceCMTP; /**< trace value for CMTP */
   const ::std::string _traceMCAPControlChannel; /**< trace value for MCAPControlChannel */
   const ::std::string _traceMCAPDataChannel; /**< trace value for MCAPDataChannel */
   const ::std::string _traceL2CAP; /**< trace value for L2CAP */
   const ::std::string _traceServiceDiscoveryServerServiceClassID; /**< trace value for ServiceDiscoveryServerServiceClassID */
   const ::std::string _traceBrowseGroupDescriptorServiceClassID; /**< trace value for BrowseGroupDescriptorServiceClassID */
   const ::std::string _traceSerialPort; /**< trace value for SerialPort */
   const ::std::string _traceLANAccessUsingPPP; /**< trace value for LANAccessUsingPPP */
   const ::std::string _traceDialupNetworking; /**< trace value for DialupNetworking */
   const ::std::string _traceIrMCSync; /**< trace value for IrMCSync */
   const ::std::string _traceOBEXObjectPush; /**< trace value for OBEXObjectPush */
   const ::std::string _traceOBEXFileTransfer; /**< trace value for OBEXFileTransfer */
   const ::std::string _traceIrMCSyncCommand; /**< trace value for IrMCSyncCommand */
   const ::std::string _traceHeadset; /**< trace value for Headset */
   const ::std::string _traceCordlessTelephony; /**< trace value for CordlessTelephony */
   const ::std::string _traceAudioSource; /**< trace value for AudioSource */
   const ::std::string _traceAudioSink; /**< trace value for AudioSink */
   const ::std::string _traceA_V_RemoteControlTarget; /**< trace value for A_V_RemoteControlTarget */
   const ::std::string _traceAdvancedAudioDistribution; /**< trace value for AdvancedAudioDistribution */
   const ::std::string _traceA_V_RemoteControl; /**< trace value for A_V_RemoteControl */
   const ::std::string _traceA_V_RemoteControlController; /**< trace value for A_V_RemoteControlController */
   const ::std::string _traceIntercom; /**< trace value for Intercom */
   const ::std::string _traceFax; /**< trace value for Fax */
   const ::std::string _traceHeadset_Audio_Gateway; /**< trace value for Headset_Audio_Gateway */
   const ::std::string _traceWAP; /**< trace value for WAP */
   const ::std::string _traceWAP_CLIENT; /**< trace value for WAP_CLIENT */
   const ::std::string _tracePANU; /**< trace value for PANU */
   const ::std::string _traceNAP; /**< trace value for NAP */
   const ::std::string _traceGN; /**< trace value for GN */
   const ::std::string _traceDirectPrinting; /**< trace value for DirectPrinting */
   const ::std::string _traceReferencePrinting; /**< trace value for ReferencePrinting */
   const ::std::string _traceBasic_Imaging_Profile; /**< trace value for Basic_Imaging_Profile */
   const ::std::string _traceImagingResponder; /**< trace value for ImagingResponder */
   const ::std::string _traceImagingAutomaticArchive; /**< trace value for ImagingAutomaticArchive */
   const ::std::string _traceImagingReferencedObjects; /**< trace value for ImagingReferencedObjects */
   const ::std::string _traceHandsfree; /**< trace value for Handsfree */
   const ::std::string _traceHandsfreeAudioGateway; /**< trace value for HandsfreeAudioGateway */
   const ::std::string _traceDirectPrintingReferenceObjectsService; /**< trace value for DirectPrintingReferenceObjectsService */
   const ::std::string _traceReflectedUI; /**< trace value for ReflectedUI */
   const ::std::string _traceBasicPrinting; /**< trace value for BasicPrinting */
   const ::std::string _tracePrintingStatus; /**< trace value for PrintingStatus */
   const ::std::string _traceHumanInterfaceDeviceService; /**< trace value for HumanInterfaceDeviceService */
   const ::std::string _traceHardcopyCableReplacement; /**< trace value for HardcopyCableReplacement */
   const ::std::string _traceHCR_Print; /**< trace value for HCR_Print */
   const ::std::string _traceHCR_Scan; /**< trace value for HCR_Scan */
   const ::std::string _traceCommon_ISDN_Access; /**< trace value for Common_ISDN_Access */
   const ::std::string _traceSIM_Access; /**< trace value for SIM_Access */
   const ::std::string _tracePhonebook_Access_PCE; /**< trace value for Phonebook_Access_PCE */
   const ::std::string _tracePhonebook_Access_PSE; /**< trace value for Phonebook_Access_PSE */
   const ::std::string _tracePhonebook_Access; /**< trace value for Phonebook_Access */
   const ::std::string _traceHeadset_HS; /**< trace value for Headset_HS */
   const ::std::string _traceMessage_Access_Server; /**< trace value for Message_Access_Server */
   const ::std::string _traceMessage_Notification_Server; /**< trace value for Message_Notification_Server */
   const ::std::string _traceMessage_Access_Profile; /**< trace value for Message_Access_Profile */
   const ::std::string _traceGNSS; /**< trace value for GNSS */
   const ::std::string _traceGNSS_Server; /**< trace value for GNSS_Server */
   const ::std::string _trace3D_Display; /**< trace value for 3D_Display */
   const ::std::string _trace3D_Glasses; /**< trace value for 3D_Glasses */
   const ::std::string _trace3D_Synchronization; /**< trace value for 3D_Synchronization */
   const ::std::string _traceMPS_Profile; /**< trace value for MPS_Profile */
   const ::std::string _traceMPS_SC; /**< trace value for MPS_SC */
   const ::std::string _traceCTN_Access_Service; /**< trace value for CTN_Access_Service */
   const ::std::string _traceCTN_Notification_Service; /**< trace value for CTN_Notification_Service */
   const ::std::string _traceCTN_Profile; /**< trace value for CTN_Profile */
   const ::std::string _tracePnPInformation; /**< trace value for PnPInformation */
   const ::std::string _traceGenericNetworking; /**< trace value for GenericNetworking */
   const ::std::string _traceGenericFileTransfer; /**< trace value for GenericFileTransfer */
   const ::std::string _traceGenericAudio; /**< trace value for GenericAudio */
   const ::std::string _traceGenericTelephony; /**< trace value for GenericTelephony */
   const ::std::string _traceUPNP_Service; /**< trace value for UPNP_Service */
   const ::std::string _traceUPNP_IP_Service; /**< trace value for UPNP_IP_Service */
   const ::std::string _traceESDP_UPNP_IP_PAN; /**< trace value for ESDP_UPNP_IP_PAN */
   const ::std::string _traceESDP_UPNP_IP_LAP; /**< trace value for ESDP_UPNP_IP_LAP */
   const ::std::string _traceESDP_UPNP_L2CAP; /**< trace value for ESDP_UPNP_L2CAP */
   const ::std::string _traceVideoSource; /**< trace value for VideoSource */
   const ::std::string _traceVideoSink; /**< trace value for VideoSink */
   const ::std::string _traceVideoDistribution; /**< trace value for VideoDistribution */
   const ::std::string _traceHDP; /**< trace value for HDP */
   const ::std::string _traceHDP_Source; /**< trace value for HDP_Source */
   const ::std::string _traceHDP_Sink; /**< trace value for HDP_Sink */
   const ::std::string _traceSYNCML; /**< trace value for SYNCML */

   ::std::map< int, const std::string* > _enum2TraceList; /**< map with enum + trace value pairs */

   void checkUuidList(void) const;
   bool isUnknownShortUuid(const ::std::string& value) const;
};

} //evolution
} //ccdbusif

#endif //_EVOLUTION_GENIVI_DBUS_UUIDS_H_
