/**
 * @file EcNrWrapper.cpp
 *
 * @par SW-Component
 * Wrapper
 *
 * @brief EcNr wrapper.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Wrapper implementation for EcNr.
 */

#include "EcNrWrapper.h"
#include "EcNrWorkItem.h"

#include "ICcDbusIfController.h"
#include "ICcDbusIfControllerClient.h"
#include "ICcDbusIfControllerEcNr.h"
#include "IEcNrDbusServiceSendRequestIf.h"
#include "EcNrDbusServiceCallbackIf.h"
#include "TimeoutSemaphore.h"

#define FW_S_IMPORT_INTERFACE_ASSERT
#include "framework_if_if.h"

// Following defines should never be successful, else you have programmed something wrong
// #error_indication
#define CONTROLLER_CLIENT_NULL_CHECK \
if(NULL == _ccDbusIfControllerClient)\
{\
   FW_NORMAL_ASSERT_ALWAYS();\
   return ECNR_ERROR;\
}

#define ECNR_SERVICE_SENDIF_NULL_CHECK \
if(NULL == _ecnrServiceSendIf)\
{\
   FW_NORMAL_ASSERT_ALWAYS();\
   return ECNR_ERROR;\
}

#define DBUSIF_AVAILABLE_CHECK \
if(true != _dbusIfAvailable)\
{\
   FW_NORMAL_ASSERT_ALWAYS();\
   return ECNR_ERROR;\
}

#define SERVICE_AVAILABLE_CHECK \
if(true != waitForServiceAvailable())\
{\
   return ECNR_ERROR;\
}

namespace ccdbusif {

inline static EcNrWorkItem* getNewEcNrWorkItem(void) { return new EcNrWorkItem(); }

EcNrWrapper::EcNrWrapper()
{
   _ecnrServiceCallbackIf = NULL;
   _ecnrServiceSendIf = NULL;
   _ccDbusIfControllerClient = NULL;
   _dbusIfAvailable = false;
   _serviceAvailable = false;
   _serviceSemaphore = new TimeoutSemaphore();
}

EcNrWrapper::~EcNrWrapper()
{
   if(NULL != _ecnrServiceCallbackIf)
   {
      delete _ecnrServiceCallbackIf;
   }
   _ecnrServiceSendIf = NULL;
   _ccDbusIfControllerClient = NULL;
   if(NULL != _serviceSemaphore)
   {
      delete _serviceSemaphore;
   }
}

EcnrErrorCode EcNrWrapper::init(IN ::ccdbusif::ICcDbusIfController* controller)
{
   if(NULL == controller)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return ECNR_ERROR;
   }

   // check if interfaces are already set (checking _ccDbusIfControllerClient is enough)
   if(NULL != _ccDbusIfControllerClient)
   {
      // already set
      return ECNR_OK;
   }

   EcnrErrorCode errCode = ECNR_OK;

   _ccDbusIfControllerClient = controller->getCcDbusIfControllerClient();

   ::ccdbusif::ICcDbusIfControllerEcNr* ecnrIf = controller->getCcDbusIfControllerEcNr();

   if(NULL == ecnrIf)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      errCode = ECNR_ERROR;
   }
   else
   {
      // get ECNR sending interface
      _ecnrServiceSendIf = ecnrIf->getEcNrServiceProxy();

      if(NULL == _ecnrServiceSendIf)
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         errCode = ECNR_ERROR;
      }

      // create callback handler, pass this to constructor; within callback handler call initializeResultCB()
      if(NULL == _ecnrServiceCallbackIf)
      {
         _ecnrServiceCallbackIf = new EcNrDbusServiceCallbackIf(this);
      }

      if(NULL == _ecnrServiceCallbackIf)
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         errCode = ECNR_ERROR;
      }
   }

   if(ECNR_OK != errCode)
   {
      // reset all
      _ccDbusIfControllerClient = NULL;
      _ecnrServiceSendIf = NULL;

      // do not delete callback handler
   }
   else
   {
      // set marker that D-Bus interface is available
      _dbusIfAvailable = true;

      // set callback handler
      if(NULL != _serviceSemaphore)
      {
         _serviceSemaphore->reset();
      }
      if((NULL != _ecnrServiceSendIf) && (NULL != _ecnrServiceCallbackIf))
      {
         // register callback handler
         _ecnrServiceSendIf->setCallbackIf(_ecnrServiceCallbackIf, true);
      }
   }

   return errCode;
}

EcnrErrorCode EcNrWrapper::deInit(void)
{
   // check if interfaces are already reset (checking _ccDbusIfControllerClient is enough)
   if(NULL == _ccDbusIfControllerClient)
   {
      // already reset
      return ECNR_OK;
   }

   // reset all
   _ccDbusIfControllerClient = NULL;
   _ecnrServiceSendIf = NULL;
   // reset marker that D-Bus interface is available
   _dbusIfAvailable = false;

   // do not delete callback handler

   return ECNR_OK;
}

EcnrErrorCode EcNrWrapper::initialize(IN const EcnrAppId appId, IN const EcnrConfigurationId configurationId)
{
   CONTROLLER_CLIENT_NULL_CHECK;
   ECNR_SERVICE_SENDIF_NULL_CHECK;
   DBUSIF_AVAILABLE_CHECK;
   SERVICE_AVAILABLE_CHECK;

   // create work item
   EcNrWorkItem* workitem = getNewEcNrWorkItem();

   if(workitem)
   {
      // add parameter
      workitem->setHandleSendRequest(true);

      workitem->setServiceIf(_ecnrServiceSendIf);
      workitem->setWrapper(this);

      workitem->setEventType(ECNR_EVENT_INITIALIZE);
      workitem->setAppId(appId);
      workitem->setConfigurationId(configurationId);

      // push work item
      _ccDbusIfControllerClient->pushWorkItem(workitem);

      return ECNR_OK;
   }

   return ECNR_ERROR;
}

EcnrErrorCode EcNrWrapper::destroy(IN const EcnrAppId appId)
{
   CONTROLLER_CLIENT_NULL_CHECK;
   ECNR_SERVICE_SENDIF_NULL_CHECK;
   DBUSIF_AVAILABLE_CHECK;
   SERVICE_AVAILABLE_CHECK;

   // create work item
   EcNrWorkItem* workitem = getNewEcNrWorkItem();

   if(workitem)
   {
      // add parameter
      workitem->setHandleSendRequest(true);

      workitem->setServiceIf(_ecnrServiceSendIf);
      workitem->setWrapper(this);

      workitem->setEventType(ECNR_EVENT_DESTROY);
      workitem->setAppId(appId);

      // push work item
      _ccDbusIfControllerClient->pushWorkItem(workitem);

      return ECNR_OK;
   }

   return ECNR_ERROR;
}

EcnrErrorCode EcNrWrapper::setConfiguration(IN const EcnrAppId appId, IN const EcnrConfigurationId configurationId)
{
   CONTROLLER_CLIENT_NULL_CHECK;
   ECNR_SERVICE_SENDIF_NULL_CHECK;
   DBUSIF_AVAILABLE_CHECK;
   SERVICE_AVAILABLE_CHECK;

   // create work item
   EcNrWorkItem* workitem = getNewEcNrWorkItem();

   if(workitem)
   {
      // add parameter
      workitem->setHandleSendRequest(true);

      workitem->setServiceIf(_ecnrServiceSendIf);
      workitem->setWrapper(this);

      workitem->setEventType(ECNR_EVENT_CONFIGURATION);
      workitem->setAppId(appId);
      workitem->setConfigurationId(configurationId);

      // push work item
      _ccDbusIfControllerClient->pushWorkItem(workitem);

      return ECNR_OK;
   }

   return ECNR_ERROR;
}

EcnrErrorCode EcNrWrapper::startAudio(IN const EcnrAppId appId)
{
   CONTROLLER_CLIENT_NULL_CHECK;
   ECNR_SERVICE_SENDIF_NULL_CHECK;
   DBUSIF_AVAILABLE_CHECK;
   SERVICE_AVAILABLE_CHECK;

   // create work item
   EcNrWorkItem* workitem = getNewEcNrWorkItem();

   if(workitem)
   {
      // add parameter
      workitem->setHandleSendRequest(true);

      workitem->setServiceIf(_ecnrServiceSendIf);
      workitem->setWrapper(this);

      workitem->setEventType(ECNR_EVENT_STARTAUDIO);
      workitem->setAppId(appId);

      // push work item
      _ccDbusIfControllerClient->pushWorkItem(workitem);

      return ECNR_OK;
   }

   return ECNR_ERROR;
}

EcnrErrorCode EcNrWrapper::stopAudio(IN const EcnrAppId appId)
{
   CONTROLLER_CLIENT_NULL_CHECK;
   ECNR_SERVICE_SENDIF_NULL_CHECK;
   DBUSIF_AVAILABLE_CHECK;
   SERVICE_AVAILABLE_CHECK;

   // create work item
   EcNrWorkItem* workitem = getNewEcNrWorkItem();

   if(workitem)
   {
      // add parameter
      workitem->setHandleSendRequest(true);

      workitem->setServiceIf(_ecnrServiceSendIf);
      workitem->setWrapper(this);

      workitem->setEventType(ECNR_EVENT_STOPAUDIO);
      workitem->setAppId(appId);

      // push work item
      _ccDbusIfControllerClient->pushWorkItem(workitem);

      return ECNR_OK;
   }

   return ECNR_ERROR;
}

EcnrErrorCode EcNrWrapper::setSendMuteSwitch(IN const EcnrAppId appId, IN const EcnrSendMuteSwitch sendMuteSwitch)
{
   CONTROLLER_CLIENT_NULL_CHECK;
   ECNR_SERVICE_SENDIF_NULL_CHECK;
   DBUSIF_AVAILABLE_CHECK;
   SERVICE_AVAILABLE_CHECK;

   // create work item
   EcNrWorkItem* workitem = getNewEcNrWorkItem();

   if(workitem)
   {
      // add parameter
      workitem->setHandleSendRequest(true);

      workitem->setServiceIf(_ecnrServiceSendIf);
      workitem->setWrapper(this);

      workitem->setEventType(ECNR_EVENT_MUTESWITCH);
      workitem->setAppId(appId);
      workitem->setSendMuteSwitch(sendMuteSwitch);

      // push work item
      _ccDbusIfControllerClient->pushWorkItem(workitem);

      return ECNR_OK;
   }

   return ECNR_ERROR;
}

EcnrErrorCode EcNrWrapper::getVersion(void)
{
   CONTROLLER_CLIENT_NULL_CHECK;
   ECNR_SERVICE_SENDIF_NULL_CHECK;
   DBUSIF_AVAILABLE_CHECK;
   SERVICE_AVAILABLE_CHECK;

   // create work item
   EcNrWorkItem* workitem = getNewEcNrWorkItem();

   if(workitem)
   {
      // add parameter
      workitem->setHandleSendRequest(true);

      workitem->setServiceIf(_ecnrServiceSendIf);
      workitem->setWrapper(this);

      workitem->setEventType(ECNR_EVENT_GETVERSION);

      // push work item
      _ccDbusIfControllerClient->pushWorkItem(workitem);

      return ECNR_OK;
   }

   return ECNR_ERROR;
}

void EcNrWrapper::handleServiceAvailability(IN const bool available)
{
   if(_serviceAvailable != available)
   {
      _serviceAvailable = available;
      if(true == _serviceAvailable)
      {
         if(NULL != _serviceSemaphore)
         {
            _serviceSemaphore->post();
         }
      }
   }
}

bool EcNrWrapper::waitForServiceAvailable(void)
{
   if(true == _serviceAvailable)
   {
      if(NULL != _serviceSemaphore)
      {
         _serviceSemaphore->tryWait();
      }
      return true;
   }
   else
   {
      // wait for service availability
      if(NULL != _serviceSemaphore)
      {
         (void)_serviceSemaphore->wait(SERVICE_WAITING_TIMEOUT_MS);
      }
      return _serviceAvailable;
   }
}

} //ccdbusif
