/**
 * @file EcNrWorkItem.cpp
 *
 * @par SW-Component
 * WorkItem
 *
 * @brief EcNr work item.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of Work item handling for EcNr.
 */

#include "EcNrWorkItem.h"
#include "CcDbusIfTypes.h"
#include "IEcNrDbusServiceSendRequestIf.h"
#include "EcNrWrapper.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_ASSERT
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EcNrWorkItem.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

EcNrWorkItem::EcNrWorkItem()
{
   _handleSendRequest = false;
   _ecnrServiceIf = NULL;
   _ecnrWrapper = NULL;
   _eventType = ECNR_EVENT_NOTAVAILABLE;
   _appId = 0;
   _configurationId = 0;
   _sendMuteSwitch = 0;
}

EcNrWorkItem::~EcNrWorkItem()
{
   _ecnrServiceIf = NULL;
   _ecnrWrapper = NULL;
}

void EcNrWorkItem::processWorkItem()
{
   if(true == _handleSendRequest)
   {
      processSendRequest();
   }
}

void EcNrWorkItem::processSendRequest()
{
   if(NULL == _ecnrServiceIf)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   act_t token = DEFAULT_ACT;

   switch(_eventType)
   {
      case ECNR_EVENT_INITIALIZE:
         token = _ecnrServiceIf->sendEcnrInitializeRequest(_appId, _configurationId);
         break;
      case ECNR_EVENT_DESTROY:
         token = _ecnrServiceIf->sendEcnrDestroyRequest(_appId);
         break;
      case ECNR_EVENT_CONFIGURATION:
         token = _ecnrServiceIf->sendEcnrSetConfigurationRequest(_appId, _configurationId);
         break;
      case ECNR_EVENT_STARTAUDIO:
         token = _ecnrServiceIf->sendEcnrStartAudioRequest(_appId);
         break;
      case ECNR_EVENT_STOPAUDIO:
         token = _ecnrServiceIf->sendEcnrStopAudioRequest(_appId);
         break;
      case ECNR_EVENT_MUTESWITCH:
         token = _ecnrServiceIf->sendEcnrSetSendMuteSwitchRequest(_appId, _sendMuteSwitch);
         break;
      case ECNR_EVENT_GETVERSION:
         token = _ecnrServiceIf->sendEcnrGetVersionRequest();
         break;

      default:
         // should never happen
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         ETG_TRACE_ERR(("EcNr processSendRequest called for Invalid Event"));
         break;
   }

   if(DEFAULT_ACT == token)
   {
      ETG_TRACE_ERR(("EcNr SendRequest failed for eventType: %d", _eventType));

      // in case of default token is returned we have to assume that sending DBUS message failed due to any reason
      // call related EcNrWrapper callback function with result failed and empty result data (if necessary)
      processSendRequestFailed();
   }
}

void EcNrWorkItem::processSendRequestFailed()
{
   if(NULL == _ecnrWrapper)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   switch(_eventType)
   {
      case ECNR_EVENT_INITIALIZE:
         (void)_ecnrWrapper->initializeResultCB(ECNR_REQ_FAILED);
         break;
      case ECNR_EVENT_DESTROY:
         (void)_ecnrWrapper->destroyResultCB(ECNR_REQ_FAILED);
         break;
      case ECNR_EVENT_CONFIGURATION:
         (void)_ecnrWrapper->setConfigurationResultCB(ECNR_REQ_FAILED);
         break;
      case ECNR_EVENT_STARTAUDIO:
         (void)_ecnrWrapper->startAudioResultCB(ECNR_REQ_FAILED);
         break;
      case ECNR_EVENT_STOPAUDIO:
         (void)_ecnrWrapper->stopAudioResultCB(ECNR_REQ_FAILED);
         break;
      case ECNR_EVENT_MUTESWITCH:
         (void)_ecnrWrapper->setSendMuteSwitchResultCB(ECNR_REQ_FAILED);
         break;
      case ECNR_EVENT_GETVERSION:
      {
         ::std::vector< uint32 > emptyVersion;
         ::std::string emptyVersionString("");
         ::std::string emptyVersionComment("");
         (void)_ecnrWrapper->getVersionResultCB(ECNR_REQ_FAILED, emptyVersion, emptyVersionString, emptyVersionComment);
      }
      break;

      default:
         // should never happen, handled in processSendRequest().
         break;
   }
}

} //ccdbusif
