/**
 * @file CcDbusIfWorkItemQueue.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Work Item Queue.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Work item queue for CcDbusIf.
 */

#include "CcDbusIfWorkItemQueue.h"
#include "IWorkItem.h"
#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_ASSERT
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

#include <cstring>
#include <cerrno>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/CcDbusIfWorkItemQueue.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

CcDbusIfWorkItemQueue::CcDbusIfWorkItemQueue()
{
   // _queue is empty after instantiation
   memset(&_semaphore, 0, sizeof(_semaphore));
   if(-1 == sem_init(&_semaphore, 0, 0))
   {
      // check result for debugging purpose

      /*
       * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sem_init.html:
         The sem_init() function shall fail if:

         [EINVAL]
            The value argument exceeds {SEM_VALUE_MAX}.
            => value is 0 and cannot exceed SEM_VALUE_MAX
         [ENOSPC]
            A resource required to initialize the semaphore has been exhausted, or the limit on semaphores ( {SEM_NSEMS_MAX}) has been reached.
            => should never happen
         [EPERM]
            The process lacks the appropriate privileges to initialize the semaphore.
            => should never happen else there is a general problem
         [ENOSYS]
            pshared is nonzero, but the system does not support process-shared semaphores
            => pshared is 0
       *
       */

      int errNumber = errno;
      ETG_TRACE_ERR((" CcDbusIfWorkItemQueue(): sem_init failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
      FW_NORMAL_ASSERT_ALWAYS();
   }
   // _lock needs no explicit initialization
}

CcDbusIfWorkItemQueue::~CcDbusIfWorkItemQueue()
{
   IWorkItem* workItem;

   while(0 < _queue.size())
   {
      workItem = _queue.front();
      _queue.pop_front();
      if(0 != workItem)
      {
         delete workItem;
      }
   }

   (void)sem_destroy(&_semaphore); // ignore return value
}

void CcDbusIfWorkItemQueue::push(IWorkItem* workItem)
{
   if(0 != workItem)
   {
      _lock.lock();

      _queue.push_back(workItem);

      _lock.unlock();
   }
}

void CcDbusIfWorkItemQueue::pushAndPost(IWorkItem* workItem)
{
   if(0 != workItem)
   {
      _lock.lock();

      _queue.push_back(workItem);

      _lock.unlock();

      if(-1 == sem_post(&_semaphore))
      {
         // check result for debugging purpose

         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sem_post.html:
            The sem_post() function may fail if:

            [EINVAL]
               The sem argument does not refer to a valid semaphore.
               => a valid semaphore is provided
            [EOVERFLOW]
               The maximum allowable value for a semaphore would be exceeded.
               => should never happen
          *
          */

         int errNumber = errno;
         ETG_TRACE_ERR((" pushAndPost(): sem_post failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
}

IWorkItem* CcDbusIfWorkItemQueue::pop(void)
{
   IWorkItem* workItem(0);

   _lock.lock();

   if(0 < _queue.size())
   {
      workItem = _queue.front();
      _queue.pop_front();
   }

   _lock.unlock();

   return workItem;
}

IWorkItem* CcDbusIfWorkItemQueue::waitAndPop(void)
{
   IWorkItem* workItem(0);

   for(;;)
   {
      if(-1 == sem_wait(&_semaphore))
      {
         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sem_wait.html:
            The sem_trywait() and sem_wait() functions shall fail if:

            [EAGAIN]
               The semaphore was already locked, so it cannot be immediately locked by the sem_trywait() operation ( sem_trywait() only).
               => valid for sem_trywait() only, no need to handle

            The sem_trywait() and sem_wait() functions may fail if:

            [EDEADLK]
               A deadlock condition was detected.
               => handle as error
            [EINTR]
               A signal interrupted this function.
               => handle as normal behavior, do a while loop
            [EINVAL]
               The sem argument does not refer to a valid semaphore.
               => a valid semaphore is provided
            [ETIMEDOUT]
               The call timed out before the semaphore could be locked.
               => valid for sem_timedwait() only, no need to handle
          *
          */

         int errNumber = errno;

         if(EINTR != errNumber)
         {
            ETG_TRACE_ERR((" pushAndPost(): sem_wait failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
            FW_NORMAL_ASSERT_ALWAYS();
            break;
         }
      }
      else
      {
         break;
      }
   }

   _lock.lock();

   if(0 < _queue.size())
   {
      workItem = _queue.front();
      _queue.pop_front();
   }

   _lock.unlock();

   return workItem;
}

void CcDbusIfWorkItemQueue::empty(void)
{
   IWorkItem* workItem;

   _lock.lock();

   while(0 < _queue.size())
   {
      workItem = _queue.front();
      _queue.pop_front();
      if(0 != workItem)
      {
         delete workItem;
      }
   }

   _lock.unlock();
}

} //ccdbusif
