/**
 * @file CcDbusIfController.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Controller.
 *
 * @copyright (C) 2015-2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Controller for CcDbusIf.
 */

#include "CcDbusIfController.h"
#include "CcDbusIfLauncher.h"
#include "CcDbusIfDestroyAllProxiesWorkItem.h"
#include "IDestroyAllProxies.h"

// ConnMan DBUS interface:
#include "ConnManDbusManagerFactory.h"
#include "ConnManDbusServiceFactory.h"
#include "ConnManDbusTechnologyFactory.h"

// EcNr DBUS interface:
#include "EcNrDbusServiceProxyIf.h"

// Evolution Genivi DBUS interface:
#include "EvolutionGeniviDbusAdapterFactory.h"
#include "EvolutionGeniviDbusAgentManagerFactory.h"
#include "EvolutionGeniviDbusDeviceFactory.h"
#include "EvolutionGeniviDbusDMFactory.h"
#include "EvolutionGeniviDbusServiceFactory.h"
#include "EvolutionGeniviDbusTraceFactory.h"
#include "EvolutionGeniviDbusGeniviTraceFactory.h"
#include "EvolutionGeniviDbusAgentFactory.h"
#include "EvolutionGeniviDbusDidFactory.h"
#include "EvolutionGeniviDbusSppFactory.h"
#include "EvolutionGeniviDbusObexClientFactory.h"
#include "EvolutionGeniviDbusObexSessionFactory.h"
#include "EvolutionGeniviDbusObexTransferFactory.h"
#include "EvolutionGeniviDbusHfpCallVolumeFactory.h"
#include "EvolutionGeniviDbusHfpHandsfreeFactory.h"
#include "EvolutionGeniviDbusHfpManagerFactory.h"
#include "EvolutionGeniviDbusHfpModemFactory.h"
#include "EvolutionGeniviDbusHfpNetworkOperatorFactory.h"
#include "EvolutionGeniviDbusHfpNetworkRegistrationFactory.h"
#include "EvolutionGeniviDbusHfpSiriFactory.h"
#include "EvolutionGeniviDbusHfpVoiceCallManagerFactory.h"
#include "EvolutionGeniviDbusHfpVoiceCallFactory.h"
#include "EvolutionGeniviDbusSerialFactory.h"
#include "EvolutionGeniviDbusLEAdapterFactory.h"
#include "EvolutionGeniviDbusLEDeviceFactory.h"
#include "EvolutionGeniviDbusLESeamlessPairingFactory.h"

// ObjectManger DBUS interface:
#include "OrgFreedesktopDbusObjectManagerFactory.h"

// Introspectable DBUS interface:
#include "IntrospectableDbusProxyIf.h"

// PropertiesGetAll DBUS interface:
#include "PropertiesGetAllDbusProxyIf.h"

// WBL DBUS interface:
#include "WblDbusLastIntendedModeFactory.h"
#include "WblDbusConflictManagementFactory.h"
#include "WblDbusWifiSetupFactory.h"

// include ASF DBUS application library for proxies only mode
//#include "cc/proxies/only/CcDbusIfAppLibrary.h" --- there is a naming collision for ::com::alps::bt::Agent, solution: increase include depth
#include "proxies/only/CcDbusIfAppLibrary.h"
// include ASF DBUS application library including Evolution agent
//#include "org/agent.h" --- there is a naming collision for ::org::agent, solution: increase include depth
#include "agent.h"

// include ASF DBUS component for proxies only mode
#include "cc/proxies/only/CcDbusIfComponent.h"
// include ASF DBUS component for Evolution agent
#include "org/CcDbusIfComponent.h"

#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/CcDbusIfController.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

CcDbusIfController::CcDbusIfController() :
_workItemQueue(),
_localEventSender(0),
_launcher(0),
_destroyProxyList(),
_destroyAllProxiesList(),
_destroyBluetoothRuntimeProxyList(),
_destroyAllBluetoothRuntimeProxiesList(),
_destroyProxiesSem(),
_startupSem(),
_callbackIdLock(),
_callbackId(1),
_connManManagerProxy(0),
_connManServiceProxy(0),
_connManTechnologyProxy(0),
_ecnrServiceProxy(0),
_evolutionGeniviAdapterProxy(0),
_evolutionGeniviDeviceProxy(0),
_evolutionGeniviAgentManagerProxy(0),
_evolutionGeniviDMProxy(0),
_evolutionGeniviServiceProxy(0),
_evolutionGeniviTraceProxy(0),
_evolutionGeniviGeniviTraceProxy(0),
_evolutionGeniviAgentStub(0),
_evolutionGeniviDidProxy(0),
_evolutionGeniviSppProxy(0),
_evolutionGeniviObexClientProxy(0),
_evolutionGeniviObexSessionProxy(0),
_evolutionGeniviObexTransferProxy(0),
_evolutionGeniviHfpCallVolumeProxy(0),
_evolutionGeniviHfpHandsfreeProxy(0),
_evolutionGeniviHfpManagerProxy(0),
_evolutionGeniviHfpModemProxy(0),
_evolutionGeniviHfpNetworkOperatorProxy(0),
_evolutionGeniviHfpNetworkRegistrationProxy(0),
_evolutionGeniviHfpSiriProxy(0),
_evolutionGeniviHfpVoiceCallManagerProxy(0),
_evolutionGeniviHfpVoiceCallProxy(0),
_evolutionGeniviSerialProxy(0),
_evolutionGeniviLEAdapterProxy(0),
_evolutionGeniviLEDeviceProxy(0),
_evolutionGeniviLESeamlessPairingProxy(0),
_objectManagerProxy(0),
_introspectableProxy(0),
_propertiesGetAllProxy(0),
_lastIntendedModeProxy(0),
_conflictManagementProxy(0),
_wifiSetupProxy(0)
{
   // create launcher
   _launcher = new CcDbusIfLauncher();

   // create semaphores
   _destroyProxiesSem.create();
   _startupSem.create();

   // reserve memory
   _destroyProxyList.reserve(100);
   _destroyAllProxiesList.reserve(100);
   _destroyBluetoothRuntimeProxyList.reserve(100);
   _destroyAllBluetoothRuntimeProxiesList.reserve(100);

   // ConnMan DBUS interface:
   _connManManagerProxy = ConnManDbusManagerFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(ConnManDbusManagerFactory::getInstance().getDestroyer());

   _connManServiceProxy = ConnManDbusServiceFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(ConnManDbusServiceFactory::getInstance().getDestroyer());

   _connManTechnologyProxy = ConnManDbusTechnologyFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(ConnManDbusTechnologyFactory::getInstance().getDestroyer());

   // EcNr DBUS interface:
   EcNrDbusServiceProxyIf* tmpEcNrServiceProxy = new EcNrDbusServiceProxyIf(this);
   _ecnrServiceProxy = tmpEcNrServiceProxy;
   _destroyProxyList.push_back(tmpEcNrServiceProxy);

   // Evolution Genivi DBUS interface:
   _evolutionGeniviAdapterProxy = evolution::EvolutionGeniviDbusAdapterFactory::getInstance().getProxy(this);
   if(0 != _evolutionGeniviAdapterProxy) { _destroyAllProxiesList.push_back(_evolutionGeniviAdapterProxy->getDestroyer()); }
   if(0 != _evolutionGeniviAdapterProxy) { _destroyAllBluetoothRuntimeProxiesList.push_back(_evolutionGeniviAdapterProxy->getDestroyer()); }

   _evolutionGeniviAgentManagerProxy = EvolutionGeniviDbusAgentManagerFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusAgentManagerFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusAgentManagerFactory::getInstance().getDestroyer());

   _evolutionGeniviDeviceProxy = EvolutionGeniviDbusDeviceFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusDeviceFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusDeviceFactory::getInstance().getDestroyer());

   _evolutionGeniviDMProxy = EvolutionGeniviDbusDMFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusDMFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusDMFactory::getInstance().getDestroyer());

   _evolutionGeniviServiceProxy = EvolutionGeniviDbusServiceFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusServiceFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusServiceFactory::getInstance().getDestroyer());

   _evolutionGeniviTraceProxy = EvolutionGeniviDbusTraceFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusTraceFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusTraceFactory::getInstance().getDestroyer());

   _evolutionGeniviGeniviTraceProxy = EvolutionGeniviDbusGeniviTraceFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusGeniviTraceFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusGeniviTraceFactory::getInstance().getDestroyer());

   _evolutionGeniviAgentStub = evolution::EvolutionGeniviDbusAgentFactory::getInstance().getProxy(this);
   if(0 != _evolutionGeniviAgentStub) { _destroyAllProxiesList.push_back(_evolutionGeniviAgentStub->getDestroyer()); }
   if(0 != _evolutionGeniviAgentStub) { _destroyAllBluetoothRuntimeProxiesList.push_back(_evolutionGeniviAgentStub->getDestroyer()); }

   _evolutionGeniviDidProxy = EvolutionGeniviDbusDidFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusDidFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusDidFactory::getInstance().getDestroyer());

   _evolutionGeniviSppProxy = EvolutionGeniviDbusSppFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusSppFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusSppFactory::getInstance().getDestroyer());

   _evolutionGeniviObexClientProxy = EvolutionGeniviDbusObexClientFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusObexClientFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusObexClientFactory::getInstance().getDestroyer());

   _evolutionGeniviObexSessionProxy = EvolutionGeniviDbusObexSessionFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusObexSessionFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusObexSessionFactory::getInstance().getDestroyer());

   _evolutionGeniviObexTransferProxy = EvolutionGeniviDbusObexTransferFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusObexTransferFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusObexTransferFactory::getInstance().getDestroyer());

   _evolutionGeniviHfpCallVolumeProxy = EvolutionGeniviDbusHfpCallVolumeFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusHfpCallVolumeFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusHfpCallVolumeFactory::getInstance().getDestroyer());

   _evolutionGeniviHfpHandsfreeProxy = EvolutionGeniviDbusHfpHandsfreeFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusHfpHandsfreeFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusHfpHandsfreeFactory::getInstance().getDestroyer());

   _evolutionGeniviHfpManagerProxy = EvolutionGeniviDbusHfpManagerFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusHfpManagerFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusHfpManagerFactory::getInstance().getDestroyer());

   _evolutionGeniviHfpModemProxy = EvolutionGeniviDbusHfpModemFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusHfpModemFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusHfpModemFactory::getInstance().getDestroyer());

   _evolutionGeniviHfpNetworkOperatorProxy = EvolutionGeniviDbusHfpNetworkOperatorFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusHfpNetworkOperatorFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusHfpNetworkOperatorFactory::getInstance().getDestroyer());

   _evolutionGeniviHfpNetworkRegistrationProxy = EvolutionGeniviDbusHfpNetworkRegistrationFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusHfpNetworkRegistrationFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusHfpNetworkRegistrationFactory::getInstance().getDestroyer());

   _evolutionGeniviHfpSiriProxy = EvolutionGeniviDbusHfpSiriFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusHfpSiriFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusHfpSiriFactory::getInstance().getDestroyer());

   _evolutionGeniviHfpVoiceCallManagerProxy = EvolutionGeniviDbusHfpVoiceCallManagerFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusHfpVoiceCallManagerFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusHfpVoiceCallManagerFactory::getInstance().getDestroyer());

   _evolutionGeniviHfpVoiceCallProxy = EvolutionGeniviDbusHfpVoiceCallFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusHfpVoiceCallFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusHfpVoiceCallFactory::getInstance().getDestroyer());

   _evolutionGeniviSerialProxy = evolution::EvolutionGeniviDbusSerialFactory::getInstance().getProxy(this);
   if(0 != _evolutionGeniviSerialProxy) { _destroyAllProxiesList.push_back(_evolutionGeniviSerialProxy->getDestroyer()); }
   if(0 != _evolutionGeniviSerialProxy) { _destroyAllBluetoothRuntimeProxiesList.push_back(_evolutionGeniviSerialProxy->getDestroyer()); }

   _evolutionGeniviLEAdapterProxy = EvolutionGeniviDbusLEAdapterFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusLEAdapterFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusLEAdapterFactory::getInstance().getDestroyer());

   _evolutionGeniviLEDeviceProxy = EvolutionGeniviDbusLEDeviceFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusLEDeviceFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusLEDeviceFactory::getInstance().getDestroyer());

   _evolutionGeniviLESeamlessPairingProxy = EvolutionGeniviDbusLESeamlessPairingFactory::getInstance().getProxy(this);
   _destroyProxyList.push_back(EvolutionGeniviDbusLESeamlessPairingFactory::getInstance().getDestroyer());
   _destroyBluetoothRuntimeProxyList.push_back(EvolutionGeniviDbusLESeamlessPairingFactory::getInstance().getDestroyer());

   // ObjectManager DBUS interface:
   _objectManagerProxy = freedesktop::OrgFreedesktopDbusObjectManagerFactory::getInstance().getProxy(this);
   if(0 != _objectManagerProxy) { _destroyAllProxiesList.push_back(_objectManagerProxy->getDestroyer()); }

   // Introspectable DBUS interface:
   IntrospectableDbusProxyIf* tmpIntrospectableDbusProxy = new IntrospectableDbusProxyIf(this);
   _introspectableProxy = tmpIntrospectableDbusProxy;
   _destroyProxyList.push_back(tmpIntrospectableDbusProxy);

   // PropertiesGetAll DBUS interface:
   PropertiesGetAllDbusProxyIf* tmpPropertiesGetAllDbusProxy = new PropertiesGetAllDbusProxyIf(this);
   _propertiesGetAllProxy = tmpPropertiesGetAllDbusProxy;
   _destroyProxyList.push_back(tmpPropertiesGetAllDbusProxy);

   // WBL DBUS interface:
   _lastIntendedModeProxy = wbl::WblDbusLastIntendedModeFactory::getInstance().getProxy(this);
   if(0 != _lastIntendedModeProxy) { _destroyAllProxiesList.push_back(_lastIntendedModeProxy->getDestroyer()); }
   _conflictManagementProxy = wbl::WblDbusConflictManagementFactory::getInstance().getProxy(this);
   if(0 != _conflictManagementProxy) { _destroyAllProxiesList.push_back(_conflictManagementProxy->getDestroyer()); }
   _wifiSetupProxy = wbl::WblDbusWifiSetupFactory::getInstance().getProxy(this);
   if(0 != _wifiSetupProxy) { _destroyAllProxiesList.push_back(_wifiSetupProxy->getDestroyer()); }
}

CcDbusIfController::~CcDbusIfController()
{
   _workItemQueue.empty();
   _localEventSender = 0;

   if(_launcher)
   {
      delete _launcher;
   }

   _destroyProxyList.clear();
   _destroyAllProxiesList.clear();
   _destroyBluetoothRuntimeProxyList.clear();
   _destroyAllBluetoothRuntimeProxiesList.clear();

   _destroyProxiesSem.destroy();
   _startupSem.destroy();

   // ConnMan DBUS interfaces:
   ConnManDbusManagerFactory::getInstance().destroyProxy();
   _connManManagerProxy = 0;

   ConnManDbusServiceFactory::getInstance().destroyProxy();
   _connManServiceProxy = 0;

   ConnManDbusTechnologyFactory::getInstance().destroyProxy();
   _connManTechnologyProxy = 0;

   // EcNr DBUS interface:
   if(_ecnrServiceProxy)
   {
      delete _ecnrServiceProxy;
   }

   // Evolution Genivi DBUS interface:
   evolution::EvolutionGeniviDbusAdapterFactory::getInstance().destroyProxy();
   _evolutionGeniviAdapterProxy = 0;
   EvolutionGeniviDbusDeviceFactory::getInstance().destroyProxy();
   _evolutionGeniviDeviceProxy = 0;
   EvolutionGeniviDbusAgentManagerFactory::getInstance().destroyProxy();
   _evolutionGeniviAgentManagerProxy = 0;
   EvolutionGeniviDbusDMFactory::getInstance().destroyProxy();
   _evolutionGeniviDMProxy = 0;
   EvolutionGeniviDbusServiceFactory::getInstance().destroyProxy();
   _evolutionGeniviServiceProxy = 0;
   EvolutionGeniviDbusTraceFactory::getInstance().destroyProxy();
   _evolutionGeniviTraceProxy = 0;
   EvolutionGeniviDbusGeniviTraceFactory::getInstance().destroyProxy();
   _evolutionGeniviGeniviTraceProxy = 0;
   evolution::EvolutionGeniviDbusAgentFactory::getInstance().destroyProxy();
   _evolutionGeniviAgentStub = 0;
   EvolutionGeniviDbusDidFactory::getInstance().destroyProxy();
   _evolutionGeniviDidProxy = 0;
   EvolutionGeniviDbusSppFactory::getInstance().destroyProxy();
   _evolutionGeniviSppProxy = 0;
   EvolutionGeniviDbusObexClientFactory::getInstance().destroyProxy();
   _evolutionGeniviObexClientProxy = 0;
   EvolutionGeniviDbusObexSessionFactory::getInstance().destroyProxy();
   _evolutionGeniviObexSessionProxy = 0;
   EvolutionGeniviDbusObexTransferFactory::getInstance().destroyProxy();
   _evolutionGeniviObexTransferProxy = 0;
   EvolutionGeniviDbusHfpCallVolumeFactory::getInstance().destroyProxy();
   _evolutionGeniviHfpCallVolumeProxy = 0;
   EvolutionGeniviDbusHfpHandsfreeFactory::getInstance().destroyProxy();
   _evolutionGeniviHfpHandsfreeProxy = 0;
   EvolutionGeniviDbusHfpManagerFactory::getInstance().destroyProxy();
   _evolutionGeniviHfpManagerProxy = 0;
   EvolutionGeniviDbusHfpModemFactory::getInstance().destroyProxy();
   _evolutionGeniviHfpModemProxy = 0;
   EvolutionGeniviDbusHfpNetworkOperatorFactory::getInstance().destroyProxy();
   _evolutionGeniviHfpNetworkOperatorProxy = 0;
   EvolutionGeniviDbusHfpNetworkRegistrationFactory::getInstance().destroyProxy();
   _evolutionGeniviHfpNetworkRegistrationProxy = 0;
   EvolutionGeniviDbusHfpSiriFactory::getInstance().destroyProxy();
   _evolutionGeniviHfpSiriProxy = 0;
   EvolutionGeniviDbusHfpVoiceCallManagerFactory::getInstance().destroyProxy();
   _evolutionGeniviHfpVoiceCallManagerProxy = 0;
   EvolutionGeniviDbusHfpVoiceCallFactory::getInstance().destroyProxy();
   _evolutionGeniviHfpVoiceCallProxy = 0;
   evolution::EvolutionGeniviDbusSerialFactory::getInstance().destroyProxy();
   _evolutionGeniviSerialProxy = 0;
   EvolutionGeniviDbusLEAdapterFactory::getInstance().destroyProxy();
   _evolutionGeniviLEAdapterProxy = 0;
   EvolutionGeniviDbusLEDeviceFactory::getInstance().destroyProxy();
   _evolutionGeniviLEDeviceProxy = 0;
   EvolutionGeniviDbusLESeamlessPairingFactory::getInstance().destroyProxy();
   _evolutionGeniviLESeamlessPairingProxy = 0;

   // ObjectManager DBUS interface:
   freedesktop::OrgFreedesktopDbusObjectManagerFactory::getInstance().destroyProxy();
   _objectManagerProxy = 0;

   // Introspectable DBUS interface:
   if(_introspectableProxy)
   {
      delete _introspectableProxy;
   }

   // PropertiesGetAll DBUS interface:
   if(_propertiesGetAllProxy)
   {
      delete _propertiesGetAllProxy;
   }

   // WBL DBUS interface:
   wbl::WblDbusLastIntendedModeFactory::getInstance().destroyProxy();
   _lastIntendedModeProxy = 0;
   wbl::WblDbusConflictManagementFactory::getInstance().destroyProxy();
   _conflictManagementProxy = 0;
   wbl::WblDbusWifiSetupFactory::getInstance().destroyProxy();
   _wifiSetupProxy = 0;
}

void CcDbusIfController::start(void)
{
}

void CcDbusIfController::stop(void)
{
}

void CcDbusIfController::startDbusLauncher(const LaunchType launchType, const ::std::string& configFile)
{
   if(_launcher)
   {
      const char* logArg = "--logconfig";
      char progName[1] = { 0 };
      char arg1[20] = { 0 };
      strncpy(arg1, logArg, sizeof(arg1) - 1);
      char arg2[configFile.size() + 1];
      memset(arg2, 0, configFile.size() + 1);
      strncpy(arg2, configFile.c_str(), sizeof(arg2) - 1);
      int argListSize = 3;
      char* argList[argListSize];
      argList[0] = progName;
      argList[1] = arg1;
      argList[2] = arg2;

      // check for agent modes
      if(WITH_EVOLUTION_AGENT == launchType)
      {
         // set event receiver before launching
         ::org::CcDbusIfComponent::setEventReceiver(this);
         // set startup semaphore before launching
         ::org::CcDbusIfComponent::setStartupSemaphore(&_startupSem);

         if(true == configFile.empty())
         {
            _launcher->start(::org::executeAgent);
         }
         else
         {
            _launcher->start(::org::executeAgent, argListSize, argList);
         }

         // get local message sender reference after launching
         _localEventSender = ::org::CcDbusIfComponent::getEventSender();
         FW_NORMAL_ASSERT(0 != _localEventSender);
      }
      // default mode must be the last one
      else /*if(PROXIES_ONLY == launchType)*/
      {
         // set event receiver before launching
         ::cc::proxies::only::CcDbusIfComponent::setEventReceiver(this);
         // set startup semaphore before launching
         ::cc::proxies::only::CcDbusIfComponent::setStartupSemaphore(&_startupSem);

         if(true == configFile.empty())
         {
            _launcher->start(::cc::proxies::only::executeCcDbusIfAppLibrary);
         }
         else
         {
            _launcher->start(::cc::proxies::only::executeCcDbusIfAppLibrary, argListSize, argList);
         }

         // get local message sender reference after launching
         _localEventSender = ::cc::proxies::only::CcDbusIfComponent::getEventSender();
         FW_NORMAL_ASSERT(0 != _localEventSender);
      }
   }
}

void CcDbusIfController::stopDbusLauncher(void)
{
   // reset local message sender reference
   _localEventSender = 0;

   if(_launcher)
   {
      _launcher->stop();
   }
}

void CcDbusIfController::prepareStop(void)
{
   ETG_TRACE_USR1((" prepareStop(): enter"));

   pushWorkItem(new CcDbusIfDestroyAllProxiesWorkItem(this));

   _destroyProxiesSem.wait();

   ETG_TRACE_USR1((" prepareStop(): exit"));
}

unsigned int CcDbusIfController::getCallbackId(void)
{
   unsigned int id;

   _callbackIdLock.lock();

   id = _callbackId++;

   _callbackIdLock.unlock();

   return id;
}

void CcDbusIfController::destroyAllProxies(void)
{
   ETG_TRACE_USR1((" destroyAllProxies(): enter"));

   for(size_t i = 0; i < _destroyProxyList.size(); i++)
   {
      if(0 != _destroyProxyList[i])
      {
         _destroyProxyList[i]->destroyProxy();
      }
   }

   for(size_t i = 0; i < _destroyAllProxiesList.size(); i++)
   {
      if(0 != _destroyAllProxiesList[i])
      {
         _destroyAllProxiesList[i]->handleDestroyAllProxies();
      }
   }

   ETG_TRACE_USR1((" destroyAllProxies(): exit"));

   _destroyProxiesSem.post();
}

void CcDbusIfController::pushWorkItem(IWorkItem* workItem)
{
   _workItemQueue.push(workItem); // lock is done inside
   FW_IF_NULL_PTR_RETURN(_localEventSender);
   _localEventSender->handleSendLocalEvent(0);
}

void CcDbusIfController::destroyAllBluetoothRuntimeProxies(void)
{
   ETG_TRACE_USR1((" destroyAllBluetoothRuntimeProxies(): enter"));

   for(size_t i = 0; i < _destroyBluetoothRuntimeProxyList.size(); i++)
   {
      if(0 != _destroyBluetoothRuntimeProxyList[i])
      {
         _destroyBluetoothRuntimeProxyList[i]->destroyAllRuntimeProxies();
      }
   }

   for(size_t i = 0; i < _destroyAllBluetoothRuntimeProxiesList.size(); i++)
   {
      if(0 != _destroyAllBluetoothRuntimeProxiesList[i])
      {
         _destroyAllBluetoothRuntimeProxiesList[i]->handleDestroyAllRuntimeProxies();
      }
   }

   ETG_TRACE_USR1((" destroyAllBluetoothRuntimeProxies(): exit"));
}

void CcDbusIfController::handleReceiveLocalEvent(const unsigned int event)
{
   (void)(event);

   IWorkItem* workItem(_workItemQueue.pop());
   if(0 != workItem)
   {
      // ETG_TRACE_USR2((" handleWorkItemQueue(): +++process workItem"));

      workItem->processWorkItem();
      delete workItem;
   }
}

ICcDbusIfControllerClient* CcDbusIfController::getCcDbusIfControllerClient(void)
{
   return this;
}

ICcDbusIfControllerConnMan* CcDbusIfController::getCcDbusIfControllerConnMan(void)
{
   return this;
}

ICcDbusIfControllerEcNr* CcDbusIfController::getCcDbusIfControllerEcNr(void)
{
   return this;
}

ICcDbusIfControllerEvolutionGenivi* CcDbusIfController::getCcDbusIfControllerEvolutionGenivi(void)
{
   return this;
}

ICcDbusIfControllerObjectManager* CcDbusIfController::getCcDbusIfControllerObjectManager(void)
{
   return this;
}

ICcDbusIfControllerIntrospectable* CcDbusIfController::getCcDbusIfControllerIntrospectable(void)
{
   return this;
}

ICcDbusIfControllerPropertiesGetAll* CcDbusIfController::getCcDbusIfControllerPropertiesGetAll(void)
{
   return this;
}

ICcDbusIfControllerWbl* CcDbusIfController::getCcDbusIfControllerWbl(void)
{
   return this;
}

IConnManDbusManagerSendRequestIf* CcDbusIfController::getConnManManagerProxy(void)
{
   return _connManManagerProxy;
}

IConnManDbusServiceSendRequestIf* CcDbusIfController::getConnManServiceProxy(void)
{
   return _connManServiceProxy;
}

IConnManDbusTechnologySendRequestIf* CcDbusIfController::getConnManTechnologyProxy(void)
{
   return _connManTechnologyProxy;
}

IEcNrDbusServiceSendRequestIf* CcDbusIfController::getEcNrServiceProxy(void)
{
   return _ecnrServiceProxy;
}

IEvolutionGeniviDbusAdapterSendRequestIf* CcDbusIfController::getEvolutionGeniviAdapterProxy(void)
{
   return _evolutionGeniviAdapterProxy;
}

IEvolutionGeniviDbusAgentManagerSendRequestIf* CcDbusIfController::getEvolutionGeniviAgentManagerProxy(void)
{
   return _evolutionGeniviAgentManagerProxy;
}

IEvolutionGeniviDbusDeviceSendRequestIf* CcDbusIfController::getEvolutionGeniviDeviceProxy(void)
{
   return _evolutionGeniviDeviceProxy;
}

IEvolutionGeniviDbusDMSendRequestIf* CcDbusIfController::getEvolutionGeniviDMProxy(void)
{
   return _evolutionGeniviDMProxy;
}

IEvolutionGeniviDbusServiceSendRequestIf* CcDbusIfController::getEvolutionGeniviServiceProxy(void)
{
   return _evolutionGeniviServiceProxy;
}

IEvolutionGeniviDbusTraceSendRequestIf* CcDbusIfController::getEvolutionGeniviTraceProxy(void)
{
   return _evolutionGeniviTraceProxy;
}

IEvolutionGeniviDbusGeniviTraceSendRequestIf* CcDbusIfController::getEvolutionGeniviGeniviTraceProxy(void)
{
   return _evolutionGeniviGeniviTraceProxy;
}

IEvolutionGeniviDbusAgentSendRequestIf* CcDbusIfController::getEvolutionGeniviAgentStub(void)
{
   return _evolutionGeniviAgentStub;
}

IEvolutionGeniviDbusDidSendRequestIf* CcDbusIfController::getEvolutionGeniviDidProxy(void)
{
   return _evolutionGeniviDidProxy;
}

IEvolutionGeniviDbusSppSendRequestIf* CcDbusIfController::getEvolutionGeniviSppProxy(void)
{
   return _evolutionGeniviSppProxy;
}

IEvolutionGeniviDbusObexClientSendRequestIf* CcDbusIfController::getEvolutionGeniviObexClientProxy(void)
{
   return _evolutionGeniviObexClientProxy;
}

IEvolutionGeniviDbusObexSessionSendRequestIf* CcDbusIfController::getEvolutionGeniviObexSessionProxy(void)
{
   return _evolutionGeniviObexSessionProxy;
}

IEvolutionGeniviDbusObexTransferSendRequestIf* CcDbusIfController::getEvolutionGeniviObexTransferProxy(void)
{
   return _evolutionGeniviObexTransferProxy;
}

IEvolutionGeniviDbusHfpCallVolumeSendRequestIf* CcDbusIfController::getEvolutionGeniviHfpCallVolumeProxy(void)
{
   return _evolutionGeniviHfpCallVolumeProxy;
}

IEvolutionGeniviDbusHfpHandsfreeSendRequestIf* CcDbusIfController::getEvolutionGeniviHfpHandsfreeProxy(void)
{
   return _evolutionGeniviHfpHandsfreeProxy;
}

IEvolutionGeniviDbusHfpManagerSendRequestIf* CcDbusIfController::getEvolutionGeniviHfpManagerProxy(void)
{
   return _evolutionGeniviHfpManagerProxy;
}

IEvolutionGeniviDbusHfpModemSendRequestIf* CcDbusIfController::getEvolutionGeniviHfpModemProxy(void)
{
   return _evolutionGeniviHfpModemProxy;
}

IEvolutionGeniviDbusHfpNetworkOperatorSendRequestIf* CcDbusIfController::getEvolutionGeniviHfpNetworkOperatorProxy(void)
{
   return _evolutionGeniviHfpNetworkOperatorProxy;
}

IEvolutionGeniviDbusHfpNetworkRegistrationSendRequestIf* CcDbusIfController::getEvolutionGeniviHfpNetworkRegistrationProxy(void)
{
   return _evolutionGeniviHfpNetworkRegistrationProxy;
}

IEvolutionGeniviDbusHfpSiriSendRequestIf* CcDbusIfController::getEvolutionGeniviHfpSiriProxy(void)
{
   return _evolutionGeniviHfpSiriProxy;
}

IEvolutionGeniviDbusHfpVoiceCallManagerSendRequestIf* CcDbusIfController::getEvolutionGeniviHfpVoiceCallManagerProxy(void)
{
   return _evolutionGeniviHfpVoiceCallManagerProxy;
}

IEvolutionGeniviDbusHfpVoiceCallSendRequestIf* CcDbusIfController::getEvolutionGeniviHfpVoiceCallProxy(void)
{
   return _evolutionGeniviHfpVoiceCallProxy;
}

IEvolutionGeniviDbusSerialSendRequestIf* CcDbusIfController::getEvolutionGeniviSerialProxy(void)
{
   return _evolutionGeniviSerialProxy;
}

IEvolutionGeniviDbusLEAdapterSendRequestIf* CcDbusIfController::getEvolutionGeniviLEAdapterProxy(void)
{
   return _evolutionGeniviLEAdapterProxy;
}

IEvolutionGeniviDbusLEDeviceSendRequestIf* CcDbusIfController::getEvolutionGeniviLEDeviceProxy(void)
{
   return _evolutionGeniviLEDeviceProxy;
}

IEvolutionGeniviDbusLESeamlessPairingSendRequestIf* CcDbusIfController::getEvolutionGeniviLESeamlessPairingProxy(void)
{
   return _evolutionGeniviLESeamlessPairingProxy;
}

IOrgFreedesktopDbusObjectManagerSendRequestIf* CcDbusIfController::getObjectManagerProxy(void)
{
   return _objectManagerProxy;
}

IIntrospectableDbusSendRequestIf* CcDbusIfController::getIntrospectableProxy(void)
{
   return _introspectableProxy;
}

IPropertiesGetAllDbusSendRequestIf* CcDbusIfController::getPropertiesGetAllProxy(void)
{
   return _propertiesGetAllProxy;
}

IWblDbusLastIntendedModeSendRequestIf* CcDbusIfController::getWblLastIntendedModeProxy(void)
{
   return _lastIntendedModeProxy;
}

IWblDbusConflictManagementSendRequestIf* CcDbusIfController::getWblConflictManagementProxy(void)
{
   return _conflictManagementProxy;
}

IWblDbusWifiSetupSendRequestIf* CcDbusIfController::getWblWifiSetupProxy(void)
{
   return _wifiSetupProxy;
}

} //ccdbusif
