/**
 * @file ConnManDbusServiceProperties.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief ConnMan DBUS Properties for Service.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details ConnMan DBUS Properties for Service.
 */

#include "ConnManDbusServiceProperties.h"

namespace ccdbusif {
namespace connman {

ConnManDbusServiceProperties::ConnManDbusServiceProperties() : DbusProperties("Service"),
   // properties
   _propertyType("Type"),
   _propertySecurity("Security"),
   _propertyState("State"),
   _propertyError("Error"),
   _propertyStrength("Strength"),
   _propertyFavorite("Favorite"),
   _propertyImmutable("Immutable"),
   _propertyAutoConnect("AutoConnect"),
   _propertyName("Name"),
   _propertyRoaming("Roaming"),
   _propertyEthernet("Ethernet"),
   _propertyIPv4("IPv4"),
   _propertyIPv4Configuration("IPv4.Configuration"),
   _propertyIPv6("IPv6"),
   _propertyIPv6Configuration("IPv6.Configuration"),
   _propertyNameservers("Nameservers"),
   _propertyNameserversConfiguration("Nameservers.Configuration"),
   _propertyTimeservers("Timeservers"),
   _propertyTimeserversConfiguration("Timeservers.Configuration"),
   _propertyDomains("Domains"),
   _propertyDomainsConfiguration("Domains.Configuration"),
   _propertyProxy("Proxy"),
   _propertyProxyConfiguration("Proxy.Configuration"),
   _propertyProvider("Provider"),
   // -------------------------------------------------------------------------
   _propertyEthernetMethod("Method"),
   _propertyEthernetInterface("Interface"),
   _propertyEthernetAddress("Address"),
   _propertyEthernetMtu("MTU"),
   _propertyEthernetSpeed("Speed"),
   _propertyEthernetDuplex("Duplex"),
   // -------------------------------------------------------------------------
   _propertyIpv4Method("Method"),
   _propertyIpv4Address("Address"),
   _propertyIpv4Netmask("Netmask"),
   _propertyIpv4Gateway("Gateway"),
   // -------------------------------------------------------------------------
   _propertyIpv6Method("Method"),
   _propertyIpv6Address("Address"),
   _propertyIpv6PrefixLength("PrefixLength"),
   _propertyIpv6Gateway("Gateway"),
   _propertyIpv6Privacy("Privacy"),
   // -------------------------------------------------------------------------
   _propertyProxyMethod("Method"),
   _propertyProxyURL("URL"),
   _propertyProxyServers("Servers"),
   _propertyProxyExcludes("Excludes"),
   // -------------------------------------------------------------------------
   _propertyProviderHost("Host"),
   _propertyProviderDomain("Domain"),
   _propertyProviderName("Name"),
   _propertyProviderType("Type"),

   // supported state values
   _stateIdle("idle"),
   _stateFailure("failure"),
   _stateAssociation("association"),
   _stateConfiguration("configuration"),
   _stateReady("ready"),
   _stateDisconnect("disconnect"),
   _stateOnline("online"),

   // supported type values
   _typeSystem("system"),
   _typeEthernet("ethernet"),
   _typeWifi("wifi"),
   _typeBluetooth("bluetooth"),
   _typeCellular("cellular"),
   _typeGps("gps"),
   _typeVpn("vpn"),
   _typeGadget("gadget")
{
   _string2EnumList[_propertyType] = (int)SERVICE_TYPE;
   _string2EnumList[_propertySecurity] = (int)SERVICE_SECURITY;
   _string2EnumList[_propertyState] = (int)SERVICE_STATE;
   _string2EnumList[_propertyError] = (int)SERVICE_ERROR;
   _string2EnumList[_propertyStrength] = (int)SERVICE_STRENGTH;
   _string2EnumList[_propertyFavorite] = (int)SERVICE_FAVORITE;
   _string2EnumList[_propertyImmutable] = (int)SERVICE_IMMUTABLE;
   _string2EnumList[_propertyAutoConnect] = (int)SERVICE_AUTOCONNECT;
   _string2EnumList[_propertyName] = (int)SERVICE_NAME;
   _string2EnumList[_propertyRoaming] = (int)SERVICE_ROAMING;
   _string2EnumList[_propertyEthernet] = (int)SERVICE_ETHERNET;
   _string2EnumList[_propertyIPv4] = (int)SERVICE_IPV4;
   _string2EnumList[_propertyIPv4Configuration] = (int)SERVICE_IPV4_CONFIGURATION;
   _string2EnumList[_propertyIPv6] = (int)SERVICE_IPV6;
   _string2EnumList[_propertyIPv6Configuration] = (int)SERVICE_IPV6_CONFIGURATION;
   _string2EnumList[_propertyNameservers] = (int)SERVICE_NAMESERVERS;
   _string2EnumList[_propertyNameserversConfiguration] = (int)SERVICE_NAMESERVERS_CONFIGURATION;
   _string2EnumList[_propertyTimeservers] = (int)SERVICE_TIMESERVERS;
   _string2EnumList[_propertyTimeserversConfiguration] = (int)SERVICE_TIMESERVERS_CONFIGURATION;
   _string2EnumList[_propertyDomains] = (int)SERVICE_DOMAINS;
   _string2EnumList[_propertyDomainsConfiguration] = (int)SERVICE_DOMAINS_CONFIGURATION;
   _string2EnumList[_propertyProxy] = (int)SERVICE_PROXY;
   _string2EnumList[_propertyProxyConfiguration] = (int)SERVICE_PROXY_CONFIGURATION;
   _string2EnumList[_propertyProvider] = (int)SERVICE_PROVIDER;
   // -------------------------------------------------------------------------
   _string2EnumList[_propertyEthernetMethod] = (int)SERVICE_ETHERNET_METHOD;
   _string2EnumList[_propertyEthernetInterface] = (int)SERVICE_ETHERNET_INTERFACE;
   _string2EnumList[_propertyEthernetAddress] = (int)SERVICE_ETHERNET_ADDRESS;
   _string2EnumList[_propertyEthernetMtu] = (int)SERVICE_ETHERNET_MTU;
   _string2EnumList[_propertyEthernetSpeed] = (int)SERVICE_ETHERNET_SPEED;
   _string2EnumList[_propertyEthernetDuplex] = (int)SERVICE_ETHERNET_DUPLEX;
   // -------------------------------------------------------------------------
   // do not add following entries else property parsing can lead to wrong enum value
#if 0
   _string2EnumList[_propertyIpv4Method] = (int)SERVICE_IPV4_METHOD;
   _string2EnumList[_propertyIpv4Address] = (int)SERVICE_IPV4_ADDRESS;
   _string2EnumList[_propertyIpv4Netmask] = (int)SERVICE_IPV4_NETMASK;
   _string2EnumList[_propertyIpv4Gateway] = (int)SERVICE_IPV4_GATEWAY;
   // -------------------------------------------------------------------------
   _string2EnumList[_propertyIpv6Method] = (int)SERVICE_IPV6_METHOD;
   _string2EnumList[_propertyIpv6Address] = (int)SERVICE_IPV6_ADDRESS;
   _string2EnumList[_propertyIpv6PrefixLength] = (int)SERVICE_IPV6_PREFIX_LENGTH;
   _string2EnumList[_propertyIpv6Gateway] = (int)SERVICE_IPV6_GATEWAY;
   _string2EnumList[_propertyIpv6Privacy] = (int)SERVICE_IPV6_PRIVACY;
   // -------------------------------------------------------------------------
   _string2EnumList[_propertyProxyMethod] = (int)SERVICE_PROXY_METHOD;
   _string2EnumList[_propertyProxyURL] = (int)SERVICE_PROXY_URL;
   _string2EnumList[_propertyProxyServers] = (int)SERVICE_PROXY_SERVERS;
   _string2EnumList[_propertyProxyExcludes] = (int)SERVICE_PROXY_EXCLUDES;
   // -------------------------------------------------------------------------
   _string2EnumList[_propertyProviderHost] = (int)SERVICE_PROVIDER_HOST;
   _string2EnumList[_propertyProviderDomain] = (int)SERVICE_PROVIDER_DOMAIN;
   _string2EnumList[_propertyProviderName] = (int)SERVICE_PROVIDER_NAME;
   _string2EnumList[_propertyProviderType] = (int)SERVICE_PROVIDER_TYPE;
#endif

   _enum2StringList[(int)SERVICE_TYPE] = &_propertyType;
   _enum2StringList[(int)SERVICE_SECURITY] = &_propertySecurity;
   _enum2StringList[(int)SERVICE_STATE] = &_propertyState;
   _enum2StringList[(int)SERVICE_ERROR] = &_propertyError;
   _enum2StringList[(int)SERVICE_STRENGTH] = &_propertyStrength;
   _enum2StringList[(int)SERVICE_FAVORITE] = &_propertyFavorite;
   _enum2StringList[(int)SERVICE_IMMUTABLE] = &_propertyImmutable;
   _enum2StringList[(int)SERVICE_AUTOCONNECT] = &_propertyAutoConnect;
   _enum2StringList[(int)SERVICE_NAME] = &_propertyName;
   _enum2StringList[(int)SERVICE_ROAMING] = &_propertyRoaming;
   _enum2StringList[(int)SERVICE_ETHERNET] = &_propertyEthernet;
   _enum2StringList[(int)SERVICE_IPV4] = &_propertyIPv4;
   _enum2StringList[(int)SERVICE_IPV4_CONFIGURATION] = &_propertyIPv4Configuration;
   _enum2StringList[(int)SERVICE_IPV6] = &_propertyIPv6;
   _enum2StringList[(int)SERVICE_IPV6_CONFIGURATION] = &_propertyIPv6Configuration;
   _enum2StringList[(int)SERVICE_NAMESERVERS] = &_propertyNameservers;
   _enum2StringList[(int)SERVICE_NAMESERVERS_CONFIGURATION] = &_propertyNameserversConfiguration;
   _enum2StringList[(int)SERVICE_TIMESERVERS] = &_propertyTimeservers;
   _enum2StringList[(int)SERVICE_TIMESERVERS_CONFIGURATION] = &_propertyTimeserversConfiguration;
   _enum2StringList[(int)SERVICE_DOMAINS] = &_propertyDomains;
   _enum2StringList[(int)SERVICE_DOMAINS_CONFIGURATION] = &_propertyDomainsConfiguration;
   _enum2StringList[(int)SERVICE_PROXY] = &_propertyProxy;
   _enum2StringList[(int)SERVICE_PROXY_CONFIGURATION] = &_propertyProxyConfiguration;
   _enum2StringList[(int)SERVICE_PROVIDER] = &_propertyProvider;
   // -------------------------------------------------------------------------
   _enum2StringList[(int)SERVICE_ETHERNET_METHOD] = &_propertyEthernetMethod;
   _enum2StringList[(int)SERVICE_ETHERNET_INTERFACE] = &_propertyEthernetInterface;
   _enum2StringList[(int)SERVICE_ETHERNET_ADDRESS] = &_propertyEthernetAddress;
   _enum2StringList[(int)SERVICE_ETHERNET_MTU] = &_propertyEthernetMtu;
   _enum2StringList[(int)SERVICE_ETHERNET_SPEED] = &_propertyEthernetSpeed;
   _enum2StringList[(int)SERVICE_ETHERNET_DUPLEX] = &_propertyEthernetDuplex;
   // -------------------------------------------------------------------------
   // do not add following entries else property parsing can lead to wrong enum value
#if 0
   _enum2StringList[(int)SERVICE_IPV4_METHOD] = &_propertyIpv4Method;
   _enum2StringList[(int)SERVICE_IPV4_ADDRESS] = &_propertyIpv4Address;
   _enum2StringList[(int)SERVICE_IPV4_NETMASK] = &_propertyIpv4Netmask;
   _enum2StringList[(int)SERVICE_IPV4_GATEWAY] = &_propertyIpv4Gateway;
   // -------------------------------------------------------------------------
   _enum2StringList[(int)SERVICE_IPV6_METHOD] = &_propertyIpv6Method;
   _enum2StringList[(int)SERVICE_IPV6_ADDRESS] = &_propertyIpv6Address;
   _enum2StringList[(int)SERVICE_IPV6_PREFIX_LENGTH] = &_propertyIpv6PrefixLength;
   _enum2StringList[(int)SERVICE_IPV6_GATEWAY] = &_propertyIpv6Gateway;
   _enum2StringList[(int)SERVICE_IPV6_PRIVACY] = &_propertyIpv6Privacy;
   // -------------------------------------------------------------------------
   _enum2StringList[(int)SERVICE_PROXY_METHOD] = &_propertyProxyMethod;
   _enum2StringList[(int)SERVICE_PROXY_URL] = &_propertyProxyURL;
   _enum2StringList[(int)SERVICE_PROXY_SERVERS] = &_propertyProxyServers;
   _enum2StringList[(int)SERVICE_PROXY_EXCLUDES] = &_propertyProxyExcludes;
   // -------------------------------------------------------------------------
   _enum2StringList[(int)SERVICE_PROVIDER_HOST] = &_propertyProviderHost;
   _enum2StringList[(int)SERVICE_PROVIDER_DOMAIN] = &_propertyProviderDomain;
   _enum2StringList[(int)SERVICE_PROVIDER_NAME] = &_propertyProviderName;
   _enum2StringList[(int)SERVICE_PROVIDER_TYPE] = &_propertyProviderType;
#endif

   _state2EnumList[_stateIdle] = (int)SERVICE_STATE_IDLE;
   _state2EnumList[_stateFailure] = (int)SERVICE_STATE_FAILURE;
   _state2EnumList[_stateAssociation] = (int)SERVICE_STATE_ASSOCIATION;
   _state2EnumList[_stateConfiguration] = (int)SERVICE_STATE_CONFIGURATION;
   _state2EnumList[_stateReady] = (int)SERVICE_STATE_READY;
   _state2EnumList[_stateDisconnect] = (int)SERVICE_STATE_DISCONNECT;
   _state2EnumList[_stateOnline] = (int)SERVICE_STATE_ONLINE;

   _enum2StateList[(int)SERVICE_STATE_IDLE] = &_stateIdle;
   _enum2StateList[(int)SERVICE_STATE_FAILURE] = &_stateFailure;
   _enum2StateList[(int)SERVICE_STATE_ASSOCIATION] = &_stateAssociation;
   _enum2StateList[(int)SERVICE_STATE_CONFIGURATION] = &_stateConfiguration;
   _enum2StateList[(int)SERVICE_STATE_READY] = &_stateReady;
   _enum2StateList[(int)SERVICE_STATE_DISCONNECT] = &_stateDisconnect;
   _enum2StateList[(int)SERVICE_STATE_ONLINE] = &_stateOnline;

   _type2EnumList[_typeSystem] = (int)SERVICE_TYPE_SYSTEM;
   _type2EnumList[_typeEthernet] = (int)SERVICE_TYPE_ETHERNET;
   _type2EnumList[_typeWifi] = (int)SERVICE_TYPE_WIFI;
   _type2EnumList[_typeBluetooth] = (int)SERVICE_TYPE_BLUETOOTH;
   _type2EnumList[_typeCellular] = (int)SERVICE_TYPE_CELLULAR;
   _type2EnumList[_typeGps] = (int)SERVICE_TYPE_GPS;
   _type2EnumList[_typeVpn] = (int)SERVICE_TYPE_VPN;
   _type2EnumList[_typeGadget] = (int)SERVICE_TYPE_GADGET;

   _enum2TypeList[(int)SERVICE_TYPE_SYSTEM] = &_typeSystem;
   _enum2TypeList[(int)SERVICE_TYPE_ETHERNET] = &_typeEthernet;
   _enum2TypeList[(int)SERVICE_TYPE_WIFI] = &_typeWifi;
   _enum2TypeList[(int)SERVICE_TYPE_BLUETOOTH] = &_typeBluetooth;
   _enum2TypeList[(int)SERVICE_TYPE_CELLULAR] = &_typeCellular;
   _enum2TypeList[(int)SERVICE_TYPE_GPS] = &_typeGps;
   _enum2TypeList[(int)SERVICE_TYPE_VPN] = &_typeVpn;
   _enum2TypeList[(int)SERVICE_TYPE_GADGET] = &_typeGadget;
}

ConnManDbusServiceProperties::~ConnManDbusServiceProperties()
{
}

ServiceProperty ConnManDbusServiceProperties::getProperty2Enum(const ::std::string& value) const
{
   return (ServiceProperty)convertString2Enum(value);
}

ServiceState ConnManDbusServiceProperties::getState2Enum(const ::std::string& value) const
{
   return (ServiceState)convertString2Enum(_state2EnumList, value);
}

ServiceType ConnManDbusServiceProperties::getType2Enum(const ::std::string& value) const
{
   return (ServiceType)convertString2Enum(_type2EnumList, value);
}

const ::std::string& ConnManDbusServiceProperties::getProperty2String(const ServiceProperty value) const
{
   return convertEnum2String((const int)value);
}

const ::std::string& ConnManDbusServiceProperties::getProperty2StringTrace(const ServiceProperty value) const
{
   if(SERVICE_UNKNOWN <= value)
   {
      return _unknownString;
   }

   const ::std::string& result = convertEnum2String((const int)value, false);

   if(_dummyString == result)
   {
      return _unknownString;
   }

   return result;
}

const ::std::string& ConnManDbusServiceProperties::getState2String(const ServiceState value) const
{
   return convertEnum2String(_enum2StateList, (const int)value);
}

const ::std::string& ConnManDbusServiceProperties::getState2StringTrace(const ServiceState value) const
{
   if(SERVICE_STATE_UNKNOWN <= value)
   {
      return _unknownString;
   }

   const ::std::string& result = convertEnum2String(_enum2StateList, (const int)value, false);

   if(_dummyString == result)
   {
      return _unknownString;
   }

   return result;
}

const ::std::string& ConnManDbusServiceProperties::getType2String(const ServiceType value) const
{
   return convertEnum2String(_enum2TypeList, (const int)value);
}

const ::std::string& ConnManDbusServiceProperties::getType2StringTrace(const ServiceType value) const
{
   if(SERVICE_TYPE_UNKNOWN <= value)
   {
      return _unknownString;
   }

   const ::std::string& result = convertEnum2String(_enum2TypeList, (const int)value, false);

   if(_dummyString == result)
   {
      return _unknownString;
   }

   return result;
}

} //connman
} //ccdbusif
