/**
 * @file ConnManDbusManagerProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief ConnMan DBUS Proxy for Manager.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details ConnMan DBUS proxy for Manager.
 */

#include "ConnManDbusManagerProxyIf.h"
#include "IConnManDbusManagerCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "ICcDbusIfControllerClient.h"
#include "CcDbusIfCreateProxyWorkItem.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

using namespace ::net::connman::Manager;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ConnManDbusManagerProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

ConnManDbusManagerProxyIf::ConnManDbusManagerProxyIf()
{
   // _managerProxy is set later
   _callbackIf = NULL;
}

ConnManDbusManagerProxyIf::ConnManDbusManagerProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _managerProxy is set later
   _callbackIf = NULL;
}

ConnManDbusManagerProxyIf::~ConnManDbusManagerProxyIf()
{
   _callbackIf = NULL;
}

void ConnManDbusManagerProxyIf::createProxy(void)
{
   ETG_TRACE_USR3((" createProxy(): ConnManDbusManager"));

   if(NULL != _callbackIf)
   {
      if(true == _enableProxy)
      {
         _managerProxy = ManagerProxy::createProxy("evoConnManManagerPort", *this);
      }
   }
}

void ConnManDbusManagerProxyIf::destroyProxy(void)
{
   ETG_TRACE_USR3((" destroyProxy(): Manager"));

   if(NULL != _managerProxy.get())
   {
      _managerProxy->sendDeregisterAll();
      _managerProxy.reset();
   }
}

void ConnManDbusManagerProxyIf::destroyAllRuntimeProxies(void)
{
   ETG_TRACE_USR3((" destroyAllRuntimeProxies(): Manager"));

   // nothing to do because this is a static proxy
}

void ConnManDbusManagerProxyIf::setCallbackIf(IConnManDbusManagerCallbackIf* callbackIf, bool enableProxy)
{
   _callbackIf = callbackIf;

   if(NULL != _callbackIf)
   {
      _enableProxy = enableProxy;
   }

   // callback interface is now available => create proxy now
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfCreateProxyWorkItem(this));
   }
}

// ServiceAvailableIF implementation --- start
void ConnManDbusManagerProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onAvailable(): Manager"));

   if((NULL != _managerProxy.get()) && (proxy == _managerProxy))
   {
      if(NULL != _callbackIf)
      {
         _managerProxy->sendPropertyChangedRegister(*_callbackIf);
         // not used --- _managerProxy->sendTechnologyAddedRegister(*_callbackIf);
         // not used --- _managerProxy->sendTechnologyRemovedRegister(*_callbackIf);
         _managerProxy->sendServicesChangedRegister(*_callbackIf);

         _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(), _managerProxy->getDBusObjectPath());
      }
   }
}

void ConnManDbusManagerProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onUnavailable(): Manager"));

   if((NULL != _managerProxy.get()) && (proxy == _managerProxy))
   {
      _managerProxy->sendDeregisterAll();

      if(NULL != _callbackIf)
      {
         _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(), _managerProxy->getDBusObjectPath());
      }
   }
}
// ServiceAvailableIF implementation --- end

// IConnManDbusManagerProxyIf implementation --- start
act_t ConnManDbusManagerProxyIf::sendGetPropertiesRequest(void)
{
   if((NULL != _callbackIf) && (NULL != _managerProxy.get()))
   {
      return _managerProxy->sendGetPropertiesRequest(*_callbackIf);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t ConnManDbusManagerProxyIf::sendGetTechnologiesRequest(void)
{
   if((NULL != _callbackIf) && (NULL != _managerProxy.get()))
   {
      return _managerProxy->sendGetTechnologiesRequest(*_callbackIf);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t ConnManDbusManagerProxyIf::sendGetServicesRequest(void)
{
   if((NULL != _callbackIf) && (NULL != _managerProxy.get()))
   {
      return _managerProxy->sendGetServicesRequest(*_callbackIf);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

// IConnManDbusManagerProxyIf implementation --- end

} //ccdbusif
