/**
 * @file ProxyMetaData.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief DBUS Proxy Meta Data.
 *
 * @copyright (C) 2015-2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation for meta data of a DBUS proxy interface.
 */

#include "ProxyMetaData.h"

namespace ccdbusif {

ProxyMetaData::ProxyMetaData() :
objPath(),
busName(),
busType(BUS_TYPE_UNKNOWN)
{
}

ProxyMetaData::ProxyMetaData(const ::std::string& inputObjPath, const ::std::string& inputBusName, const DbusBusType inputBusType) :
objPath(inputObjPath),
busName(inputBusName),
busType(inputBusType)
{
}

ProxyMetaData::ProxyMetaData(const ProxyMetaData& ref) :
objPath(ref.objPath),
busName(ref.busName),
busType(ref.busType)
{
}

ProxyMetaData& ProxyMetaData::operator=(const ProxyMetaData& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   objPath = ref.objPath;
   busName = ref.busName;
   busType = ref.busType;

   return *this;
}

bool ProxyMetaData::operator==(const ProxyMetaData& ref) const
{
   return (0 == compare(ref));
}

bool ProxyMetaData::operator!=(const ProxyMetaData& ref) const
{
   return (0 != compare(ref));
}

bool ProxyMetaData::operator<(const ProxyMetaData& ref) const
{
   return (0 > compare(ref));
}

bool ProxyMetaData::operator>(const ProxyMetaData& ref) const
{
   return (0 < compare(ref));
}

ProxyMetaData::~ProxyMetaData()
{
}

int ProxyMetaData::compare(const ProxyMetaData& ref) const
{
   int result;

   result = objPath.compare(ref.objPath);
   if(result < 0)
   {
      return -1;
   }
   else if(result > 0)
   {
      return 1;
   }

   // object path is equal

   result = busName.compare(ref.busName);
   if(result < 0)
   {
      return -1;
   }
   else if(result > 0)
   {
      return 1;
   }

   // bus name is equal

   if(busType < ref.busType)
   {
      return -1;
   }
   else if(busType > ref.busType)
   {
      return 1;
   }

   // bus type is equal

   return 0;
}

} //ccdbusif
