/**
 * @file CommonBaseProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Common Base Proxy.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of base class for proxies.
 */

#include "CommonBaseProxyIf.h"
#include "ICcDbusIfControllerClient.h"
#include "CreateProxyWorkItem.h"
#include "DestroyProxyWorkItem.h"
#include "FwAssert.h"
#include "FwUtils.h"
#include "FwFormattedDataPrint.h"
#include "TraceClasses.h"
#include "FwTrace.h"
#include <cstdio>
#include <cstring>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/CommonBaseProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

CommonBaseProxyIf::CommonBaseProxyIf(ICcDbusIfControllerClient* client) :
_enableProxy(false),
_controllerClient(client)
{
}

CommonBaseProxyIf::CommonBaseProxyIf(const CommonBaseProxyIf& ref) :
_enableProxy(ref._enableProxy),
_controllerClient(ref._controllerClient)
{
}

CommonBaseProxyIf& CommonBaseProxyIf::operator=(const CommonBaseProxyIf& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   _enableProxy = ref._enableProxy;
   _controllerClient = ref._controllerClient;

   return *this;
}

CommonBaseProxyIf::~CommonBaseProxyIf()
{
   _controllerClient = 0;
}

void CommonBaseProxyIf::processCreateProxy(ICreateProxy* creator, const bool createDirectly, const unsigned int callbackId, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   if(true == createDirectly)
   {
      // in ASF thread context we can directly create the proxy
      internalCreateProxy(callbackId, objPath, busName, busType);
   }
   else
   {
      // use work item to process in ASF thread context
      FW_IF_NULL_PTR_RETURN(_controllerClient);
      FW_IF_NULL_PTR_RETURN(creator);
      _controllerClient->pushWorkItem(new CreateProxyWorkItem(creator, callbackId, objPath, busName, busType));
   }
}

void CommonBaseProxyIf::processDestroyProxy(IDestroyProxy* creator, const bool destroyDirectly, const unsigned int callbackId, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   if(true == destroyDirectly)
   {
      // in ASF thread context we can directly destroy the proxy
      internalDestroyProxy(callbackId, objPath, busName, busType);
   }
   else
   {
      // use work item to process in ASF thread context
      FW_IF_NULL_PTR_RETURN(_controllerClient);
      FW_IF_NULL_PTR_RETURN(creator);
      _controllerClient->pushWorkItem(new DestroyProxyWorkItem(creator, callbackId, objPath, busName, busType));
   }
}

void CommonBaseProxyIf::printThreadInfo(const ::std::string& description) const
{
   char buffer[240];
   char threadName[17];
   ::fw::FormattedOutputPid processId(::fw::getProcessId());
   ::fw::FormattedOutputPid threadId(::fw::getThreadId());
   (void)snprintf(buffer, sizeof(buffer), "%s: PID=%s TID=%s NAME=%s", description.c_str(), processId.c_str(), threadId.c_str(), ::fw::getThreadName(threadName, sizeof(threadName)));

   ETG_TRACE_ERR((" ### %s", buffer));
}

} //ccdbusif
