/**
 * @file ProxyManager.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Proxy management.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Proxy management.
 */

#ifndef _PROXY_MANAGEMENT_H_
#define _PROXY_MANAGEMENT_H_

#include "ProxyMetaData.h"
#include "boost/shared_ptr.hpp"

#define FW_S_IMPORT_INTERFACE_ASSERT
#include "framework_if_if.h"

namespace ccdbusif {

/**
 * Type definition for DBUS proxy manager.
 * HINT: No locking mechanisms are implemented. Therefore access to this object is only allowed from same thread or
 *       locking mechanisms are applied outside.
 */
template < class T >
class ProxyManager
{
public:
   typedef ::boost::shared_ptr< T > ProxyT;

   ProxyManager()
   {
      // _proxyMetaData2ProxyInstance
      _maxNumberOfProxyInstances = 0;
      _lastCount = 0;
   }

   ProxyManager(const ProxyManager& ref)
   {
      _proxyMetaData2ProxyInstance = ref._proxyMetaData2ProxyInstance;
      _maxNumberOfProxyInstances = ref._maxNumberOfProxyInstances;
      _lastCount = ref._lastCount;
   }

   ProxyManager& operator=(const ProxyManager& ref)
   {
      if(this == &ref)
      {
         return *this;
      }

      _proxyMetaData2ProxyInstance = ref._proxyMetaData2ProxyInstance;
      _maxNumberOfProxyInstances = ref._maxNumberOfProxyInstances;
      _lastCount = ref._lastCount;

      return *this;
   }

   bool operator==(const ProxyManager& ref) const
   {
      bool result = true;

      result = (true == result) && (_proxyMetaData2ProxyInstance == ref._proxyMetaData2ProxyInstance);
      result = (true == result) && (_maxNumberOfProxyInstances == ref._maxNumberOfProxyInstances);
      result = (true == result) && (_lastCount == ref._lastCount);

      return result;
   }

   bool operator!=(const ProxyManager& ref) const
   {
      return !(operator==(ref));
   }

   virtual ~ProxyManager()
   {
   }

   void setMaxNumberOfProxyInstances(const unsigned int number)
   {
      // 0 means unlimited
      _maxNumberOfProxyInstances = number;
   }

   unsigned int getNumberOfProxyInstances(void) const
   {
      return (unsigned int)_proxyMetaData2ProxyInstance.size();
   }

   bool isInstanceAvailable(const ProxyMetaData& metaData)
   {
      typename ::std::map<ProxyMetaData, ProxyT>::iterator it = _proxyMetaData2ProxyInstance.find(metaData);

      return (_proxyMetaData2ProxyInstance.end() != it);
   }

   bool isInstanceAvailable(const ::std::string& objPath)
   {
      ProxyMetaData metaData;
      metaData.objPath = objPath;

      return isInstanceAvailable(metaData);
   }

   bool isInstanceAvailable(const ::std::string& busName, const ::std::string& objPath, const DbusBusType busType, void* user)
   {
      (void)(user);
      ProxyMetaData metaData;
      metaData.busName = busName;
      metaData.objPath = objPath;
      metaData.busType = busType;

      return isInstanceAvailable(metaData);
   }

   bool isProxyAvailable(ProxyT& instance, const ProxyMetaData& metaData)
   {
      ProxyMetaData data1;
      ProxyMetaData data2;

      data1 = metaData;

      // do not use find because user is set to NULL; first found entry is used; TODO: maybe remove user and improve
      for(typename ::std::map<ProxyMetaData, ProxyT>::iterator it = _proxyMetaData2ProxyInstance.begin(); it != _proxyMetaData2ProxyInstance.end(); ++it)
      {
         data2 = it->first;

         if(data1 == data2)
         {
            // related proxy already exists
            instance = it->second;
            return true;
         }
      }

      return false;
   }

   bool isProxyAvailable(ProxyT& instance, const ::std::string& objPath)
   {
      ProxyMetaData metaData;
      metaData.objPath = objPath;

      return isProxyAvailable(instance, metaData);
   }

   bool isProxyAvailable(ProxyT& instance, const ::std::string& busName, const ::std::string& objPath, const DbusBusType busType, void* user)
   {
      (void)(user);
      ProxyMetaData metaData;
      metaData.busName = busName;
      metaData.objPath = objPath;
      metaData.busType = busType;

      return isProxyAvailable(instance, metaData);
   }

   bool isInstanceAvailable(ProxyT& instance, const ProxyMetaData& metaData)
   {
      typename ::std::map<ProxyMetaData, ProxyT>::iterator it = _proxyMetaData2ProxyInstance.find(metaData);

      if(_proxyMetaData2ProxyInstance.end() != it)
      {
         instance = it->second;

         return true;
      }

      return false;
   }

   bool isInstanceAvailable(ProxyT& instance, const ::std::string& objPath)
   {
      ProxyMetaData metaData;
      metaData.objPath = objPath;

      return isInstanceAvailable(instance, metaData);
   }

   bool isInstanceAvailable(ProxyT& instance, const ::std::string& busName, const ::std::string& objPath, const DbusBusType busType, void* user)
   {
      (void)(user);
      ProxyMetaData metaData;
      metaData.busName = busName;
      metaData.objPath = objPath;
      metaData.busType = busType;

      return isInstanceAvailable(instance, metaData);
   }

   void addProxyInstance(const ProxyMetaData& metaData, ProxyT& instance)
   {
      typename ::std::map<ProxyMetaData, ProxyT>::iterator it = _proxyMetaData2ProxyInstance.find(metaData);

      if(_proxyMetaData2ProxyInstance.end() != it)
      {
         // entry is already stored => can happen because function is called after reception of different signals
      }
      else
      {
         // entry is not stored
         _proxyMetaData2ProxyInstance[metaData] = instance;

         // check size
         if(0 < _maxNumberOfProxyInstances)
         {
            if(_proxyMetaData2ProxyInstance.size() > _maxNumberOfProxyInstances)
            {
               // should never happen - forgot to remove instance?
               // #error_indication
               FW_NORMAL_ASSERT_ALWAYS();
            }
         }
      }
   }

   void addProxyInstance(const ::std::string& objPath, ProxyT& instance)
   {
      ProxyMetaData metaData;
      metaData.objPath = objPath;

      addProxyInstance(metaData, instance);
   }

   void addProxyInstance(const ::std::string& busName, const ::std::string& objPath, const DbusBusType busType, void* user, ProxyT& instance)
   {
      (void)(user);
      ProxyMetaData metaData;
      metaData.busName = busName;
      metaData.objPath = objPath;
      metaData.busType = busType;

      addProxyInstance(metaData, instance);
   }

   void removeProxyInstance(const ProxyMetaData& metaData)
   {
      typename ::std::map<ProxyMetaData, ProxyT>::iterator it = _proxyMetaData2ProxyInstance.find(metaData);

      if(_proxyMetaData2ProxyInstance.end() == it)
      {
         // entry is already erased => can happen
      }
      else
      {
         // entry is not erased
         it->second.reset();
         _lastCount = it->second.use_count();
         _proxyMetaData2ProxyInstance.erase(it);
      }
   }

   void removeProxyInstance(const ::std::string& objPath)
   {
      ProxyMetaData metaData;
      metaData.objPath = objPath;

      removeProxyInstance(metaData);
   }

   void removeProxyInstance(const ::std::string& busName, const ::std::string& objPath, const DbusBusType busType, void* user)
   {
      (void)(user);
      ProxyMetaData metaData;
      metaData.busName = busName;
      metaData.objPath = objPath;
      metaData.busType = busType;

      removeProxyInstance(metaData);
   }

   const ProxyT& operator[](const unsigned int index) const
   {
      if(_proxyMetaData2ProxyInstance.size() <= index)
      {
         // should never happen - forgot to check size before?
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
      }

      typename ::std::map<ProxyMetaData, ProxyT>::const_iterator it = _proxyMetaData2ProxyInstance.begin() + index;

      // old
      /*
      for(unsigned int i = 0; i < index; i++)
      {
         ++it;
      }
      */

      return it->second;
   }

   ProxyT& operator[](const unsigned int index)
   {
      if(_proxyMetaData2ProxyInstance.size() <= index)
      {
         // should never happen - forgot to check size before?
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
      }

      typename ::std::map<ProxyMetaData, ProxyT>::iterator it = _proxyMetaData2ProxyInstance.begin();

      for(unsigned int i = 0; i < index; i++)
      {
         ++it;
      }

      return it->second;
   }

   const ProxyMetaData& getMetaData(const unsigned int index)
   {
      if(_proxyMetaData2ProxyInstance.size() <= index)
      {
         // should never happen - forgot to check size before?
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
      }

      typename ::std::map<ProxyMetaData, ProxyT>::const_iterator it = _proxyMetaData2ProxyInstance.begin();

      for(unsigned int i = 0; i < index; i++)
      {
         ++it;
      }

      return it->first;
   }

   void clear(void)
   {
      _proxyMetaData2ProxyInstance.clear();
   }

   void resetAllProxiesAndClear(void)
   {
      for(typename ::std::map<ProxyMetaData, ProxyT>::iterator it = _proxyMetaData2ProxyInstance.begin(); it != _proxyMetaData2ProxyInstance.end(); ++it)
      {
         if(NULL != it->second.get())
         {
            it->second.reset();
         }
      }
      _proxyMetaData2ProxyInstance.clear();
   }

   long getCount(const ProxyMetaData& metaData) const
   {
      typename ::std::map<ProxyMetaData, ProxyT>::const_iterator it = _proxyMetaData2ProxyInstance.find(metaData);

      if(_proxyMetaData2ProxyInstance.end() == it)
      {
         // entry is already erased => can happen
         return 0;
      }
      else
      {
         // entry is not erased
         return it->second.use_count();
      }
   }

   long getCount(const ::std::string& objPath) const
   {
      ProxyMetaData metaData;
      metaData.objPath = objPath;

      return getCount(metaData);
   }

   long getCount(const ::std::string& busName, const ::std::string& objPath, const DbusBusType busType, void* user) const
   {
      (void)(user);
      ProxyMetaData metaData;
      metaData.busName = busName;
      metaData.objPath = objPath;
      metaData.busType = busType;

      return getCount(metaData);
   }

   inline long getLastCount(void) const { return _lastCount; }

   inline void setLastCount(const long value) { _lastCount = value; }

protected:
   ::std::map<ProxyMetaData, ProxyT> _proxyMetaData2ProxyInstance;
   unsigned int _maxNumberOfProxyInstances;
   long _lastCount;
};

} //ccdbusif

#endif //_PROXY_MANAGEMENT_H_
