/**
 * @file DbusParser.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief DBUS Property Parser.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details DBUS Property Parser.
 */

#ifndef _DBUS_PARSER_H_
#define _DBUS_PARSER_H_

#include "CcDbusIfTypes.h"
#include "asf/dbus/DBusVariant.h"

namespace ccdbusif {

// class forward declarations
class IDbusProperty2EnumMapping;

/**
 * DBUS property parser class.
 */
class DbusParser
{
public:
   DbusParser();
   virtual ~DbusParser();

   /**
    * Parse function.
    *
    * @param[out] outProperties: list with parsed properties
    * @param[in] matchingInterfaces: interfaces matching the given properties
    * @param[in] inProperties: map with properties (mutable)
    * @param[in] busType: bus type
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    * @param[in] mapping: reference to mapping instance
    */
   void parseProperties(::std::vector<DbusVariantProperty>& outProperties, const ::std::vector< int >& matchingInterfaces, ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties,
         const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName, const IDbusProperty2EnumMapping* mapping) const;

   /**
    * Parse function.
    *
    * @param[in] data: introspection data
    * @param[in] busType: bus type
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    * @param[in] mapping: reference to mapping instance
    */
   void parseIntrospection(const ::std::string& data,
         const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName, const IDbusProperty2EnumMapping* mapping) const;

   /**
    * Parse function.
    *
    * @param[in] addedObjectPath: added object path
    * @param[in] inInterfaces: map with interfaces (mutable)
    * @param[in] busType: bus type
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    * @param[in] mapping: reference to mapping instance
    */
   void parseInterfacesAdded(const ::std::string& addedObjectPath, ::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > >& inInterfaces,
         const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName, const IDbusProperty2EnumMapping* mapping) const;

   /**
    * Parse function.
    *
    * @param[in] removedObjectPath: removed object path
    * @param[in] inInterfaces: vector with interfaces
    * @param[in] busType: bus type
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    * @param[in] mapping: reference to mapping instance
    */
   void parseInterfacesRemoved(const ::std::string& removedObjectPath, const ::std::vector< ::std::string >& inInterfaces,
         const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName, const IDbusProperty2EnumMapping* mapping) const;

   /**
    * Parse function.
    *
    * @param[out] outPropertyList: parsed property list
    * @param[in] inProperties: map with properties (mutable)
    * @param[in] propertyName: property to be found/parsed
    * @param[in] interface: related interface
    * @param[in] mapping: reference to mapping instance
    *
    * @return = true: property found,
    * @return = false: property not found
    */
   bool findAndParseProperty(::std::vector<DbusVariantProperty>& outPropertyList, ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties, const ::std::string& propertyName, const int interface, const IDbusProperty2EnumMapping* mapping) const;

   /**
    * Parse function.
    *
    * @param[out] outPropertyList: parsed property list
    * @param[in] propertyValue: received property value (mutable)
    * @param[in] propertyName: received property name
    * @param[in] interface: related interface
    * @param[in] mapping: reference to mapping instance
    */
   void parsePropertyValue(::std::vector<DbusVariantProperty>& outPropertyList, ::asf::dbus::DBusVariant& propertyValue, const ::std::string& propertyName, const int interface, const IDbusProperty2EnumMapping* mapping) const;

   /**
    * Adds an item to dictionary.
    *
    * @param[out] dictionary: dictionary
    * @param[in] key: key
    * @param[in] value: value
    */
   void addDictionaryItem(::std::map< ::std::string, ::asf::dbus::DBusVariant >& dictionary, const ::std::string& key, const DbusVariant& value) const;

   /**
    * Adds variant value to parameter.
    *
    * @param[out] item: parameter
    * @param[in] value: value
    */
   void addVariantValue(::asf::dbus::DBusVariant& item, const DbusVariant& value) const;

   /**
    * Set traces enabled flag.
    *
    * @param[in] enabled: traces enabled flag
    */
   inline void setTraces(const bool enabled) { _tracesEnabled = enabled; }

   /**
    * Get traces enabled flag.
    *
    * @return = traces enabled flag
    */
   inline bool getTraces(void) const { return _tracesEnabled; }

private:
   bool _tracesEnabled; /**< traces enabled flag */

   /**
    * Parse function.
    *
    * @param[out] outProperties: list with parsed properties
    * @param[in] matchingInterfaces: interfaces matching the given properties
    * @param[in] inProperties: map with properties (mutable)
    * @param[in] mapping: reference to mapping instance
    */
   void parsePropertiesInternal(::std::vector<DbusVariantProperty>& outProperties, const ::std::vector< int >& matchingInterfaces, ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties, const IDbusProperty2EnumMapping* mapping) const;

   /**
    * Internal parse function.
    *
    * @param[out] outProperties: list with parsed properties
    * @param[in] interfaceList: interfaces matching the given properties
    * @param[in] iter: iterator
    * @param[in] signature: signature
    * @param[in] property: property
    * @param[in] counter: property counter
    * @param[in] depth: depth
    * @param[in] isDict: dictionary marker
    * @param[in] dictKey: dictionary key
    */
   void parseProperty(::std::vector<DbusVariantProperty>& outProperties, const ::std::vector<const ::std::map< std::string, int >*>& interfaceList, ::DBusMessageIter* iter, const ::std::string& signature, const ::std::string& property, const unsigned int counter, const unsigned int depth, const bool isDict, const ::std::string& dictKey) const;
};

} //ccdbusif

#endif //_DBUS_PARSER_H_
