/**
 * @file CcDbusIfTypes.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Types.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Types for CcDbusIf.
 */

#ifndef _CC_DBUS_IF_TYPES_H_
#define _CC_DBUS_IF_TYPES_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <string>
#include <vector>
#include <map>
#include "asf/core/Types.h"

namespace ccdbusif {

/**
 * Default value for asynchronous completion token (ACT).
 */
enum
{
   DEFAULT_ACT = (act_t)0 /**< default value */
};

/**
 * Type definition for DBUS bus type.
 */
enum DbusBusType           // DBusBusType
{
   BUS_TYPE_SESSION = 0,   // DBUS_BUS_SESSION
   BUS_TYPE_SYSTEM,        // DBUS_BUS_SYSTEM
   BUS_TYPE_ADDRESS,
   // -------------------------------------------------------------------------
   BUS_TYPE_UNKNOWN /**< end marker, must be the last one */
};

/**
 * Type definition for DBUS proxy creation state.
 */
enum DbusProxyCreationState
{
   PROXY_NOT_AVAILABLE = 0,
   PROXY_CREATING,
   PROXY_AVAILABLE,
   // -------------------------------------------------------------------------
   PROXY_UNKNOWN /**< end marker, must be the last one */
};

/**
 * Type definition for DBUS variant data type.
 */
enum DbusVariantDataType
{
   VARIANT_INT8 = 0,
   VARIANT_UINT8,
   VARIANT_INT16,
   VARIANT_UINT16,
   VARIANT_INT32,
   VARIANT_UINT32,
   VARIANT_INT64,
   VARIANT_UINT64,
   VARIANT_ENUM,
   VARIANT_BOOL,
   VARIANT_STRING,
   VARIANT_STRING_ARRAY,
   VARIANT_NONE,
   // -------------------------------------------------------------------------
   VARIANT_UNKNOWN /**< end marker, must be the last one */
};

/**
 * Type definition for DBUS basic variant data.
 */
union DbusIntVariant
{
   int8_t int8Data; /**< int8 data */
   uint8_t uint8Data; /**< uint8 data */
   int16_t int16Data; /**< int16 data */
   uint16_t uint16Data; /**< uint16 data */
   int32_t int32Data; /**< int32 data */
   uint32_t uint32Data; /**< uint32 data */
   int64_t int64Data; /**< int64 data */
   uint64_t uint64Data; /**< uint64 data */
   int enumData; /**< enum data */
   bool boolData; /**< bool data */
   bool empty; /**< empty data */
};

/**
 * Helper class to get string representation of act.
 */
class ActInfo
{
public:
   ActInfo(const act_t token);
   virtual ~ActInfo();
   inline const char* getInfo(void) const { return _buffer; }

protected:
   ActInfo();

private:
   /**
    * String size for asynchronous completion token (ACT).
    * "0x12345678"
    */
   enum
   {
      ACT_INFO_SIZE = 3 + (sizeof(act_t) << 1)
   };

   char _buffer[ACT_INFO_SIZE];
};

//------------------------------------------------------------------------------

/**
 * Type definition for DBUS variant data.
 * HINT: If there is a property containing a dictionary then each entry of the dictionary will be handled as a separate property.
 */
class DbusVariant
{
public:
   DbusVariant();
   DbusVariant(const DbusVariant& ref);
   DbusVariant& operator=(const DbusVariant& ref);
   bool operator==(const DbusVariant& ref) const;
   bool operator!=(const DbusVariant& ref) const;
   virtual ~DbusVariant();
   inline DbusVariantDataType getType(void) const { return _type; }
   inline void setInt8(const int8_t data) { _intData.int8Data = data; _type = VARIANT_INT8; }
   inline void setUInt8(const uint8_t data) { _intData.uint8Data = data; _type = VARIANT_UINT8; }
   inline void setInt16(const int16_t data) { _intData.int16Data = data; _type = VARIANT_INT16; }
   inline void setUInt16(const uint16_t data) { _intData.uint16Data = data; _type = VARIANT_UINT16; }
   inline void setInt32(const int32_t data) { _intData.int32Data = data; _type = VARIANT_INT32; }
   inline void setUInt32(const uint32_t data) { _intData.uint32Data = data; _type = VARIANT_UINT32; }
   inline void setInt64(const int64_t data) { _intData.int64Data = data; _type = VARIANT_INT64; }
   inline void setUInt64(const uint64_t data) { _intData.uint64Data = data; _type = VARIANT_UINT64; }
   inline void setBool(const bool data) { _intData.boolData = data; _type = VARIANT_BOOL; }
   inline void setEnum(const int data) { _intData.enumData = data; _type = VARIANT_ENUM; }
   inline void setString(const ::std::string& data) { _stringData = data; _type = VARIANT_STRING; }
   inline void setStringArray(const ::std::vector<std::string>& data) { _stringArray = data; _type = VARIANT_STRING_ARRAY; }
   inline ::std::vector<std::string>& getStringArrayMutable(void) { _type = VARIANT_STRING_ARRAY; return _stringArray; }
   inline void setNone(void) { _intData.empty = true; _type = VARIANT_NONE; }
   int8_t getInt8(void) const;
   uint8_t getUInt8(void) const;
   int16_t getInt16(void) const;
   uint16_t getUInt16(void) const;
   int32_t getInt32(void) const;
   uint32_t getUInt32(void) const;
   int64_t getInt64(void) const;
   uint64_t getUInt64(void) const;
   bool getBool(void) const;
   int getEnum(void) const;
   const ::std::string& getString(void) const;
   const ::std::vector<std::string>& getStringArray(void) const;
   bool getNone(void) const;
   void reset(void);

private:
   DbusVariantDataType _type; /**< data type */
   DbusIntVariant _intData; /**< integer */
   ::std::string _stringData; /**< string */
   ::std::vector<std::string> _stringArray; /**< string array */
};

//------------------------------------------------------------------------------

/**
 * Type definition for DBUS variant property.
 */
class DbusVariantProperty
{
public:
   DbusVariantProperty();
   DbusVariantProperty(const DbusVariantProperty& ref);
   DbusVariantProperty& operator=(const DbusVariantProperty& ref);
   bool operator==(const DbusVariantProperty& ref) const;
   bool operator!=(const DbusVariantProperty& ref) const;
   virtual ~DbusVariantProperty();
   void reset(void);
   // member with public access
   int propEnum; /**< enum value */
   DbusVariant propData; /**< data */
};

//------------------------------------------------------------------------------

/**
 * Type definition for a DBUS property entry.
 * A DBUS property entry contains a pointer to a string with property name and an enum value.
 */
class DbusPropertyEntry
{
public:
   DbusPropertyEntry();
   DbusPropertyEntry(const DbusPropertyEntry& ref);
   DbusPropertyEntry& operator=(const DbusPropertyEntry& ref);
   bool operator==(const DbusPropertyEntry& ref) const;
   bool operator!=(const DbusPropertyEntry& ref) const;
   bool operator<(const DbusPropertyEntry& ref) const;
   bool operator>(const DbusPropertyEntry& ref) const;
   virtual ~DbusPropertyEntry();
   // member with public access
   const ::std::string* propName; /**< pointer to a string with property name */
   int propEnum; /**< enum value */

protected:
   inline void setPropName(const ::std::string* name) { propName = name; }
   int compare(const DbusPropertyEntry& ref) const;
};

//------------------------------------------------------------------------------

} //ccdbusif

#endif //_CC_DBUS_IF_TYPES_H_
