/**
 * @file WblObjectPathManagerWbl.cpp
 *
 * @par SW-Component
 * State machine for WBL object path manager
 *
 * @brief Implementation of WBL object path manager state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of WBL object path manager state machine.
 */

#include "WblObjectPathManagerWbl.h"

#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WblObjectPathManagerWbl.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace wbl {

WblObjectPathManagerWbl::WblObjectPathManagerWbl() :
_id2ObjectPath(),
_objectPath2Id(),
_usedIds(),
_lowerId(1),
_nextFreeId(_lowerId)
{
   const size_t listSize(20);
   _usedIds.reserve(listSize);
   _usedIds.insert(_usedIds.begin(), listSize, false);
}

WblObjectPathManagerWbl::~WblObjectPathManagerWbl()
{
}

void WblObjectPathManagerWbl::setId2ObjectPath(IN const BTSWblWiFiSetupId id, IN const BTSObjectPath& objPath)
{
   if(_id2ObjectPath.end() == _id2ObjectPath.find(id))
   {
      _id2ObjectPath[id] = objPath;
      _objectPath2Id[objPath] = id;
   }
}

void WblObjectPathManagerWbl::removeId2ObjectPath(IN const BTSObjectPath& objPath)
{
   ::std::map< BTSObjectPath, BTSWblWiFiSetupId >::iterator it1 = _objectPath2Id.find(objPath);
   if(_objectPath2Id.end() != it1)
   {
      ::std::map< BTSWblWiFiSetupId, BTSObjectPath >::iterator it2 = _id2ObjectPath.find(it1->second);
      if(_id2ObjectPath.end() != it2)
      {
         _id2ObjectPath.erase(it2);
      }
      _objectPath2Id.erase(it1);
   }
}

bool WblObjectPathManagerWbl::getObjectPath4Id(OUT BTSObjectPath& objPath, IN const BTSWblWiFiSetupId id) const
{
   bool found(false);

   ::std::map< BTSWblWiFiSetupId, BTSObjectPath >::const_iterator it = _id2ObjectPath.find(id);
   if(_id2ObjectPath.end() != it)
   {
      objPath = it->second;
      found = true;
   }

   return found;
}

bool WblObjectPathManagerWbl::getId4ObjectPath(OUT BTSWblWiFiSetupId& id, IN const BTSObjectPath& objPath) const
{
   bool found(false);

   ::std::map< BTSObjectPath, BTSWblWiFiSetupId >::const_iterator it = _objectPath2Id.find(objPath);
   if(_objectPath2Id.end() != it)
   {
      id = it->second;
      found = true;
   }

   return found;
}

BTSWblWiFiSetupId WblObjectPathManagerWbl::getFreeId(void)
{
   return getNextFreeId();
}

BTSWblWiFiSetupId WblObjectPathManagerWbl::getNextFreeId(void)
{
   const BTSWblWiFiSetupId returnId(_nextFreeId);

   BTSWblWiFiSetupId tmpFreeId = getNextId(_nextFreeId);
   bool found(false);
   while((returnId != tmpFreeId) && (false == found))
   {
      if((tmpFreeId < _usedIds.size()) && (false == _usedIds[tmpFreeId]))
      {
         found = true;
      }
      else
      {
         tmpFreeId = getNextId(tmpFreeId);
      }
   }

   if(false == found)
   {
      // programming error?
      FW_NORMAL_ASSERT_ALWAYS();
      return 0;
   }

   if(returnId < _usedIds.size())
   {
      _usedIds[returnId] = true;
   }
   else
   {
      // programming error?
      FW_NORMAL_ASSERT_ALWAYS();
      return 0;
   }

   _nextFreeId = tmpFreeId;

   return returnId;
}

BTSWblWiFiSetupId WblObjectPathManagerWbl::getNextId(IN const BTSWblWiFiSetupId id) const
{
   BTSWblWiFiSetupId returnId(id + 1);
   if(returnId >= _usedIds.size())
   {
      returnId = _lowerId;
   }
   return returnId;
}

BTSWblWiFiSetupId WblObjectPathManagerWbl::getPrevId(IN const BTSWblWiFiSetupId id) const
{
   BTSWblWiFiSetupId returnId(id);
   if(returnId > _lowerId)
   {
      --returnId;
   }
   else
   {
      returnId = (BTSWblWiFiSetupId)(_usedIds.size() - 1);
   }
   return returnId;
}

} //wbl
} //btstackif
