/**
 * @file WblLastModeWbl.cpp
 *
 * @par SW-Component
 * State machine for WBL last mode
 *
 * @brief Implementation of WBL last mode state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of WBL last mode state machine.
 */

#include "WblLastModeWbl.h"
#include "IWblLastModeCallback.h"
#include "Bts2Ipc_MessageWrapper_WBL.h"
#include "WblObjectPathManagerWbl.h"

#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WblLastModeWbl.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace wbl {

WblLastModeWbl::WblLastModeWbl() :
_callback(0),
_controlIf(0),
_objectPathManager(0)
{
}

WblLastModeWbl::~WblLastModeWbl()
{
   _callback = 0;
   _controlIf = 0;
   _objectPathManager = 0;
}

void WblLastModeWbl::setCallback(IN IWblLastModeCallback* callback)
{
   _callback = callback;

   FW_NORMAL_ASSERT(0 != _callback);
}

void WblLastModeWbl::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);
}

void WblLastModeWbl::getLastMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   createGetLastIntendedModeMsg(bts2IpcMsgList);
}

IWblLastModeRequest* WblLastModeWbl::getRequestIf(void)
{
   return this;
}

void WblLastModeWbl::setObjectPathManagerIf(IN IWblObjectPathManagerWbl* manager)
{
   _objectPathManager = manager;

   FW_NORMAL_ASSERT(0 != _objectPathManager);
}

void WblLastModeWbl::updateLastMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupStatusWblList& wifiSetupStatusList)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManager);

   BTSWblLastIntendedModeList accessPointStatusList;
   BTSWblLastIntendedMode entry;

   for(size_t i = 0; i < wifiSetupStatusList.size(); i++)
   {
      if(BTS_WBL_WIFI_MODE_AP == wifiSetupStatusList[i].mode)
      {
         BTSWblWiFiSetupId storedId(0);
         if(true == _objectPathManager->getId4ObjectPath(storedId, wifiSetupStatusList[i].objPath))
         {
            entry.id = storedId;
         }
         else
         {
            // get free id and store new mapping
            entry.id = _objectPathManager->getFreeId();
            if(0 < entry.id)
            {
               _objectPathManager->setId2ObjectPath(entry.id, wifiSetupStatusList[i].objPath);
            }
            else
            {
               FW_NORMAL_ASSERT_ALWAYS();
            }
         }

         if(0 < entry.id)
         {
            entry.status.frequency = wifiSetupStatusList[i].frequency;
            entry.status.type = wifiSetupStatusList[i].type;
            entry.status.powerState = wifiSetupStatusList[i].powerState;
            entry.status.mode = wifiSetupStatusList[i].mode;

            accessPointStatusList.push_back(entry);
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }
      }
   }

   _callback->updateLastMode(bts2IpcMsgList, bts2AppMsgList, messageItem, result, accessPointStatusList);
}

void WblLastModeWbl::createGetLastIntendedModeMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList)
{
   Bts2Ipc_GetLastIntendedMode* msg = ptrNew_Bts2Ipc_GetLastIntendedMode();
   if(0 != msg)
   {
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

} //wbl
} //btstackif
