/**
 * @file IWblObjectPathManagerWbl.h
 *
 * @par SW-Component
 * State machine for WBL object path manager
 *
 * @brief Interface definition for WBL object path manager state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the WBL interface for object path manager state machine.
 */

#ifndef _I_WBL_OBJECT_PATH_MANAGER_WBL_H_
#define _I_WBL_OBJECT_PATH_MANAGER_WBL_H_

#include "WblStackTypes.h"

namespace btstackif {
namespace wbl {

/**
 * Interface definition.
 */
class IWblObjectPathManagerWbl
{
public:
   /**
    * Destructor.
    */
   virtual ~IWblObjectPathManagerWbl() {}

   /**
    * Set WiFi setup id vs. object path mapping.
    *
    * @param[in] id: WiFi setup id
    * @param[in] objPath: object path
    */
   virtual void setId2ObjectPath(IN const BTSWblWiFiSetupId id, IN const BTSObjectPath& objPath) = 0;

   /**
    * Remove WiFi setup id vs. object path mapping.
    *
    * @param[in] objPath: object path
    */
   virtual void removeId2ObjectPath(IN const BTSObjectPath& objPath) = 0;

   /**
    * Get object path for WiFi setup id.
    *
    * @param[out] objPath: object path
    * @param[in] id: WiFi setup id
    *
    * @return = true: mapping found,
    * @return = false: mapping not found
    */
   virtual bool getObjectPath4Id(OUT BTSObjectPath& objPath, IN const BTSWblWiFiSetupId id) const = 0;

   /**
    * Get WiFi setup id for object path.
    *
    * @param[out] id: WiFi setup id
    * @param[in] objPath: object path
    *
    * @return = true: mapping found,
    * @return = false: mapping not found
    */
   virtual bool getId4ObjectPath(OUT BTSWblWiFiSetupId& id, IN const BTSObjectPath& objPath) const = 0;

   /**
    * Get free WiFi setup id.
    *
    * @return = free WiFi setup id
    */
   virtual BTSWblWiFiSetupId getFreeId(void) = 0;
};

} //wbl
} //btstackif

#endif //_I_WBL_OBJECT_PATH_MANAGER_WBL_H_
