/**
 * @file WblWifiSetupCallbackIf.cpp
 *
 * @par SW-Component
 * BtStackIf
 *
 * @brief Wbl DBUS Callback for WifiSetup.
 *
 * @copyright (C) 2016 - 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Wbl DBUS Callback implementation for WifiSetup.
 */

#include "WblWifiSetupCallbackIf.h"
#include "Ipc2Bts_MessageWrapper_WBL.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"
#include "WblUtils.h"
#include "cc_dbus_if/WblDbusParser.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_DBUS
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WblWifiSetupCallbackIf.cpp.trc.h"
#endif
#endif

using namespace ::org::bosch::wbl::wifisetup;

namespace btstackif {
namespace wbl {

WblWifiSetupCallbackIf::WblWifiSetupCallbackIf(IDbusRecHandler* recHandler) :
DbusCallbackIf(recHandler),
_callbackId(0)
{
}

WblWifiSetupCallbackIf::~WblWifiSetupCallbackIf()
{
}

// "ServiceAvailableIF" implementation --- start
void WblWifiSetupCallbackIf::onAvailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/)
{
   (void)(busName);
   (void)(busType);

   BTSDbusServiceAvailability availabilityEvent(BTS_DBUS_SERVICE_NOT_AVAILABLE);

   if(true == available)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityWbl* msg = ptrNew_Ipc2Bts_ServiceAvailabilityWbl();

   if(0 != msg)
   {
      msg->setInterface(BTS_WBL_DBUS_SERVICE_WIFI_SETUP);
      msg->setAvailabilityEvent(availabilityEvent);
      msg->setObjPath(objPath);
   }

   onSignal(msg, ::ccdbusif::DEFAULT_ACT, true);
}

void WblWifiSetupCallbackIf::onUnavailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/)
{
   (void)(busName);
   (void)(busType);

   BTSDbusServiceAvailability availabilityEvent(BTS_DBUS_SERVICE_NOT_AVAILABLE);

   if(true == available)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityWbl* msg = ptrNew_Ipc2Bts_ServiceAvailabilityWbl();

   if(0 != msg)
   {
      msg->setInterface(BTS_WBL_DBUS_SERVICE_WIFI_SETUP);
      msg->setAvailabilityEvent(availabilityEvent);
      msg->setObjPath(objPath);
   }

   onSignal(msg, ::ccdbusif::DEFAULT_ACT, true);
}
// "ServiceAvailableIF" implementation --- end

// /org/bosch/wbl/wifisetupProxy implementation --- start
void WblWifiSetupCallbackIf::onDeAuthenticateStationErrorCb(const ::boost::shared_ptr< DeAuthenticateStationError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblWifiSetupCallbackIf::onDeAuthenticateStationResponseCb(const ::boost::shared_ptr< DeAuthenticateStationResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(response);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblWifiSetupCallbackIf::onManageBandwidthErrorCb(const ::boost::shared_ptr< ManageBandwidthError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblWifiSetupCallbackIf::onManageBandwidthResponseCb(const ::boost::shared_ptr< ManageBandwidthResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(response);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblWifiSetupCallbackIf::onSetRestrictionsErrorCb(const ::boost::shared_ptr< SetRestrictionsError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblWifiSetupCallbackIf::onSetRestrictionsResponseCb(const ::boost::shared_ptr< SetRestrictionsResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(response);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblWifiSetupCallbackIf::onModeErrorCb(const ::boost::shared_ptr< ModeError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_WifiModeUpdate* msg = ptrNew_Ipc2Bts_WifiModeUpdate();
   if(0 != msg)
   {
      msg->setObjPath(objPath);
   }
   onError(msg, error);
}

void WblWifiSetupCallbackIf::onModeUpdateCb(const ::boost::shared_ptr< ModeUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_WifiModeUpdate* msg = ptrNew_Ipc2Bts_WifiModeUpdate();
   if(0 != msg)
   {
      msg->setObjPath(objPath);

      ::ccdbusif::wbl::WblDbusParser parser;
      parser.setTraces(true);
      BTSWblWifiMode mode(BTS_WBL_WIFI_MODE_STA);
      if(false == update->getMode().empty())
      {
         mode = convertWifiMode(parser.getWiFiMode2Enum(update->getMode()));
      }
      msg->setMode(mode);
   }
   onUpdate(msg, update->getAct());
}

void WblWifiSetupCallbackIf::onInterfaceErrorCb(const ::boost::shared_ptr< InterfaceError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblWifiSetupCallbackIf::onInterfaceUpdateCb(const ::boost::shared_ptr< InterfaceUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(update);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblWifiSetupCallbackIf::onMacAddressErrorCb(const ::boost::shared_ptr< MacAddressError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblWifiSetupCallbackIf::onMacAddressUpdateCb(const ::boost::shared_ptr< MacAddressUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(update);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblWifiSetupCallbackIf::onPowerStateErrorCb(const ::boost::shared_ptr< PowerStateError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_WifiPowerStateUpdate* msg = ptrNew_Ipc2Bts_WifiPowerStateUpdate();
   if(0 != msg)
   {
      msg->setObjPath(objPath);
   }
   onError(msg, error);
}

void WblWifiSetupCallbackIf::onPowerStateUpdateCb(const ::boost::shared_ptr< PowerStateUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_WifiPowerStateUpdate* msg = ptrNew_Ipc2Bts_WifiPowerStateUpdate();
   if(0 != msg)
   {
      msg->setObjPath(objPath);

      ::ccdbusif::wbl::WblDbusParser parser;
      parser.setTraces(true);
      BTSWblWifiPowerState powerState(BTS_WBL_WIFI_STATE_DEVICE_UNAVAILABLE);
      if(false == update->getPowerState().empty())
      {
         powerState = convertWifiPowerState(parser.getPowerState2Enum(update->getPowerState()));
      }
      msg->setPowerState(powerState);
   }
   onUpdate(msg, update->getAct());
}

void WblWifiSetupCallbackIf::onPowerFailureReasonErrorCb(const ::boost::shared_ptr< PowerFailureReasonError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblWifiSetupCallbackIf::onPowerFailureReasonUpdateCb(const ::boost::shared_ptr< PowerFailureReasonUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(update);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblWifiSetupCallbackIf::onFrequencyErrorCb(const ::boost::shared_ptr< FrequencyError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_WifiFrequencyUpdate* msg = ptrNew_Ipc2Bts_WifiFrequencyUpdate();
   if(0 != msg)
   {
      msg->setObjPath(objPath);
   }
   onError(msg, error);
}

void WblWifiSetupCallbackIf::onFrequencyUpdateCb(const ::boost::shared_ptr< FrequencyUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_WifiFrequencyUpdate* msg = ptrNew_Ipc2Bts_WifiFrequencyUpdate();
   if(0 != msg)
   {
      msg->setObjPath(objPath);

      ::ccdbusif::wbl::WblDbusParser parser;
      parser.setTraces(true);
      BTSWblWifiFrequency frequency(BTS_WBL_WIFI_FREQUENCY_2_4_GHZ);
      if(false == update->getFrequency().empty())
      {
         frequency = convertWifiFrequency(parser.getWiFiFrequency2Enum(update->getFrequency()));
      }
      msg->setFrequency(frequency);
   }
   onUpdate(msg, update->getAct());
}

void WblWifiSetupCallbackIf::onCurrentOperatingChannelErrorCb(const ::boost::shared_ptr< CurrentOperatingChannelError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblWifiSetupCallbackIf::onCurrentOperatingChannelUpdateCb(const ::boost::shared_ptr< CurrentOperatingChannelUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(update);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblWifiSetupCallbackIf::onAPConfigErrorCb(const ::boost::shared_ptr< APConfigError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_AccessPointConfigTypeUpdate* msg = ptrNew_Ipc2Bts_AccessPointConfigTypeUpdate();
   if(0 != msg)
   {
      msg->setObjPath(objPath);
   }
   onError(msg, error);
}

void WblWifiSetupCallbackIf::onAPConfigUpdateCb(const ::boost::shared_ptr< APConfigUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_AccessPointConfigTypeUpdate* msg = ptrNew_Ipc2Bts_AccessPointConfigTypeUpdate();
   if(0 != msg)
   {
      msg->setObjPath(objPath);

      BTSWblAccessPointConfigType type(BTS_WBL_AP_CONFIG_NORMAL);
      ::ccdbusif::wbl::WblDbusParser parser;
      parser.setTraces(true);
      BTSDbusPropertyList outPropertyList;

      if(true == parser.findAndParseProperty(outPropertyList, update->getAPConfigMutable(), parser.getLastIntendedModeProperty2String(::ccdbusif::wbl::LAST_INTENDED_MODE_TYPE), (int)::ccdbusif::wbl::IF_LAST_INTENDED_MODE))
      {
         if(1 == outPropertyList.size())
         {
            const ::ccdbusif::DbusVariantProperty& outProperty = outPropertyList[0];
            type = convertAccessPointConfigType(parser.getWiFiType2Enum(outProperty.propData.getString()));
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }
      }
      // else: parameter is only valid for AP configuration; e.g. in STA mode such parameter is not available

      msg->setType(type);
   }
   onUpdate(msg, update->getAct());
}

void WblWifiSetupCallbackIf::onRestrictionsErrorCb(const ::boost::shared_ptr< RestrictionsError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblWifiSetupCallbackIf::onRestrictionsUpdateCb(const ::boost::shared_ptr< RestrictionsUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(update);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblWifiSetupCallbackIf::onSupportedFrequenciesErrorCb(const ::boost::shared_ptr< SupportedFrequenciesError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_SupportedFrequenciesUpdate* msg = ptrNew_Ipc2Bts_SupportedFrequenciesUpdate();
   if(0 != msg)
   {
      msg->setObjPath(objPath);
   }
   onError(msg, error);
}

void WblWifiSetupCallbackIf::onSupportedFrequenciesUpdateCb(const ::boost::shared_ptr< SupportedFrequenciesUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // ETG_TRACE_USR1((" onSupportedFrequenciesUpdateCb(): objPath=%s", objPath.c_str()));

   Ipc2Bts_SupportedFrequenciesUpdate* msg = ptrNew_Ipc2Bts_SupportedFrequenciesUpdate();
   if(0 != msg)
   {
      msg->setObjPath(objPath);

      BTSWblSupportedRestrictionList& restrictionList = msg->getSupportedRestrictionsMutable();
      BTSDbusPropertyList ifProperties;
      ::std::vector< int > matchingInterfaces;
      matchingInterfaces.push_back((int)::ccdbusif::wbl::IF_LAST_INTENDED_MODE);
      ::ccdbusif::wbl::WblDbusParser parser;
      parser.setTraces(true);
      ::std::vector< ::std::map< ::std::string, ::asf::dbus::DBusVariant > >& supportedFrequencies = update->getSupportedFrequenciesMutable();
      restrictionList.reserve(supportedFrequencies.size());

      // ETG_TRACE_USR1((" onSupportedFrequenciesUpdateCb(): supportedFrequencies.size()=%u", supportedFrequencies.size()));

      for(size_t i = 0; i < supportedFrequencies.size(); i++)
      {
         parser.parseProperties(ifProperties, matchingInterfaces, supportedFrequencies[i], busType, busName, objPath, interfaceName);

         /*
          * needed:
          * restriction.frequency
          * restriction.technologies[]:
          *    type
          *    setting
          */
         BTSWblSupportedRestriction restriction;
         // restriction.frequency = BTS_WBL_WIFI_FREQUENCY_2_4_GHZ;
         BTSWblSupportedTechnologyList& technologies = restriction.technologies;
         technologies.reserve(2);
         BTSWblSupportedTechnology technology;
         // technology.type = BTS_WBL_AP_CONFIG_NORMAL;
         // technology.setting = BTS_WBL_SUPPORTED_SETTING_NOT_ALLOWED;
         bool frequencyFound(false);
         bool technologyFound(false);

         for(size_t i = 0; i < ifProperties.size(); i++)
         {
            switch((::ccdbusif::wbl::LastIntendedModeProperty)ifProperties[i].propEnum)
            {
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_LASTINTENDEDMODE:
                  FW_NORMAL_ASSERT_ALWAYS();
                  break;
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_APCONFIG:
                  FW_NORMAL_ASSERT_ALWAYS();
                  break;
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_POWERSTATE:
                  FW_NORMAL_ASSERT_ALWAYS();
                  break;
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_WIFIMODE:
                  FW_NORMAL_ASSERT_ALWAYS();
                  break;
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_FREQUENCY:
                  restriction.frequency = convertWifiFrequency(parser.getWiFiFrequency2Enum(ifProperties[i].propData.getString()));
                  frequencyFound = true;
                  break;
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_TYPE:
                  FW_NORMAL_ASSERT_ALWAYS();
                  break;
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_SPITECHNOLOGIES:
                  FW_NORMAL_ASSERT_ALWAYS();
                  break;
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_CARPLAYWIRELESS:
                  technology.type = BTS_WBL_AP_CONFIG_CPW;
                  technology.setting = convertSupportedSetting(parser.getSupportedSetting2Enum(ifProperties[i].propData.getString()));
                  technologies.push_back(technology);
                  technologyFound = true;
                  break;
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_ANDROIDAUTOWIRELESS:
                  technology.type = BTS_WBL_AP_CONFIG_AA;
                  technology.setting = convertSupportedSetting(parser.getSupportedSetting2Enum(ifProperties[i].propData.getString()));
                  technologies.push_back(technology);
                  technologyFound = true;
                  break;
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_CONNECTION_STATUS:
                  FW_NORMAL_ASSERT_ALWAYS();
                  break;
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_CONNECTED:
                  FW_NORMAL_ASSERT_ALWAYS();
                  break;
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_HEALTH_INDICATOR:
                  FW_NORMAL_ASSERT_ALWAYS();
                  break;
               default:
                  FW_NORMAL_ASSERT_ALWAYS();
                  break;
            }
         }

         if((true == frequencyFound) && (true == technologyFound))
         {
            restrictionList.push_back(restriction);
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }

         ifProperties.clear();
      }
   }
   onUpdate(msg, update->getAct());
}

#ifdef VARIANT_S_FTR_ENABLE_CONNECTION_STATUS
void WblWifiSetupCallbackIf::onConnectionStatusErrorCb(const ::boost::shared_ptr< ConnectionStatusError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_WifiConnectionStatusUpdate* msg = ptrNew_Ipc2Bts_WifiConnectionStatusUpdate();
   if(0 != msg)
   {
      msg->setObjPath(objPath);
   }
   onError(msg, error);
}

void WblWifiSetupCallbackIf::onConnectionStatusUpdateCb(const ::boost::shared_ptr< ConnectionStatusUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   Ipc2Bts_WifiConnectionStatusUpdate* msg = ptrNew_Ipc2Bts_WifiConnectionStatusUpdate();
   if(0 != msg)
   {
      msg->setObjPath(objPath);

      BTSDbusPropertyList ifProperties;
      ::std::vector< int > matchingInterfaces;
      matchingInterfaces.push_back((int)::ccdbusif::wbl::IF_LAST_INTENDED_MODE);
      ::ccdbusif::wbl::WblDbusParser parser;
      parser.setTraces(true);

      parser.parseProperties(ifProperties, matchingInterfaces, update->getConnectionStatusMutable(), busType, busName, objPath, interfaceName);

      bool connected(false);
      BTSWblHealthinessIndicator healthIndicator(0);
      bool connectedFound(false);
      bool healthIndicatorFound(false);

      for(size_t i = 0; i < ifProperties.size(); i++)
      {
         switch((::ccdbusif::wbl::LastIntendedModeProperty)ifProperties[i].propEnum)
         {
            case ::ccdbusif::wbl::LAST_INTENDED_MODE_LASTINTENDEDMODE:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
            case ::ccdbusif::wbl::LAST_INTENDED_MODE_APCONFIG:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
            case ::ccdbusif::wbl::LAST_INTENDED_MODE_POWERSTATE:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
            case ::ccdbusif::wbl::LAST_INTENDED_MODE_WIFIMODE:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
            case ::ccdbusif::wbl::LAST_INTENDED_MODE_FREQUENCY:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
            case ::ccdbusif::wbl::LAST_INTENDED_MODE_TYPE:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
            case ::ccdbusif::wbl::LAST_INTENDED_MODE_SPITECHNOLOGIES:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
            case ::ccdbusif::wbl::LAST_INTENDED_MODE_CARPLAYWIRELESS:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
            case ::ccdbusif::wbl::LAST_INTENDED_MODE_ANDROIDAUTOWIRELESS:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
            case ::ccdbusif::wbl::LAST_INTENDED_MODE_CONNECTION_STATUS:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
            case ::ccdbusif::wbl::LAST_INTENDED_MODE_CONNECTED:
               connected = ifProperties[i].propData.getBool();
               connectedFound = true;
               break;
            case ::ccdbusif::wbl::LAST_INTENDED_MODE_HEALTH_INDICATOR:
               healthIndicator = ifProperties[i].propData.getUInt32();
               healthIndicatorFound = true;
               break;
            default:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
         }
      }

      if((true == connectedFound) && (true == healthIndicatorFound))
      {
         msg->setConnected(connected);
         msg->setHealthIndicator(healthIndicator);
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
   onUpdate(msg, update->getAct());
}
#endif
// /org/bosch/wbl/wifisetupProxy implementation --- end

} //wbl
} //btstackif
