/**
 * @file WblLastIntendedModeCallbackIf.cpp
 *
 * @par SW-Component
 * BtStackIf
 *
 * @brief Wbl DBUS Callback for LastIntendedMode.
 *
 * @copyright (C) 2016 - 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Wbl DBUS Callback implementation for LastIntendedMode.
 */

#include "WblLastIntendedModeCallbackIf.h"
#include "Ipc2Bts_MessageWrapper_WBL.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"
#include "WblUtils.h"
#include "cc_dbus_if/WblDbusParser.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_DBUS
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WblLastIntendedModeCallbackIf.cpp.trc.h"
#endif
#endif

using namespace ::org::bosch::wbl::LastIntendedMode;

namespace btstackif {
namespace wbl {

WblLastIntendedModeCallbackIf::WblLastIntendedModeCallbackIf(IDbusRecHandler* recHandler) :
DbusCallbackIf(recHandler),
_callbackId(0)
{
}

WblLastIntendedModeCallbackIf::~WblLastIntendedModeCallbackIf()
{
}

// "ServiceAvailableIF" implementation --- start
void WblLastIntendedModeCallbackIf::onAvailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);

   BTSDbusServiceAvailability availabilityEvent(BTS_DBUS_SERVICE_NOT_AVAILABLE);

   if(true == available)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityWbl* msg = ptrNew_Ipc2Bts_ServiceAvailabilityWbl();

   if(0 != msg)
   {
      msg->setInterface(BTS_WBL_DBUS_SERVICE_LAST_INTENDED_MODE);
      msg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(msg, ::ccdbusif::DEFAULT_ACT, true);
}

void WblLastIntendedModeCallbackIf::onUnavailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);

   BTSDbusServiceAvailability availabilityEvent(BTS_DBUS_SERVICE_NOT_AVAILABLE);

   if(true == available)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityWbl* msg = ptrNew_Ipc2Bts_ServiceAvailabilityWbl();

   if(0 != msg)
   {
      msg->setInterface(BTS_WBL_DBUS_SERVICE_LAST_INTENDED_MODE);
      msg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(msg, ::ccdbusif::DEFAULT_ACT, true);
}
// "ServiceAvailableIF" implementation --- end

// /org/bosch/wbl/LastIntendedModeProxy implementation --- start
void WblLastIntendedModeCallbackIf::onLastIntendedModeErrorCb(const ::boost::shared_ptr< LastIntendedModeError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   onError(ptrNew_Ipc2Bts_LastIntendedModeUpdate(), error);
}

void WblLastIntendedModeCallbackIf::onLastIntendedModeUpdateCb(const ::boost::shared_ptr< LastIntendedModeUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   Ipc2Bts_LastIntendedModeUpdate* msg = ptrNew_Ipc2Bts_LastIntendedModeUpdate();

   if(0 != msg)
   {
      ::ccdbusif::wbl::WblDbusParser parser;
      parser.setTraces(true);
      BTSWblWiFiSetupStatusWblList& statusList = msg->getWifiSetupStatusListMutable();
      BTSWblWiFiSetupStatusWbl entry;
      BTSDbusPropertyList ifProperties;
      ::std::vector< int > matchingInterfaces;
      matchingInterfaces.push_back((int)::ccdbusif::wbl::IF_LAST_INTENDED_MODE);
      ::std::vector< LastIntendedModeStruct >& modes = update->getLastIntendedModeMutable();
      statusList.reserve(modes.size());

      // ETG_TRACE_USR3((" onLastIntendedModeUpdateCb(): modes.size()=%u", modes.size()));

      for(size_t i = 0; i < modes.size(); i++)
      {
         entry.objPath = modes[i].getElem1();
         entry.powerState = BTS_WBL_WIFI_STATE_DEVICE_UNAVAILABLE;
         entry.mode = BTS_WBL_WIFI_MODE_STA;
         entry.frequency = BTS_WBL_WIFI_FREQUENCY_2_4_GHZ;
         entry.type = BTS_WBL_AP_CONFIG_NORMAL;

         // const ::std::string& getElem1(): object path
         // ::std::map< ::std::string, ::asf::dbus::DBusVariant >& getElem2Mutable(): dictionary

         // ETG_TRACE_USR3((" onLastIntendedModeUpdateCb(): [%u]: objPath=%s", i, modes[i].getElem1().c_str()));
         parser.parseProperties(ifProperties, matchingInterfaces, modes[i].getElem2Mutable(), busType, busName, objPath, interfaceName);

         for(size_t i = 0; i < ifProperties.size(); i++)
         {
            switch((::ccdbusif::wbl::LastIntendedModeProperty)ifProperties[i].propEnum)
            {
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_LASTINTENDEDMODE:
                  FW_NORMAL_ASSERT_ALWAYS();
                  break;
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_APCONFIG:
                  FW_NORMAL_ASSERT_ALWAYS();
                  break;
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_POWERSTATE:
                  if(false == ifProperties[i].propData.getString().empty())
                  {
                     entry.powerState = convertWifiPowerState(parser.getPowerState2Enum(ifProperties[i].propData.getString()));
                  }
                  break;
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_WIFIMODE:
                  if(false == ifProperties[i].propData.getString().empty())
                  {
                     entry.mode = convertWifiMode(parser.getWiFiMode2Enum(ifProperties[i].propData.getString()));
                  }
                  break;
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_FREQUENCY:
                  if(false == ifProperties[i].propData.getString().empty())
                  {
                     entry.frequency = convertWifiFrequency(parser.getWiFiFrequency2Enum(ifProperties[i].propData.getString()));
                  }
                  break;
               case ::ccdbusif::wbl::LAST_INTENDED_MODE_TYPE:
                  entry.type = convertAccessPointConfigType(parser.getWiFiType2Enum(ifProperties[i].propData.getString()));
                  break;
               default:
                  FW_NORMAL_ASSERT_ALWAYS();
                  break;
            }
         }

         statusList.push_back(entry);
         ifProperties.clear();
      }

      onUpdate(msg, update->getAct());
   }
}
// /org/bosch/wbl/LastIntendedModeProxy implementation --- end

} //wbl
} //btstackif
