/**
 * @file WblDbusAccessMain.cpp
 *
 * @par SW-Component
 * IPC
 *
 * @brief DBUS WBL handling.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of DBUS WBL handling.
 */

#include "cc_dbus_if/ICcDbusIfController.h"
#include "cc_dbus_if/ICcDbusIfControllerClient.h"

#include "WblDbusAccessMain.h"
#include "Bts2Ipc_MessageWrapper_WBL.h"
#include "Ipc2Bts_MessageWrapper_WBL.h"
#include "DbusWorkItem.h"
#include "WblDbusIfHandler.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_DBUS
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WblDbusAccessMain.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace wbl {

WblDbusAccessMain::WblDbusAccessMain() :
_count(2U)
{
   setResponseTimeoutHandler(this);
   createTimeoutControlList(_count);
}

WblDbusAccessMain::~WblDbusAccessMain()
{
   destroyTimeoutControlList();
   destroyDbusIfHandlerList();
}

void WblDbusAccessMain::sendBts2IpcMessage(IN Bts2Ipc_BaseMessage* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if(true != _dbusIfAvailable)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }

   if(0 == _controllerClient)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }

   const BTSFunctionBlock functionBlock((BTSFunctionBlock)ptrMessage->getFunctionBlock());

   if((BTS_FB_NONE < functionBlock) && (functionBlock < BTS_FB_LAST))
   {
      // push to waiting queue
      _bts2IpcWaitingQueue.push(ptrMessage);
      // trigger CcDbusIf library thread
      _controllerClient->pushWorkItem(new DbusWorkItemTransmit(this));
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }
}

void WblDbusAccessMain::stop(void)
{
   _terminateWorkerThread = true;
}

void WblDbusAccessMain::run(void)
{
   _terminateWorkerThread = false;
}

BTSErrorCode WblDbusAccessMain::setCcDbusIfControllerIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         IN ::ccdbusif::ICcDbusIfController* controller, IN const ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   ETG_TRACE_USR1((" setCcDbusIfControllerIf(): [enter]"));

   if(0 == controller)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return BTS_INVALID_PARAM;
   }

   if(BTS_USER_MODE_LAST <= userMode)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return BTS_INVALID_PARAM;
   }

   // check if interfaces are already set (checking _controllerClient is enough)
   if(0 != _controllerClient)
   {
      // already set
      return BTS_OK;
   }

   FW_NORMAL_ASSERT(component == _component); // already set
   FW_NORMAL_ASSERT(stackInterface == _stackInterface); // already set
   FW_NORMAL_ASSERT(subComponent == BTS_FB_WBL); // already set
   FW_NORMAL_ASSERT(true == configuration.wblActive); // configuration for WBL part is needed

   // create needed dbus if handler
   createDbusIfHandler(_component, _stackInterface, BTS_FB_WBL, userMode, configuration.wblConfiguration);

   BTSErrorCode errCode(BTS_OK);

   _controllerClient = controller->getCcDbusIfControllerClient();

   if(0 == _controllerClient)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      errCode = BTS_INVALID_PARAM;
   }
   else
   {
      // check dbus if handler list
      if(BTS_OK != executeSetCcDbusIfControllerIf(_component, _stackInterface, BTS_FB_WBL, userMode, controller, dbusInterfaces, configuration))
      {
         FW_NORMAL_ASSERT_ALWAYS();
         ETG_TRACE_FATAL((" setCcDbusIfControllerIf(): at least one dbus if handler failed"));
         errCode = BTS_ERROR;
      }
   }

   if(BTS_OK != errCode)
   {
      // reset all
      _controllerClient = 0;

      // do not delete callback handler
   }
   else
   {
      // set marker that D-Bus interface is available
      _dbusIfAvailable = true;
   }

   ETG_TRACE_USR1((" setCcDbusIfControllerIf(): [exit] _dbusIfAvailable=%d", _dbusIfAvailable));

   return errCode;
}

void WblDbusAccessMain::resetCcDbusIfControllerIf(void)
{
   // check if interfaces are already reset (checking _controllerClient is enough)
   if(0 == _controllerClient)
   {
      // already reset
      return;
   }

   // reset all
   _controllerClient = 0;

   // check dbus if handler list
   executeResetCcDbusIfControllerIf();

   // reset marker that D-Bus interface is available
   _dbusIfAvailable = false;

   // do not delete callback handler

   ETG_TRACE_USR1((" resetCcDbusIfControllerIf(): _dbusIfAvailable=%d", _dbusIfAvailable));
}

void WblDbusAccessMain::handleTimerTick(void)
{
   if(false == _timerHandlingEnabled)
   {
      return;
   }

   if(0 == _controllerClient)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // trigger CcDbusIf library thread
   _controllerClient->pushWorkItem(new DbusWorkItemTimeout(this));
}

void WblDbusAccessMain::setSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   FW_IF_NULL_PTR_RETURN(testCommand);

   // check dbus if handler list
   if(false == executeSimulationTestCommand(testCommand, testData))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      ETG_TRACE_FATAL((" setSimulationTestCommand(): testCommand %s not processed", testCommand));
   }
}

void WblDbusAccessMain::setSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   FW_IF_NULL_PTR_RETURN(testCommand);
   FW_IF_NULL_PTR_RETURN(testData);

   // check dbus if handler list
   if(false == executeSimulationTestCommand(testCommand, testData))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      ETG_TRACE_FATAL((" setSimulationTestCommand(): testCommand %s not processed", testCommand));
   }
}

void WblDbusAccessMain::setSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData)
{
   FW_IF_NULL_PTR_RETURN(testCommand);

   // check dbus if handler list
   if(false == executeSimulationTestCommand(testCommand, testData))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      ETG_TRACE_FATAL((" setSimulationTestCommand(): testCommand %s not processed", testCommand));
   }
}

void WblDbusAccessMain::createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent)
{
   BTSWblDbusServiceInterface serviceInterface = (BTSWblDbusServiceInterface)interface;

   if(BTS_WBL_DBUS_SERVICE_LAST <= serviceInterface)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(BTS_DBUS_SERVICE_LAST <= availabilityEvent)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   Ipc2Bts_ServiceAvailabilityWbl* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityWbl();

   if(0 != ptrMsg)
   {
      ptrMsg->setInterface(serviceInterface);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
      ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
   }

   pushIpc2BtsMessage(ptrMsg, true);
}

void WblDbusAccessMain::createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent, IN const BTSBusName& busName, IN const BTSObjectPath& objPath, IN const BTSCommonEnumClass busType)
{
   BTSWblDbusServiceInterface serviceInterface = (BTSWblDbusServiceInterface)interface;

   if(BTS_WBL_DBUS_SERVICE_LAST <= serviceInterface)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(BTS_DBUS_SERVICE_LAST <= availabilityEvent)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(((BTSCommonEnumClass)::ccdbusif::BUS_TYPE_SYSTEM != busType) && ((BTSCommonEnumClass)::ccdbusif::BUS_TYPE_SESSION != busType))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   Ipc2Bts_ServiceAvailabilityWbl* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityWbl();

   if(0 != ptrMsg)
   {
      ptrMsg->setInterface(serviceInterface);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
      ptrMsg->setBusType(busType);
      ptrMsg->setBusName(busName);
      ptrMsg->setObjPath(objPath);
      ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
   }

   pushIpc2BtsMessage(ptrMsg, true);
}

void WblDbusAccessMain::threadFunction(void* arguments)
{
   (void)arguments;
}

void WblDbusAccessMain::setTerminate(void* arguments)
{
   (void)arguments;
}

void WblDbusAccessMain::handleDbusTrmQueue(void)
{
   // HINT: This function is called within context of ASF component thread.

   int counter = 0;
   Bts2Ipc_BaseMessage* ptrMessage;
   BTSFunctionBlock functionBlock;

   while(counter < MAX_NUMBER_BTS2IPC_MGS_PROCESSING)
   {
      ptrMessage = _bts2IpcWaitingQueue.pop();
      if(0 != ptrMessage)
      {
         // do trace
         ptrMessage->doOutputTrace();

         functionBlock = (BTSFunctionBlock)ptrMessage->getFunctionBlock();

         if((BTS_FB_NONE < functionBlock) && (functionBlock < BTS_FB_LAST))
         {
            bool deleteBts2IpcMsg(false);
            bool sendErrorIpc2BtsMsg(false);

            // check dbus if handler list
            if(false == processMessage(deleteBts2IpcMsg, sendErrorIpc2BtsMsg, ptrMessage))
            {
               FW_NORMAL_ASSERT_ALWAYS();
               ETG_TRACE_FATAL((" handleDbusTrmQueue(): Bts2Ipc 0x%04X not processed", ptrMessage->getTraceOpCode()));
               delete ptrMessage;
            }
            else
            {
               // check for delete and sending error / success
               if(true == sendErrorIpc2BtsMsg)
               {
                  Ipc2Bts_BaseMessage* ipc2BtsMessage(0);

                  // check dbus if handler list
                  if(false == createErrorMessage(&ipc2BtsMessage, ptrMessage, BTS_IPC_SENDING_FAILED))
                  {
                     FW_NORMAL_ASSERT_ALWAYS();
                     ETG_TRACE_FATAL((" handleDbusTrmQueue(): Bts2Ipc 0x%04X not processed", ptrMessage->getTraceOpCode()));
                  }
                  else
                  {
                     if(0 == ipc2BtsMessage)
                     {
                        FW_NORMAL_ASSERT_ALWAYS();
                        ETG_TRACE_FATAL((" handleDbusTrmQueue(): Bts2Ipc 0x%04X not processed", ptrMessage->getTraceOpCode()));
                     }
                     else
                     {
                        // check dbus if handler list
                        if(false == transferMessageData(ipc2BtsMessage, ptrMessage))
                        {
                           FW_NORMAL_ASSERT_ALWAYS();
                           ETG_TRACE_FATAL((" handleDbusTrmQueue(): Bts2Ipc 0x%04X not processed", ptrMessage->getTraceOpCode()));
                        }

                        // copy meta data
                        ipc2BtsMessage->setBtsDestinationFunctionBlock(ptrMessage->getBtsSourceFunctionBlock());
                        ipc2BtsMessage->setApp2BtsCompareItem(ptrMessage->getApp2BtsCompareItem());

                        // set response flag
                        ipc2BtsMessage->setResponseMessageFlag(true);

                        pushIpc2BtsMessage(ipc2BtsMessage);
                     }
                  }

                  // in error case we will always delete the message
                  deleteBts2IpcMsg = true;
               }

               if(true == deleteBts2IpcMsg)
               {
                  // can be error case e.g. sending failed or sending of response message (no answer on DBUS level)
                  delete ptrMessage;
               }
               else
               {
                  // handle sending success
                  handleSendingSuccess(ptrMessage);
               }
            }
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
            delete ptrMessage;
         }

         counter++;
      }
      else
      {
         counter = MAX_NUMBER_BTS2IPC_MGS_PROCESSING;
      }
   }
}

void WblDbusAccessMain::handleReceivedDbusMessage(IN Ipc2Bts_BaseMessage* ptrMessage, IN const bool highPrio /*= false*/)
{
   // HINT: This function is called within context of ASF component thread.

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // no check for _dbusIfAvailable necessary because this is receiving direction

   if(0 == _controllerClient)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }

   // check for high priority message; only signals are allowed to be high priority messages; no explicit check is implemented; programmer has to ensure correct implementation/usage
   if(true == highPrio)
   {
      // for signals no matching Bts2Ipc message is available in working queue therefore no check is necessary
      pushIpc2BtsMessage(ptrMessage, highPrio);
   }
   else
   {
      // push to callback queue and process directly
      _ipc2BtsCallbackQueue.push(ptrMessage, false);
      handleDbusCallbackMessage();
   }
}

void WblDbusAccessMain::handleDbusCallbackMessage(void)
{
   int counter = 0;
   Ipc2Bts_BaseMessage* ptrIpc2BtsMessage;
   Bts2Ipc_BaseMessage* ptrBts2IpcQueueMessage;

   while(counter < MAX_NUMBER_IPC2BTS_MGS_PROCESSING)
   {
      ptrIpc2BtsMessage = _ipc2BtsCallbackQueue.pop(false);
      if(0 != ptrIpc2BtsMessage)
      {
         // find matching request message
         ptrBts2IpcQueueMessage = findAndRemoveBts2IpcWorkingMessage(ptrIpc2BtsMessage, false);

         if((0 < ptrIpc2BtsMessage->getErrorName().length()) || (0 < ptrIpc2BtsMessage->getErrorMessage().length()))
         {
            ETG_TRACE_USR1((" handleDbusCallbackMessage(): error name   =%s", ptrIpc2BtsMessage->getErrorName().c_str()));
            ETG_TRACE_USR1((" handleDbusCallbackMessage(): error message=%s", ptrIpc2BtsMessage->getErrorMessage().c_str()));
         }

         if(0 != ptrBts2IpcQueueMessage)
         {
            // remove timer entry
            releaseTimeoutControlEntry(ptrBts2IpcQueueMessage);

            ETG_TRACE_USR3((" handleDbusCallbackMessage(): Bts2Ipc 0x%08X found in working queue", ptrBts2IpcQueueMessage->getMessageTraceId()));

            // check dbus if handler list
            if(false == transferMessageData(ptrIpc2BtsMessage, ptrBts2IpcQueueMessage))
            {
               FW_NORMAL_ASSERT_ALWAYS();
               ETG_TRACE_FATAL((" handleDbusCallbackMessage(): Bts2Ipc 0x%04X not processed", ptrBts2IpcQueueMessage->getTraceOpCode()));
            }

            // copy meta data
            ptrIpc2BtsMessage->setBtsDestinationFunctionBlock(ptrBts2IpcQueueMessage->getBtsSourceFunctionBlock());
            ptrIpc2BtsMessage->setApp2BtsCompareItem(ptrBts2IpcQueueMessage->getApp2BtsCompareItem());

            // set response flag
            ptrIpc2BtsMessage->setResponseMessageFlag(true);

            delete ptrBts2IpcQueueMessage;
         }
         else
         {
            ETG_TRACE_USR3((" handleDbusCallbackMessage(): Bts2Ipc message not available => signal/indication message received"));
         }

         //check and remove Bts2Ipc working messages if object path was removed
         removeBts2IpcWorkingMessagesOnObjectPathRemovedSignal(ptrIpc2BtsMessage);

         pushIpc2BtsMessage(ptrIpc2BtsMessage);

         counter++;
      }
      else
      {
         counter = MAX_NUMBER_IPC2BTS_MGS_PROCESSING;
      }
   }
}

void WblDbusAccessMain::handleDbusTimeoutEvent(void)
{
   _timerMaster.requestCurrentTime();
   _timerMaster.checkElapsedTimer();
}

void WblDbusAccessMain::handleDbusResponseTimeout(Bts2Ipc_BaseMessage* ptrMessage)
{
   // HINT: This function is called within context of ASF component thread.

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   Bts2Ipc_BaseMessage* ptrBts2IpcQueueMessage(ptrMessage);

   // remove request message
   removeBts2IpcWorkingMessage(ptrBts2IpcQueueMessage, false);

   // remove timer entry
   releaseTimeoutControlEntry(ptrBts2IpcQueueMessage);

   ETG_TRACE_ERR((" handleDbusResponseTimeout(): Bts2Ipc 0x%08X found in working queue", ptrBts2IpcQueueMessage->getMessageTraceId()));

   Ipc2Bts_BaseMessage* ipc2BtsMessage(0);

   // check dbus if handler list
   if(false == createErrorMessage(&ipc2BtsMessage, ptrBts2IpcQueueMessage, BTS_IPC_METHOD_RETURN_TIMEOUT))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      ETG_TRACE_FATAL((" handleDbusResponseTimeout(): Bts2Ipc 0x%04X not processed", ptrBts2IpcQueueMessage->getTraceOpCode()));
   }
   else
   {
      if(0 == ipc2BtsMessage)
      {
         FW_NORMAL_ASSERT_ALWAYS();
         ETG_TRACE_FATAL((" handleDbusResponseTimeout(): Bts2Ipc 0x%04X not processed", ptrBts2IpcQueueMessage->getTraceOpCode()));
      }
      else
      {
         // check dbus if handler list
         if(false == transferMessageData(ipc2BtsMessage, ptrBts2IpcQueueMessage))
         {
            FW_NORMAL_ASSERT_ALWAYS();
            ETG_TRACE_FATAL((" handleDbusResponseTimeout(): Bts2Ipc 0x%04X not processed", ptrBts2IpcQueueMessage->getTraceOpCode()));
         }

         // copy meta data
         ipc2BtsMessage->setBtsDestinationFunctionBlock(ptrBts2IpcQueueMessage->getBtsSourceFunctionBlock());
         ipc2BtsMessage->setApp2BtsCompareItem(ptrBts2IpcQueueMessage->getApp2BtsCompareItem());

         // set response flag
         ipc2BtsMessage->setResponseMessageFlag(true);

         pushIpc2BtsMessage(ipc2BtsMessage);
      }
   }

   delete ptrBts2IpcQueueMessage;
}

void WblDbusAccessMain::removeBts2IpcWorkingMessagesOnObjectPathRemovedSignal(IN Ipc2Bts_BaseMessage* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if (0 < ptrMessage->getObjectId().size())
   {
      Bts2Ipc_BaseMessage* ptrBts2IpcQueueMessage = NULL;
      ::std::vector<Bts2Ipc_BaseMessage*> bts2IpcMsgList;

      // take messages from the queue for processing and remove them from the queue
      findAndRemoveBts2IpcWorkingMessages(bts2IpcMsgList, ptrMessage, false);

      for(size_t i = 0; i < bts2IpcMsgList.size(); i++)
      {
         ptrBts2IpcQueueMessage = bts2IpcMsgList[i];

         if (NULL != ptrBts2IpcQueueMessage)
         {
            // remove timer entry
            releaseTimeoutControlEntry(ptrBts2IpcQueueMessage);

            ETG_TRACE_ERR((" removeBts2IpcWorkingMessagesOnObjectPathRemovedSignal(): Bts2Ipc 0x%08X found in working queue", ptrBts2IpcQueueMessage->getMessageTraceId()));

            Ipc2Bts_BaseMessage* ipc2BtsMessage(0);

            // check dbus if handler list
            if (false == createErrorMessage(&ipc2BtsMessage, ptrBts2IpcQueueMessage, BTS_IPC_DBUS_ERROR_NO_SERVER))
            {
               FW_NORMAL_ASSERT_ALWAYS();
               ETG_TRACE_FATAL((" removeBts2IpcWorkingMessagesOnObjectPathRemovedSignal(): Bts2Ipc 0x%04X not processed", ptrBts2IpcQueueMessage->getTraceOpCode()));
            }
            else
            {
               if (0 == ipc2BtsMessage)
               {
                  FW_NORMAL_ASSERT_ALWAYS();
                  ETG_TRACE_FATAL((" removeBts2IpcWorkingMessagesOnObjectPathRemovedSignal(): Bts2Ipc 0x%04X not processed", ptrBts2IpcQueueMessage->getTraceOpCode()));
               }
               else
               {
                  // check dbus if handler list
                  if(false == transferMessageData(ipc2BtsMessage, ptrBts2IpcQueueMessage))
                  {
                     FW_NORMAL_ASSERT_ALWAYS();
                     ETG_TRACE_FATAL((" removeBts2IpcWorkingMessagesOnObjectPathRemovedSignal(): Bts2Ipc 0x%04X not processed", ptrBts2IpcQueueMessage->getTraceOpCode()));
                  }

                  // copy meta data
                  ipc2BtsMessage->setBtsDestinationFunctionBlock(ptrBts2IpcQueueMessage->getBtsSourceFunctionBlock());
                  ipc2BtsMessage->setApp2BtsCompareItem(ptrBts2IpcQueueMessage->getApp2BtsCompareItem());

                  // set response flag
                  ipc2BtsMessage->setResponseMessageFlag(true);

                  pushIpc2BtsMessage(ipc2BtsMessage);
               }
            }

            delete ptrBts2IpcQueueMessage;
         }
      }
   }
}

void WblDbusAccessMain::createDbusIfHandler(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode, IN const BTSLocalWblConfiguration& configuration)
{
   (void)(userMode);
   (void)(configuration);

   IDbusIfHandler* handler(new WblDbusIfHandler(component, stackInterface, subComponent));
   if(0 != handler)
   {
      handler->setDbusBaseIf(this);
      handler->setDbusRecHandlerIf(this);
      addDbusIfHandler(handler);
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

} //wbl
} //btstackif
