/**
 * @file WblControlMain.cpp
 *
 * @par SW-Component
 * Main
 *
 * @brief WBL Control Main.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of WBL control main functionality.
 */

#include "WblControlMain.h"
#include "App2Bts_MessageWrapper.h"
#include "Ipc2Bts_MessageWrapper_WBL.h"
#include "ICheckWaitingQueue.h"
#include "IExtendedTimeoutHandler.h"
#include "BtsTimerPool.h"
#include "WblControlHandler.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WblControlMain.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace wbl {

WblControlMain::WblControlMain()
: BasicControl(BTS_FB_WBL)
{
}

WblControlMain::~WblControlMain()
{
}

void WblControlMain::pushApp2BtsMessage(IN App2Bts_BaseMessage* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if(_functionBlock != ptrMessage->getFunctionBlock())
   {
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }

   // HINT: if any error happens now the error has to be handled within this function or within any called sub-function

   // do a general pre-check; e.g. check for DBUS service availability or ongoing startup
   bool continueProcessing(false);
   bool rejectRequest(false);
   BTSCommonEnumClass resultCode(BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);
   BTSCommonEnumClass statusCode(BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);
   bool skipOpCodeCheck(false);

   // check control handler list
   if(false == executeApp2BtsMsgPrecheck(continueProcessing, rejectRequest, resultCode, statusCode, skipOpCodeCheck, ptrMessage))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      ETG_TRACE_FATAL((" pushApp2BtsMessage(): App2Bts 0x%04X not processed", ptrMessage->getTraceOpCode()));
   }
   else
   {
      if(false == continueProcessing)
      {
         // message was processed, no further action necessary
         return;
      }
   }

   // check for rejecting request
   if(true == rejectRequest)
   {
      ::std::vector< App2Bts_BaseMessage* > msgList;
      msgList.push_back(ptrMessage);
      sendDirectAnswerForApp2BtsMessages(msgList, resultCode, statusCode);
      delete ptrMessage;
      return;
   }

   bool similarOpCodeInWorkingQueue = false;

   // check for skipping similar opcode check
   if(false == skipOpCodeCheck)
   {
      // get similar opcodes
      ::std::vector< BTSApp2BtsMessageMasterCompareItem > itemList;
      getSimilarOpCodes(itemList, ptrMessage);

      // check if opcode is in working queue
      similarOpCodeInWorkingQueue = isSimilarOpCodeInWorkingQueue(itemList);
   }

   // check if opcode is in working queue
   if(true == similarOpCodeInWorkingQueue)
   {
      // push to waiting queue
      pushApp2BtsMsgToWaitingQueue(ptrMessage, false); // single worker thread
      ETG_TRACE_USR3((" pushApp2BtsMessage(): App2Bts 0x%04X pushed to waiting queue", ptrMessage->getTraceOpCode()));
   }
   else
   {
      ::std::vector< Bts2Ipc_BaseMessage* > sendBts2IpcMsgList;
      sendBts2IpcMsgList.reserve(10);
      ::std::vector< Bts2App_BaseMessage* > sendBts2AppMsgList;
      sendBts2AppMsgList.reserve(10);
      bool deleteApp2BtsMsg = false;
      BTSApp2BtsMessageCompareItem compareItem;
      ptrMessage->getCompareItem(compareItem);

      ETG_TRACE_USR3((" pushApp2BtsMessage(): App2Bts 0x%04X to be processed", ptrMessage->getTraceOpCode()));

      // handle message depending on opcode
      // - set marker to delete message in sub-handler function if necessary
      // - push message to working queue in sub-handler function if necessary
      // - create direct answer message (Bts2App) in sub-handler function if necessary
      // - create Bts2Ipc message in sub-handler function if necessary
      // - handle any error in sub-handler function because there is the best place to handle

      // check control handler list
      if(false == processApp2BtsMessage(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, ptrMessage))
      {
         FW_NORMAL_ASSERT_ALWAYS();
         ETG_TRACE_FATAL((" pushApp2BtsMessage(): App2Bts 0x%04X not processed", ptrMessage->getTraceOpCode()));
         deleteApp2BtsMsg = true;
      }

      if(0 < sendBts2IpcMsgList.size())
      {
         _bts2IpcMessageWasSent = true; // this works only if single thread handling
      }

      sendBts2IpcMessageList(sendBts2IpcMsgList, compareItem);

      sendBts2AppMessageList(sendBts2AppMsgList);

      if(true == deleteApp2BtsMsg)
      {
         delete ptrMessage;
      }
   }
}

void WblControlMain::pushIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if(_functionBlock != ptrMessage->getBtsDestinationFunctionBlock())
   {
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }

   // HINT: if any error happens now the error has to be handled within this function or within any called sub-function

   ::std::vector< Bts2Ipc_BaseMessage* > sendBts2IpcMsgList;
   sendBts2IpcMsgList.reserve(10);
   ::std::vector< Bts2App_BaseMessage* > sendBts2AppMsgList;
   sendBts2AppMsgList.reserve(10);
   BTSHandleIpc2BtsMessageItem ipc2BtsMessageItem;

   // find related APP2BTS message
   // for req -> cfm sequence there must be a message in working queue
   // for ind sequence there can be a message in working queue
   ipc2BtsMessageItem.message = findApp2BtsWorkingMessageWrapper(ptrMessage->getApp2BtsCompareItem());

   // special handling is needed for e.g. connect/disconnect
   ipc2BtsMessageItem.item.opCode = App2BtsOC_None;

   // handle message depending on opcode
   // - do not delete Ipc2Bts message in sub-handler function because this done at the end of this function
   // - set marker to delete related App2Bts message in sub-handler function if necessary
   // - create Bts2App message in sub-handler function if necessary
   // - create Bts2Ipc message in sub-handler function if necessary
   // - handle any error in sub-handler function because there is the best place to handle

   // check control handler list
   if(false == processIpc2BtsMessage(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, ptrMessage))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      ETG_TRACE_FATAL((" pushIpc2BtsMessage(): Ipc2Bts 0x%04X not processed", ptrMessage->getTraceOpCode()));
   }

   if(0 < sendBts2IpcMsgList.size())
   {
      if(0 != ipc2BtsMessageItem.message)
      {
         BTSApp2BtsMessageCompareItem compareItem;
         ipc2BtsMessageItem.message->getCompareItem(compareItem);
         sendBts2IpcMessageList(sendBts2IpcMsgList, compareItem);
      }
      else
      {
         // response message: it could be possible that there is no related application message
         // NOTE: it can also happen due to wrong implementation
         BTSApp2BtsMessageCompareItem compareItem;
         sendBts2IpcMessageList(sendBts2IpcMsgList, compareItem);
      }
   }

   sendBts2AppMessageList(sendBts2AppMsgList);

   // delete received message
   delete ptrMessage;

   // remove related APP2BTS message and check waiting queue
   if((true == ipc2BtsMessageItem.deleteMessage) && (0 != ipc2BtsMessageItem.message))
   {
      removeApp2BtsWorkingMessage(ipc2BtsMessageItem.message);

      if(App2BtsOC_None == ipc2BtsMessageItem.item.opCode)
      {
         checkWaitingQueue(ipc2BtsMessageItem.message);
      }

      delete ipc2BtsMessageItem.message;
   }

   // check waiting queue (special handling for connect/disconnect)
   if((App2BtsOC_None != ipc2BtsMessageItem.item.opCode) && (App2BtsOC_Ignore != ipc2BtsMessageItem.item.opCode))
   {
      if(0 != ipc2BtsMessageItem.checkIf)
      {
         ipc2BtsMessageItem.checkIf->checkWaitingQueue(ipc2BtsMessageItem.item);
      }
      else
      {
         checkWaitingQueueExtended(ipc2BtsMessageItem.item);
      }
   }
}

void WblControlMain::setStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         OUT ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   ETG_TRACE_USR1((" setStackConfiguration(): [enter]"));

   FW_NORMAL_ASSERT(component == _component); // already set
   FW_NORMAL_ASSERT(stackInterface == _stackInterface); // already set
   FW_NORMAL_ASSERT(subComponent == BTS_FB_WBL); // already set
   FW_NORMAL_ASSERT(true == configuration.wblActive); // configuration for WBL part is needed

   // create needed control handler
   createControlHandler(_component, _stackInterface, BTS_FB_WBL, userMode, configuration.wblConfiguration);

   // check control handler list
   executeStackConfiguration(_component, _stackInterface, BTS_FB_WBL, userMode, dbusInterfaces, configuration);

   ETG_TRACE_USR1((" setStackConfiguration(): [exit]"));
}

void WblControlMain::triggerInitializedCallback(void)
{
   // check control handler list
   if(false == processTriggerInitializedCallback())
   {
      FW_NORMAL_ASSERT_ALWAYS();
      ETG_TRACE_FATAL((" triggerInitializedCallback(): not processed"));
   }
}

void WblControlMain::createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent)
{
   BTSWblDbusServiceInterface serviceInterface = (BTSWblDbusServiceInterface)interface;

   if(BTS_WBL_DBUS_SERVICE_LAST <= serviceInterface)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(BTS_DBUS_SERVICE_LAST <= availabilityEvent)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   Ipc2Bts_ServiceAvailabilityWbl* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityWbl();

   if(0 != ptrMsg)
   {
      ptrMsg->setInterface(serviceInterface);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
      ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
   }

   sendInternalIpc2BtsMessage(ptrMsg, true);
}

void WblControlMain::createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent, IN const BTSBusName& busName, IN const BTSObjectPath& objPath, IN const BTSCommonEnumClass busType)
{
   BTSWblDbusServiceInterface serviceInterface = (BTSWblDbusServiceInterface)interface;

   if(BTS_WBL_DBUS_SERVICE_LAST <= serviceInterface)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(BTS_DBUS_SERVICE_LAST <= availabilityEvent)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(((BTSCommonEnumClass)::ccdbusif::BUS_TYPE_SYSTEM != busType) && ((BTSCommonEnumClass)::ccdbusif::BUS_TYPE_SESSION != busType))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   Ipc2Bts_ServiceAvailabilityWbl* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityWbl();

   if(0 != ptrMsg)
   {
      ptrMsg->setInterface(serviceInterface);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
      ptrMsg->setBusType(busType);
      ptrMsg->setBusName(busName);
      ptrMsg->setObjPath(objPath);
      ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
   }

   sendInternalIpc2BtsMessage(ptrMsg, true);
}

void WblControlMain::handleTimeout(IN IExtendedTimeoutHandler* handler, IN const BTSTimerId timerId)
{
   FW_IF_NULL_PTR_RETURN(handler);

   // HINT: if any error happens now the error has to be handled within this function or within any called sub-function

   ::std::vector< Bts2Ipc_BaseMessage* > sendBts2IpcMsgList;
   sendBts2IpcMsgList.reserve(10);
   ::std::vector< Bts2App_BaseMessage* > sendBts2AppMsgList;
   sendBts2AppMsgList.reserve(10);
   BTSHandleIpc2BtsMessageItem ipc2BtsMessageItem;

   // find related APP2BTS message => not possible in timeout function

   // special handling is needed for e.g. connect/disconnect
   ipc2BtsMessageItem.item.opCode = App2BtsOC_None;

   // handle timeout
   handler->handleExtendedTimeout(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, timerId);

   // NOTE: next part is identical to pushIpc2BtsMessage() except deleting message
   if(0 < sendBts2IpcMsgList.size())
   {
      if(0 != ipc2BtsMessageItem.message)
      {
         BTSApp2BtsMessageCompareItem compareItem;
         ipc2BtsMessageItem.message->getCompareItem(compareItem);
         sendBts2IpcMessageList(sendBts2IpcMsgList, compareItem);
      }
      else
      {
         // response message: it could be possible that there is no related application message
         // NOTE: it can also happen due to wrong implementation
         BTSApp2BtsMessageCompareItem compareItem;
         sendBts2IpcMessageList(sendBts2IpcMsgList, compareItem);
      }
   }

   sendBts2AppMessageList(sendBts2AppMsgList);

   // remove related APP2BTS message and check waiting queue
   if((true == ipc2BtsMessageItem.deleteMessage) && (0 != ipc2BtsMessageItem.message))
   {
      removeApp2BtsWorkingMessage(ipc2BtsMessageItem.message);

      if(App2BtsOC_None == ipc2BtsMessageItem.item.opCode)
      {
         checkWaitingQueue(ipc2BtsMessageItem.message);
      }

      delete ipc2BtsMessageItem.message;
   }

   // check waiting queue (special handling for connect/disconnect)
   if((App2BtsOC_None != ipc2BtsMessageItem.item.opCode) && (App2BtsOC_Ignore != ipc2BtsMessageItem.item.opCode))
   {
      if(0 != ipc2BtsMessageItem.checkIf)
      {
         ipc2BtsMessageItem.checkIf->checkWaitingQueue(ipc2BtsMessageItem.item);
      }
      else
      {
         checkWaitingQueueExtended(ipc2BtsMessageItem.item);
      }
   }
}

void WblControlMain::setSubControlTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   FW_IF_NULL_PTR_RETURN(testCommand);

   // check control handler list
   if(false == executeSubControlTestCommand(testCommand, testData))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      ETG_TRACE_FATAL((" setSubControlTestCommand(): testCommand %s not processed", testCommand));
   }
}

void WblControlMain::setSubControlTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   FW_IF_NULL_PTR_RETURN(testCommand);
   FW_IF_NULL_PTR_RETURN(testData);

   // check control handler list
   if(false == executeSubControlTestCommand(testCommand, testData))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      ETG_TRACE_FATAL((" setSubControlTestCommand(): testCommand %s not processed", testCommand));
   }
}

void WblControlMain::sendDirectAnswerForApp2BtsMessages(IN const ::std::vector< App2Bts_BaseMessage* >& msgList, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   if(0 == msgList.size())
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   App2Bts_BaseMessage* ptrMessage;

   for(size_t i = 0; i < msgList.size(); i++)
   {
      ptrMessage = msgList[i];

      if(0 == ptrMessage)
      {
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      if(_functionBlock != ptrMessage->getFunctionBlock())
      {
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      if(true == _handleDoubledRequests)
      {
         const BTSApp2BtsOpcode opCode(ptrMessage->getOpCode());
         ::std::vector< Bts2App_BaseMessage* > sendBts2AppMsgList;
         sendBts2AppMsgList.reserve(10);

         if((App2BtsOC_WblBlockStart < opCode) && (opCode < App2BtsOC_WblBlockEnd))
         {
            // check control handler list
            if(false == processDoubledApp2BtsMessages(sendBts2AppMsgList, ptrMessage, resultCode, statusCode))
            {
               FW_NORMAL_ASSERT_ALWAYS();
               ETG_TRACE_FATAL((" sendDirectAnswerForApp2BtsMessages(): App2Bts 0x%04X not processed", ptrMessage->getTraceOpCode()));
            }
         }
         else
         {
            // opcode in wrong range
            FW_NORMAL_ASSERT_ALWAYS();
         }

         sendBts2AppMessageList(sendBts2AppMsgList);
      }
   }
}

void WblControlMain::getSimilarOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, IN const App2Bts_BaseMessage* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   const BTSApp2BtsOpcode opCode(ptrMessage->getOpCode());

   if((App2BtsOC_WblBlockStart < opCode) && (opCode < App2BtsOC_WblBlockEnd))
   {
      // check control handler list
      if(false == requestSimilarOpCodes(itemList, ptrMessage))
      {
         FW_NORMAL_ASSERT_ALWAYS();
         ETG_TRACE_FATAL((" getSimilarOpCodes(): App2Bts 0x%04X not processed", ptrMessage->getTraceOpCode()));

         // add at least input opcode
         BTSApp2BtsMessageMasterCompareItem item;
         item.opCode = opCode;
         itemList.push_back(item);
      }
   }
   else
   {
      // opcode in wrong range
      FW_NORMAL_ASSERT_ALWAYS();

      // add at least input opcode
      BTSApp2BtsMessageMasterCompareItem item;
      item.opCode = opCode;
      itemList.push_back(item);
   }
}

void WblControlMain::getMatchingOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& highPrioItemList, IN const App2Bts_BaseMessage* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   const BTSApp2BtsOpcode opCode(ptrMessage->getOpCode());

   if((App2BtsOC_WblBlockStart < opCode) && (opCode < App2BtsOC_WblBlockEnd))
   {
      // check control handler list
      if(false == requestMatchingOpCodes(itemList, highPrioItemList, ptrMessage))
      {
         FW_NORMAL_ASSERT_ALWAYS();
         ETG_TRACE_FATAL((" getMatchingOpCodes(): App2Bts 0x%04X not processed", ptrMessage->getTraceOpCode()));

         // add at least input opcode
         BTSApp2BtsMessageMasterCompareItem item;
         ptrMessage->getCompareItem(item);
         item.opCode = opCode;
         itemList.push_back(item);
      }
   }
   else
   {
      // opcode in wrong range
      FW_NORMAL_ASSERT_ALWAYS();

      // add at least input opcode
      BTSApp2BtsMessageMasterCompareItem item;
      ptrMessage->getCompareItem(item);
      item.opCode = opCode;
      itemList.push_back(item);
   }
}

bool WblControlMain::skipSimilarOpCodeCheck(IN const App2Bts_BaseMessage* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN_FALSE(ptrMessage);

   // function not used
   FW_NORMAL_ASSERT_ALWAYS();

   return false;
}

void WblControlMain::checkWaitingQueueExtended(IN const BTSApp2BtsMessageCompareItem& compareItem)
{
   (void)(compareItem);

   // function not used
   FW_NORMAL_ASSERT_ALWAYS();
}

void WblControlMain::handleDoubledApp2BtsMessages(IN const ::std::vector< App2Bts_BaseMessage* >& msgList)
{
   if(0 == msgList.size())
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   sendDirectAnswerForApp2BtsMessages(msgList, (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);
}

void WblControlMain::createControlHandler(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode, IN const BTSLocalWblConfiguration& configuration)
{
   (void)(userMode);
   (void)(configuration);

   IControlHandler* handler(new WblControlHandler(component, stackInterface, subComponent));
   if(0 != handler)
   {
      handler->setControlIf(this);
      handler->setTimerPoolIf(&TimerPool::getInstance());
      addControlHandler(handler);
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

} //wbl
} //btstackif
