/**
 * @file WblControlHandler.cpp
 *
 * @par SW-Component
 * Control
 *
 * @brief WBL Control Handler.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of WBL control handling.
 */

#include "WblControlHandler.h"
#include "IBasicControl.h"
#include "ITimerPool.h"
#include "App2Bts_MessageWrapper.h"
#include "Ipc2Bts_MessageWrapper_WBL.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#include "WblStartup.h"
#include "WblLastMode.h"
#include "WblWiFiSetupStatus.h"

#include "WblStartupWbl.h"
#include "WblLastModeWbl.h"
#include "WblWiFiSetupStatusWbl.h"
#include "WblObjectPathManagerWbl.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WblControlHandler.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace wbl {

WblControlHandler::WblControlHandler(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent) :
_component(component),
_stackInterface(stackInterface),
_subComponent(subComponent),
_basicControlIf(0),
_timerPool(0),
_wblStartup(0),
_wblLastMode(0),
_wblWiFiSetupStatus(0),
_wblStartupWbl(0),
_wblLastModeWbl(0),
_wblWiFiSetupStatusWbl(0),
_wblObjectPathManagerWbl(0)
{
}

WblControlHandler::WblControlHandler(const WblControlHandler& ref) :
_component(ref._component),
_stackInterface(ref._stackInterface),
_subComponent(ref._subComponent),
_basicControlIf(0),
_timerPool(0),
_wblStartup(0),
_wblLastMode(0),
_wblWiFiSetupStatus(0),
_wblStartupWbl(0),
_wblLastModeWbl(0),
_wblWiFiSetupStatusWbl(0),
_wblObjectPathManagerWbl(0)
{
   // DO NOT USE!!!
   FW_NORMAL_ASSERT_ALWAYS();
}

WblControlHandler& WblControlHandler::operator=(const WblControlHandler& ref)
{
   // DO NOT USE!!!
   FW_NORMAL_ASSERT_ALWAYS();

   if(this == &ref)
   {
      return *this;
   }

   // ignore given parameter

   return *this;
}

WblControlHandler::~WblControlHandler()
{
   _basicControlIf = 0;
   _timerPool = 0;

   // destroy SMs
   destroySMs();

   _wblStartup = 0;
   _wblLastMode = 0;
   _wblWiFiSetupStatus = 0;
   _wblStartupWbl = 0;
   _wblLastModeWbl = 0;
   _wblWiFiSetupStatusWbl = 0;
   _wblObjectPathManagerWbl = 0;
}

void WblControlHandler::setControlIf(IN IBasicControl* control)
{
   _basicControlIf = control;

   FW_NORMAL_ASSERT(0 != _basicControlIf);
}

void WblControlHandler::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPool = timerPool;

   FW_NORMAL_ASSERT(0 != _timerPool);
}

void WblControlHandler::enable(IN const bool enable)
{
   (void)(enable);
   // ignore
}

void WblControlHandler::initialize(void)
{
   // ignore
}

void WblControlHandler::destroy(void)
{
   // ignore
}

void WblControlHandler::setStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         OUT ::std::vector<BTSDbusInterfaceItem>& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   (void)(userMode);

   ETG_TRACE_USR1((" setStackConfiguration(): [enter]"));

   FW_IF_NULL_PTR_RETURN(_basicControlIf);
   FW_IF_NULL_PTR_RETURN(_timerPool);

   FW_NORMAL_ASSERT(component == _component); // already set
   FW_NORMAL_ASSERT(stackInterface == _stackInterface); // already set
   FW_NORMAL_ASSERT(subComponent == _subComponent); // already set
   FW_NORMAL_ASSERT(true == configuration.wblActive); // configuration for WBL part is needed

   // create SMs
   initializeSMs();

   // assign SMs
   assignSMs();

   FW_IF_NULL_PTR_RETURN(_wblStartupWbl);
   IWblStartupWbl& wblStartupWbl = *_wblStartupWbl;
   wblStartupWbl.addInterfaces(dbusInterfaces);

   ETG_TRACE_USR1((" setStackConfiguration(): [exit]"));
}

bool WblControlHandler::doApp2BtsMsgPrecheck(OUT bool& continueProcessing, OUT bool& rejectRequest, OUT BTSCommonEnumClass& resultCode, OUT BTSCommonEnumClass& statusCode, OUT bool& skipOpCodeCheck, IN App2Bts_BaseMessage* message)
{
   (void)(rejectRequest);
   (void)(resultCode);
   (void)(statusCode);
   (void)(skipOpCodeCheck);

   FW_IF_NULL_PTR_RETURN_FALSE(message);
   App2Bts_BaseMessage& messageRef(*message);

   bool handled(true);
   const BTSApp2BtsOpcode opCode(messageRef.getOpCode());

   // no check for service availability because we will not track any timeout

   switch(opCode)
   {
      case App2BtsOC_SetWblDummy:
         // continue
         break;
      case App2BtsOC_TriggerFbWblInitialized:
         // continue
         break;
      case App2BtsOC_AllowWblCommunication:
         // continue
         break;
      case App2BtsOC_BlockWblCommunication:
         // continue
         break;
      case App2BtsOC_GetWblCommunicationStatus:
         // continue
         break;
      case App2BtsOC_GetWblServiceAvailability:
         // continue
         break;
      case App2BtsOC_GetWblLastIntendedMode:
         // continue
         break;
      case App2BtsOC_GetWblSupportedRestrictions:
         // continue
         break;
      case App2BtsOC_GetWblWiFiSetupStatus:
         // continue
         break;
      case App2BtsOC_GetWblHealthinessIndicator:
         // continue
         break;
      default:
         handled = false;
         break;
   }

   continueProcessing = handled;

   return handled;
}

bool WblControlHandler::getSimilarOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, IN const App2Bts_BaseMessage* message) const
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   const App2Bts_BaseMessage& messageRef(*message);

   bool handled(true);
   const BTSApp2BtsOpcode opCode(messageRef.getOpCode());
   BTSApp2BtsMessageMasterCompareItem item;

   switch(opCode)
   {
      case App2BtsOC_SetWblDummy:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_TriggerFbWblInitialized:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_AllowWblCommunication:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_BlockWblCommunication:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_GetWblCommunicationStatus:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_GetWblServiceAvailability:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_GetWblLastIntendedMode:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_GetWblSupportedRestrictions:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_GetWblWiFiSetupStatus:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_GetWblHealthinessIndicator:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      default:
         handled = false;
         break;
   }

   return handled;
}

bool WblControlHandler::getMatchingOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& highPrioItemList, IN const App2Bts_BaseMessage* message) const
{
   (void)(highPrioItemList);

   FW_IF_NULL_PTR_RETURN_FALSE(message);
   const App2Bts_BaseMessage& messageRef(*message);

   bool handled(true);
   const BTSApp2BtsOpcode opCode(messageRef.getOpCode());
   BTSApp2BtsMessageMasterCompareItem item;
   messageRef.getCompareItem(item);

   switch(opCode)
   {
      case App2BtsOC_SetWblDummy:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_TriggerFbWblInitialized:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_AllowWblCommunication:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_BlockWblCommunication:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_GetWblCommunicationStatus:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_GetWblServiceAvailability:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_GetWblLastIntendedMode:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_GetWblSupportedRestrictions:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_GetWblWiFiSetupStatus:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      case App2BtsOC_GetWblHealthinessIndicator:
         item.opCode = opCode;
         itemList.push_back(item);
         break;
      default:
         handled = false;
         break;
   }

   return handled;
}

bool WblControlHandler::triggerInitializedCallback(void)
{
   FW_IF_NULL_PTR_RETURN_FALSE(_wblStartup);
   IWblStartup& wblStartup = *_wblStartup;

   wblStartup.processTriggerInitializedCallback();

   return true;
}

bool WblControlHandler::handleDoubledApp2BtsMessages(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_BaseMessage* message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   App2Bts_BaseMessage& messageRef(*message);

   bool handled(true);
   const BTSApp2BtsOpcode opCode(messageRef.getOpCode());

   // handle doubled messages depending on opcode
   // - create direct answer message (Bts2App) in sub-handler function
   // - handle any error in sub-handler function because there is the best place to handle
   // - do not delete message; this is done outside

   switch(opCode)
   {
      case App2BtsOC_SetWblDummy:
         handleDoubledApp2BtsSetWblDummy(bts2AppMsgList, static_cast< App2Bts_SetWblDummy& >(messageRef), resultCode, statusCode);
         break;
      case App2BtsOC_TriggerFbWblInitialized:
         handleDoubledApp2BtsTriggerFbWblInitialized(bts2AppMsgList, static_cast< App2Bts_TriggerFbWblInitialized& >(messageRef), resultCode, statusCode);
         break;
      case App2BtsOC_AllowWblCommunication:
         handleDoubledApp2BtsAllowWblCommunication(bts2AppMsgList, static_cast< App2Bts_AllowWblCommunication& >(messageRef), resultCode, statusCode);
         break;
      case App2BtsOC_BlockWblCommunication:
         handleDoubledApp2BtsBlockWblCommunication(bts2AppMsgList, static_cast< App2Bts_BlockWblCommunication& >(messageRef), resultCode, statusCode);
         break;
      case App2BtsOC_GetWblCommunicationStatus:
         handleDoubledApp2BtsGetWblCommunicationStatus(bts2AppMsgList, static_cast< App2Bts_GetWblCommunicationStatus& >(messageRef), resultCode, statusCode);
         break;
      case App2BtsOC_GetWblServiceAvailability:
         handleDoubledApp2BtsGetWblServiceAvailability(bts2AppMsgList, static_cast< App2Bts_GetWblServiceAvailability& >(messageRef), resultCode, statusCode);
         break;
      case App2BtsOC_GetWblLastIntendedMode:
         handleDoubledApp2BtsGetWblLastIntendedMode(bts2AppMsgList, static_cast< App2Bts_GetWblLastIntendedMode& >(messageRef), resultCode, statusCode);
         break;
      case App2BtsOC_GetWblSupportedRestrictions:
         handleDoubledApp2BtsGetWblSupportedRestrictions(bts2AppMsgList, static_cast< App2Bts_GetWblSupportedRestrictions& >(messageRef), resultCode, statusCode);
         break;
      case App2BtsOC_GetWblWiFiSetupStatus:
         handleDoubledApp2BtsGetWblWiFiSetupStatus(bts2AppMsgList, static_cast< App2Bts_GetWblWiFiSetupStatus& >(messageRef), resultCode, statusCode);
         break;
      case App2BtsOC_GetWblHealthinessIndicator:
         handleDoubledApp2BtsGetWblHealthinessIndicator(bts2AppMsgList, static_cast< App2Bts_GetWblHealthinessIndicator& >(messageRef), resultCode, statusCode);
         break;
      default:
         handled = false;
         break;
   }

   return handled;
}

bool WblControlHandler::handleApp2BtsMessage(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_BaseMessage* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   App2Bts_BaseMessage& messageRef(*message);

   bool handled(true);
   const BTSApp2BtsOpcode opCode(messageRef.getOpCode());

   // handle message depending on opcode
   // - set marker to delete message in sub-handler function if necessary
   // - push message to working queue in sub-handler function if necessary
   // - create direct answer message (Bts2App) in sub-handler function if necessary
   // - create Bts2Ipc message in sub-handler function if necessary
   // - handle any error in sub-handler function because there is the best place to handle

   switch(opCode)
   {
      case App2BtsOC_SetWblDummy:
         handleApp2BtsSetWblDummy(bts2IpcMsgList, bts2AppMsgList, deleteApp2BtsMessage, static_cast< App2Bts_SetWblDummy& >(messageRef));
         break;
      case App2BtsOC_TriggerFbWblInitialized:
         handleApp2BtsTriggerFbWblInitialized(bts2IpcMsgList, bts2AppMsgList, deleteApp2BtsMessage, static_cast< App2Bts_TriggerFbWblInitialized& >(messageRef));
         break;
      case App2BtsOC_AllowWblCommunication:
         handleApp2BtsAllowWblCommunication(bts2IpcMsgList, bts2AppMsgList, deleteApp2BtsMessage, static_cast< App2Bts_AllowWblCommunication& >(messageRef));
         break;
      case App2BtsOC_BlockWblCommunication:
         handleApp2BtsBlockWblCommunication(bts2IpcMsgList, bts2AppMsgList, deleteApp2BtsMessage, static_cast< App2Bts_BlockWblCommunication& >(messageRef));
         break;
      case App2BtsOC_GetWblCommunicationStatus:
         handleApp2BtsGetWblCommunicationStatus(bts2IpcMsgList, bts2AppMsgList, deleteApp2BtsMessage, static_cast< App2Bts_GetWblCommunicationStatus& >(messageRef));
         break;
      case App2BtsOC_GetWblServiceAvailability:
         handleApp2BtsGetWblServiceAvailability(bts2IpcMsgList, bts2AppMsgList, deleteApp2BtsMessage, static_cast< App2Bts_GetWblServiceAvailability& >(messageRef));
         break;
      case App2BtsOC_GetWblLastIntendedMode:
         handleApp2BtsGetWblLastIntendedMode(bts2IpcMsgList, bts2AppMsgList, deleteApp2BtsMessage, static_cast< App2Bts_GetWblLastIntendedMode& >(messageRef));
         break;
      case App2BtsOC_GetWblSupportedRestrictions:
         handleApp2BtsGetWblSupportedRestrictions(bts2IpcMsgList, bts2AppMsgList, deleteApp2BtsMessage, static_cast< App2Bts_GetWblSupportedRestrictions& >(messageRef));
         break;
      case App2BtsOC_GetWblWiFiSetupStatus:
         handleApp2BtsGetWblWiFiSetupStatus(bts2IpcMsgList, bts2AppMsgList, deleteApp2BtsMessage, static_cast< App2Bts_GetWblWiFiSetupStatus& >(messageRef));
         break;
      case App2BtsOC_GetWblHealthinessIndicator:
         handleApp2BtsGetWblHealthinessIndicator(bts2IpcMsgList, bts2AppMsgList, deleteApp2BtsMessage, static_cast< App2Bts_GetWblHealthinessIndicator& >(messageRef));
         break;
      default:
         handled = false;
         break;
   }

   return handled;
}

bool WblControlHandler::handleIpc2BtsMessage(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BaseMessage* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   Ipc2Bts_BaseMessage& messageRef(*message);

   bool handled(true);
   const BTSWblIpc2BtsOpcode opCode((BTSWblIpc2BtsOpcode)messageRef.getOpCode());

   // handle message depending on opcode
   // - do not delete Ipc2Bts message in sub-handler function because this done outside
   // - set marker to delete related App2Bts message in sub-handler function if necessary
   // - create Bts2App message in sub-handler function if necessary
   // - create Bts2Ipc message in sub-handler function if necessary
   // - handle any error in sub-handler function because there is the best place to handle

   switch(opCode)
   {
      case Ipc2BtsOC_ServiceAvailabilityWbl:
         handleIpc2BtsServiceAvailabilityWbl(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast< Ipc2Bts_ServiceAvailabilityWbl& >(messageRef));
         break;
      case Ipc2BtsOC_GetManagedObjectsWbl:
         handleIpc2BtsGetManagedObjectsWbl(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast< Ipc2Bts_GetManagedObjectsWbl& >(messageRef));
         break;
      case Ipc2BtsOC_AddWifiSetupObjectPath:
         handleIpc2BtsAddWifiSetupObjectPath(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast< Ipc2Bts_AddWifiSetupObjectPath& >(messageRef));
         break;
      case Ipc2BtsOC_DelWifiSetupObjectPath:
         handleIpc2BtsDelWifiSetupObjectPath(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast< Ipc2Bts_DelWifiSetupObjectPath& >(messageRef));
         break;
      case Ipc2BtsOC_LastIntendedModeUpdate:
         handleIpc2BtsLastIntendedModeUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast< Ipc2Bts_LastIntendedModeUpdate& >(messageRef));
         break;
      case Ipc2BtsOC_SupportedFrequenciesUpdate:
         handleIpc2BtsSupportedFrequenciesUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast< Ipc2Bts_SupportedFrequenciesUpdate& >(messageRef));
         break;
      case Ipc2BtsOC_WifiModeUpdate:
         handleIpc2BtsWifiModeUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast< Ipc2Bts_WifiModeUpdate& >(messageRef));
         break;
      case Ipc2BtsOC_WifiPowerStateUpdate:
         handleIpc2BtsWifiPowerStateUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast< Ipc2Bts_WifiPowerStateUpdate& >(messageRef));
         break;
      case Ipc2BtsOC_WifiFrequencyUpdate:
         handleIpc2BtsWifiFrequencyUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast< Ipc2Bts_WifiFrequencyUpdate& >(messageRef));
         break;
      case Ipc2BtsOC_AccessPointConfigTypeUpdate:
         handleIpc2BtsAccessPointConfigTypeUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast< Ipc2Bts_AccessPointConfigTypeUpdate& >(messageRef));
         break;
      case Ipc2BtsOC_WifiConnectionStatusUpdate:
         handleIpc2BtsWifiConnectionStatusUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast< Ipc2Bts_WifiConnectionStatusUpdate& >(messageRef));
         break;
      default:
         handled = false;
         break;
   }

   return handled;
}

bool WblControlHandler::setSubControlTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   (void)(testCommand);
   (void)(testData);
   bool handled(false);
   return handled;
}

bool WblControlHandler::setSubControlTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   (void)(testCommand);
   (void)(testData);
   bool handled(false);
   return handled;
}

bool WblControlHandler::handleTTFisCommand(IN const unsigned char* testData)
{
   (void)(testData);
   bool handled(false);
   return handled;
}

bool WblControlHandler::handleApp2BtsTestCommand(IN const unsigned char* testData)
{
   (void)(testData);
   bool handled(false);
   return handled;
}

void WblControlHandler::initializeSMs(void)
{
   if(0 == _wblStartup)
   {
      _wblStartup = new WblStartup();
   }
   FW_NORMAL_ASSERT(0 != _wblStartup);

   if(0 == _wblLastMode)
   {
      _wblLastMode = new WblLastMode();
   }
   FW_NORMAL_ASSERT(0 != _wblLastMode);

   if(0 == _wblWiFiSetupStatus)
   {
      _wblWiFiSetupStatus = new WblWiFiSetupStatus();
   }
   FW_NORMAL_ASSERT(0 != _wblWiFiSetupStatus);

   if(0 == _wblStartupWbl)
   {
      _wblStartupWbl = new WblStartupWbl();
   }
   FW_NORMAL_ASSERT(0 != _wblStartupWbl);

   if(0 == _wblLastModeWbl)
   {
      _wblLastModeWbl = new WblLastModeWbl();
   }
   FW_NORMAL_ASSERT(0 != _wblLastModeWbl);

   if(0 == _wblWiFiSetupStatusWbl)
   {
      _wblWiFiSetupStatusWbl = new WblWiFiSetupStatusWbl();
   }
   FW_NORMAL_ASSERT(0 != _wblWiFiSetupStatusWbl);

   if(0 == _wblObjectPathManagerWbl)
   {
      _wblObjectPathManagerWbl = new WblObjectPathManagerWbl();
   }
   FW_NORMAL_ASSERT(0 != _wblObjectPathManagerWbl);
}

void WblControlHandler::destroySMs(void)
{
   if(0 != _wblStartup)
   {
      delete _wblStartup;
      _wblStartup = 0;
   }

   if(0 != _wblLastMode)
   {
      delete _wblLastMode;
      _wblLastMode = 0;
   }

   if(0 != _wblWiFiSetupStatus)
   {
      delete _wblWiFiSetupStatus;
      _wblWiFiSetupStatus = 0;
   }

   if(0 != _wblStartupWbl)
   {
      delete _wblStartupWbl;
      _wblStartupWbl = 0;
   }

   if(0 != _wblLastModeWbl)
   {
      delete _wblLastModeWbl;
      _wblLastModeWbl = 0;
   }

   if(0 != _wblWiFiSetupStatusWbl)
   {
      delete _wblWiFiSetupStatusWbl;
      _wblWiFiSetupStatusWbl = 0;
   }

   if(0 != _wblObjectPathManagerWbl)
   {
      delete _wblObjectPathManagerWbl;
      _wblObjectPathManagerWbl = 0;
   }
}

void WblControlHandler::assignSMs(void)
{
   FW_IF_NULL_PTR_RETURN(_basicControlIf);
   FW_IF_NULL_PTR_RETURN(_timerPool);
   FW_IF_NULL_PTR_RETURN(_wblStartup);
   FW_IF_NULL_PTR_RETURN(_wblLastMode);
   FW_IF_NULL_PTR_RETURN(_wblWiFiSetupStatus);
   FW_IF_NULL_PTR_RETURN(_wblStartupWbl);
   FW_IF_NULL_PTR_RETURN(_wblLastModeWbl);
   FW_IF_NULL_PTR_RETURN(_wblWiFiSetupStatusWbl);
   FW_IF_NULL_PTR_RETURN(_wblObjectPathManagerWbl);

   _wblStartup->setInstance(_wblStartupWbl->getRequestIf());
   _wblStartup->setControlIf(_basicControlIf);
   _wblStartup->setTimerPoolIf(_timerPool);
   _wblStartup->registerObserver(_wblLastMode->getStartupObserver());
   _wblStartup->registerObserver(_wblWiFiSetupStatus->getStartupObserver());

   // nothing to do for _wblStartupWbl

   _wblLastMode->setInstance(_wblLastModeWbl->getRequestIf());
   _wblLastMode->setControlIf(_basicControlIf);
   _wblLastMode->setTimerPoolIf(_timerPool);
   _wblLastMode->setWblStartupIf(_wblStartup);

   _wblLastModeWbl->setObjectPathManagerIf(_wblObjectPathManagerWbl);

   _wblWiFiSetupStatus->setInstance(_wblWiFiSetupStatusWbl->getRequestIf());
   _wblWiFiSetupStatus->setControlIf(_basicControlIf);
   _wblWiFiSetupStatus->setTimerPoolIf(_timerPool);
   _wblWiFiSetupStatus->setWblStartupIf(_wblStartup);

   _wblWiFiSetupStatusWbl->setObjectPathManagerIf(_wblObjectPathManagerWbl);
}

void WblControlHandler::handleDoubledApp2BtsSetWblDummy(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_SetWblDummy& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(message);
   (void)(resultCode);
   (void)(statusCode);
   // nothing to do
}

void WblControlHandler::handleDoubledApp2BtsTriggerFbWblInitialized(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_TriggerFbWblInitialized& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(message);
   (void)(resultCode);
   (void)(statusCode);
   // nothing to do
}

void WblControlHandler::handleDoubledApp2BtsAllowWblCommunication(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_AllowWblCommunication& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(message);
   (void)(resultCode);
   (void)(statusCode);
   // nothing to do
}

void WblControlHandler::handleDoubledApp2BtsBlockWblCommunication(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_BlockWblCommunication& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(message);
   (void)(resultCode);
   (void)(statusCode);
   // nothing to do
}

void WblControlHandler::handleDoubledApp2BtsGetWblCommunicationStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_GetWblCommunicationStatus& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(message);
   (void)(resultCode);
   (void)(statusCode);
   // nothing to do
}

void WblControlHandler::handleDoubledApp2BtsGetWblServiceAvailability(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_GetWblServiceAvailability& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(message);
   (void)(resultCode);
   (void)(statusCode);
   // nothing to do
}

void WblControlHandler::handleDoubledApp2BtsGetWblLastIntendedMode(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_GetWblLastIntendedMode& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(message);
   (void)(resultCode);
   (void)(statusCode);
   // nothing to do
}

void WblControlHandler::handleDoubledApp2BtsGetWblSupportedRestrictions(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_GetWblSupportedRestrictions& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(message);
   (void)(resultCode);
   (void)(statusCode);
   // nothing to do
}

void WblControlHandler::handleDoubledApp2BtsGetWblWiFiSetupStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_GetWblWiFiSetupStatus& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(message);
   (void)(resultCode);
   (void)(statusCode);
   // nothing to do
}

void WblControlHandler::handleDoubledApp2BtsGetWblHealthinessIndicator(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_GetWblHealthinessIndicator& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(message);
   (void)(resultCode);
   (void)(statusCode);
   // nothing to do
}

void WblControlHandler::handleApp2BtsSetWblDummy(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetWblDummy& message)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(message);
   deleteApp2BtsMessage = true;
}

void WblControlHandler::handleApp2BtsTriggerFbWblInitialized(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_TriggerFbWblInitialized& message)
{
   FW_IF_NULL_PTR_RETURN(_basicControlIf);
   FW_IF_NULL_PTR_RETURN(_wblStartup);
   IWblStartup& wblStartup = *_wblStartup;

   if(true == wblStartup.processTriggerInitializedCallback(bts2IpcMsgList, bts2AppMsgList, message))
   {
      _basicControlIf->pushApp2BtsMsgToWorkingQueue(message.getMessagePtr(), false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void WblControlHandler::handleApp2BtsAllowWblCommunication(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_AllowWblCommunication& message)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(message);
   deleteApp2BtsMessage = true;
}

void WblControlHandler::handleApp2BtsBlockWblCommunication(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_BlockWblCommunication& message)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(message);
   deleteApp2BtsMessage = true;
}

void WblControlHandler::handleApp2BtsGetWblCommunicationStatus(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetWblCommunicationStatus& message)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(message);
   deleteApp2BtsMessage = true;
}

void WblControlHandler::handleApp2BtsGetWblServiceAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetWblServiceAvailability& message)
{
   FW_IF_NULL_PTR_RETURN(_basicControlIf);
   FW_IF_NULL_PTR_RETURN(_wblStartup);
   IWblStartup& wblStartup = *_wblStartup;

   if(true == wblStartup.getServiceAvailability(bts2IpcMsgList, bts2AppMsgList, message))
   {
      _basicControlIf->pushApp2BtsMsgToWorkingQueue(message.getMessagePtr(), false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void WblControlHandler::handleApp2BtsGetWblLastIntendedMode(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetWblLastIntendedMode& message)
{
   FW_IF_NULL_PTR_RETURN(_basicControlIf);
   FW_IF_NULL_PTR_RETURN(_wblLastMode);
   IWblLastMode& wblLastMode = *_wblLastMode;

   if(true == wblLastMode.getLastMode(bts2IpcMsgList, bts2AppMsgList, message))
   {
      _basicControlIf->pushApp2BtsMsgToWorkingQueue(message.getMessagePtr(), false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void WblControlHandler::handleApp2BtsGetWblSupportedRestrictions(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetWblSupportedRestrictions& message)
{
   FW_IF_NULL_PTR_RETURN(_basicControlIf);
   FW_IF_NULL_PTR_RETURN(_wblWiFiSetupStatus);
   IWblWiFiSetupStatus& wblWiFiSetupStatus = *_wblWiFiSetupStatus;

   if(true == wblWiFiSetupStatus.getSupportedRestrictions(bts2IpcMsgList, bts2AppMsgList, message))
   {
      _basicControlIf->pushApp2BtsMsgToWorkingQueue(message.getMessagePtr(), false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void WblControlHandler::handleApp2BtsGetWblWiFiSetupStatus(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetWblWiFiSetupStatus& message)
{
   FW_IF_NULL_PTR_RETURN(_basicControlIf);
   FW_IF_NULL_PTR_RETURN(_wblWiFiSetupStatus);
   IWblWiFiSetupStatus& wblWiFiSetupStatus = *_wblWiFiSetupStatus;

   if(true == wblWiFiSetupStatus.getWiFiSetupStatus(bts2IpcMsgList, bts2AppMsgList, message))
   {
      _basicControlIf->pushApp2BtsMsgToWorkingQueue(message.getMessagePtr(), false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void WblControlHandler::handleApp2BtsGetWblHealthinessIndicator(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetWblHealthinessIndicator& message)
{
   FW_IF_NULL_PTR_RETURN(_basicControlIf);
   FW_IF_NULL_PTR_RETURN(_wblWiFiSetupStatus);
   IWblWiFiSetupStatus& wblWiFiSetupStatus = *_wblWiFiSetupStatus;

   if(true == wblWiFiSetupStatus.getWiFiHealthinessIndicator(bts2IpcMsgList, bts2AppMsgList, message))
   {
      _basicControlIf->pushApp2BtsMsgToWorkingQueue(message.getMessagePtr(), false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void WblControlHandler::handleIpc2BtsServiceAvailabilityWbl(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ServiceAvailabilityWbl& message)
{
   FW_IF_NULL_PTR_RETURN(_wblStartupWbl);
   FW_IF_NULL_PTR_RETURN(_wblWiFiSetupStatusWbl);
   IWblStartupWbl& wblStartupWbl = *_wblStartupWbl;
   IWblWiFiSetupStatusWbl& wblWiFiSetupStatusWbl = *_wblWiFiSetupStatusWbl;

   wblStartupWbl.updateServiceAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getInterface(), message.getAvailabilityEvent());
   wblWiFiSetupStatusWbl.updateServiceAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getInterface(), message.getAvailabilityEvent(), message.getObjPath());
}

void WblControlHandler::handleIpc2BtsGetManagedObjectsWbl(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetManagedObjectsWbl& message)
{
   FW_IF_NULL_PTR_RETURN(_wblWiFiSetupStatusWbl);
   IWblWiFiSetupStatusWbl& wblWiFiSetupStatusWbl = *_wblWiFiSetupStatusWbl;

   wblWiFiSetupStatusWbl.handleGetManagedObjectsResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode());
}

void WblControlHandler::handleIpc2BtsAddWifiSetupObjectPath(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AddWifiSetupObjectPath& message)
{
   FW_IF_NULL_PTR_RETURN(_wblWiFiSetupStatusWbl);
   IWblWiFiSetupStatusWbl& wblWiFiSetupStatusWbl = *_wblWiFiSetupStatusWbl;

   wblWiFiSetupStatusWbl.wifiSetupEntryAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath());
}

void WblControlHandler::handleIpc2BtsDelWifiSetupObjectPath(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DelWifiSetupObjectPath& message)
{
   FW_IF_NULL_PTR_RETURN(_wblWiFiSetupStatusWbl);
   IWblWiFiSetupStatusWbl& wblWiFiSetupStatusWbl = *_wblWiFiSetupStatusWbl;

   wblWiFiSetupStatusWbl.wifiSetupEntryRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath());
}

void WblControlHandler::handleIpc2BtsLastIntendedModeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LastIntendedModeUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_wblLastModeWbl);
   IWblLastModeWbl& wblLastModeWbl = *_wblLastModeWbl;

   wblLastModeWbl.updateLastMode(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getWifiSetupStatusList());
}

void WblControlHandler::handleIpc2BtsSupportedFrequenciesUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SupportedFrequenciesUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_wblWiFiSetupStatusWbl);
   IWblWiFiSetupStatusWbl& wblWiFiSetupStatusWbl = *_wblWiFiSetupStatusWbl;

   wblWiFiSetupStatusWbl.updateSupportedRestrictions(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getObjPath(), message.getSupportedRestrictions());
}

void WblControlHandler::handleIpc2BtsWifiModeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_WifiModeUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_wblWiFiSetupStatusWbl);
   IWblWiFiSetupStatusWbl& wblWiFiSetupStatusWbl = *_wblWiFiSetupStatusWbl;

   wblWiFiSetupStatusWbl.updateWifiMode(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getObjPath(), message.getMode());
}

void WblControlHandler::handleIpc2BtsWifiPowerStateUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_WifiPowerStateUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_wblWiFiSetupStatusWbl);
   IWblWiFiSetupStatusWbl& wblWiFiSetupStatusWbl = *_wblWiFiSetupStatusWbl;

   wblWiFiSetupStatusWbl.updateWifiPowerState(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getObjPath(), message.getPowerState());
}

void WblControlHandler::handleIpc2BtsWifiFrequencyUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_WifiFrequencyUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_wblWiFiSetupStatusWbl);
   IWblWiFiSetupStatusWbl& wblWiFiSetupStatusWbl = *_wblWiFiSetupStatusWbl;

   wblWiFiSetupStatusWbl.updateWifiFrequency(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getObjPath(), message.getFrequency());
}

void WblControlHandler::handleIpc2BtsAccessPointConfigTypeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AccessPointConfigTypeUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_wblWiFiSetupStatusWbl);
   IWblWiFiSetupStatusWbl& wblWiFiSetupStatusWbl = *_wblWiFiSetupStatusWbl;

   wblWiFiSetupStatusWbl.updateAccessPointConfigType(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getObjPath(), message.getType());
}

void WblControlHandler::handleIpc2BtsWifiConnectionStatusUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_WifiConnectionStatusUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_wblWiFiSetupStatusWbl);
   IWblWiFiSetupStatusWbl& wblWiFiSetupStatusWbl = *_wblWiFiSetupStatusWbl;

   wblWiFiSetupStatusWbl.updateWifiConnectionStatus(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getObjPath(), message.getConnected(), message.getHealthIndicator());
}

} //wbl
} //btstackif
