/**
 * @file WblControlHandler.h
 *
 * @par SW-Component
 * Control
 *
 * @brief WBL Control Handler.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Class for WBL control handling.
 */

#ifndef _WBL_CONTROL_HANDLER_H_
#define _WBL_CONTROL_HANDLER_H_

#include "WblStackTypes.h"
#include "IControlHandler.h"

namespace btstackif {

// class forward declarations
// generic SMs
class IWblStartup;
class IWblLastMode;
class IWblWiFiSetupStatus;

// App2Bts messages
class App2Bts_SetWblDummy;
class App2Bts_TriggerFbWblInitialized;
class App2Bts_AllowWblCommunication;
class App2Bts_BlockWblCommunication;
class App2Bts_GetWblCommunicationStatus;
class App2Bts_GetWblServiceAvailability;
class App2Bts_GetWblLastIntendedMode;
class App2Bts_GetWblSupportedRestrictions;
class App2Bts_GetWblWiFiSetupStatus;
class App2Bts_GetWblHealthinessIndicator;

namespace wbl {

// class forward declarations
// WBL specific SMs
class IWblStartupWbl;
class IWblLastModeWbl;
class IWblWiFiSetupStatusWbl;
class IWblObjectPathManagerWbl;

// Ipc2Bts messages
class Ipc2Bts_ServiceAvailabilityWbl;
class Ipc2Bts_GetManagedObjectsWbl;
class Ipc2Bts_AddWifiSetupObjectPath;
class Ipc2Bts_DelWifiSetupObjectPath;
class Ipc2Bts_LastIntendedModeUpdate;
class Ipc2Bts_SupportedFrequenciesUpdate;
class Ipc2Bts_WifiModeUpdate;
class Ipc2Bts_WifiPowerStateUpdate;
class Ipc2Bts_WifiFrequencyUpdate;
class Ipc2Bts_AccessPointConfigTypeUpdate;
class Ipc2Bts_WifiConnectionStatusUpdate;

/**
 *
 */
class WblControlHandler : public IControlHandler
{
public:
   /**************************************************************************
    Constructor / destructor
    **************************************************************************/
   /**
    * Constructor.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    */
   WblControlHandler(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent);

private:
   /**
    * Copy constructor.
    *
    * DO NOT USE!!!
    *
    * @param[in] ref: reference of object to be copied
    */
   WblControlHandler(const WblControlHandler& ref);

   /**
    * Assignment operator.
    *
    * DO NOT USE!!!
    *
    * @param[in] ref: reference of object to be assigned
    */
   WblControlHandler& operator=(const WblControlHandler& ref);

public:
   /**
    * Destructor.
    */
   virtual ~WblControlHandler();

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setTimerPoolIf(IN ITimerPool* timerPool);

   virtual void enable(IN const bool enable);

   virtual void initialize(void);

   virtual void destroy(void);

   virtual void setStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
            OUT ::std::vector<BTSDbusInterfaceItem>& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration);

   virtual bool doApp2BtsMsgPrecheck(OUT bool& continueProcessing, OUT bool& rejectRequest, OUT BTSCommonEnumClass& resultCode, OUT BTSCommonEnumClass& statusCode, OUT bool& skipOpCodeCheck, IN App2Bts_BaseMessage* message);

   virtual bool getSimilarOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, IN const App2Bts_BaseMessage* message) const;

   virtual bool getMatchingOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& highPrioItemList, IN const App2Bts_BaseMessage* message) const;

   virtual bool triggerInitializedCallback(void);

   virtual bool handleDoubledApp2BtsMessages(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_BaseMessage* message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);

   virtual bool handleApp2BtsMessage(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_BaseMessage* message);

   virtual bool handleIpc2BtsMessage(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BaseMessage* message);

   virtual bool setSubControlTestCommand(IN const char* testCommand, IN const unsigned int testData);

   virtual bool setSubControlTestCommand(IN const char* testCommand, IN const unsigned char* testData);

   virtual bool handleTTFisCommand(IN const unsigned char* testData);

   virtual bool handleApp2BtsTestCommand(IN const unsigned char* testData);

private:
   const BTSFunctionBlock _component; /**< component (function) block => tracing only */
   const BTSInterfaceType _stackInterface; /**< sub-component (function) block => tracing only */
   const BTSFunctionBlock _subComponent; /**< stack interface to be used => tracing only */
   IBasicControl* _basicControlIf; /**< control base interface */
   ITimerPool* _timerPool; /**< timer pool */

   // generic SMs
   IWblStartup* _wblStartup; /**< generic SM for WBL startup */
   IWblLastMode* _wblLastMode; /**< generic SM for WBL last mode */
   IWblWiFiSetupStatus* _wblWiFiSetupStatus; /**< generic SM for WBL WiFi setup status */

   // WBL specific SMs
   IWblStartupWbl* _wblStartupWbl; /**< specific SM for WBL startup */
   IWblLastModeWbl* _wblLastModeWbl; /**< specific SM for WBL last mode */
   IWblWiFiSetupStatusWbl* _wblWiFiSetupStatusWbl; /**< specific SM for WBL WiFi setup status */
   IWblObjectPathManagerWbl* _wblObjectPathManagerWbl; /**< specific SM for WBL object path manager */

   /**
    * Initialize SMs.
    */
   void initializeSMs(void);

   /**
    * Destroy SMs.
    */
   void destroySMs(void);

   /**
    * Assign SMs.
    */
   void assignSMs(void);

   /**
    * Handle doubled App2Bts message.
    * Handle any error within this function.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] message: received App2Bts message
    * @param[in] resultCode: result code
    * @param[in] statusCode: status code
    */
   void handleDoubledApp2BtsSetWblDummy(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_SetWblDummy& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsTriggerFbWblInitialized(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_TriggerFbWblInitialized& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsAllowWblCommunication(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_AllowWblCommunication& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsBlockWblCommunication(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_BlockWblCommunication& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsGetWblCommunicationStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_GetWblCommunicationStatus& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsGetWblServiceAvailability(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_GetWblServiceAvailability& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsGetWblLastIntendedMode(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_GetWblLastIntendedMode& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsGetWblSupportedRestrictions(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_GetWblSupportedRestrictions& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsGetWblWiFiSetupStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_GetWblWiFiSetupStatus& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsGetWblHealthinessIndicator(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_GetWblHealthinessIndicator& message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);

   /**
    * Handle received App2Bts message.
    * Handle any error within this function.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] deleteApp2BtsMessage: marker to delete related App2Bts message
    * @param[in] message: received App2Bts message
    */
   void handleApp2BtsSetWblDummy(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetWblDummy& message);
   void handleApp2BtsTriggerFbWblInitialized(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_TriggerFbWblInitialized& message);
   void handleApp2BtsAllowWblCommunication(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_AllowWblCommunication& message);
   void handleApp2BtsBlockWblCommunication(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_BlockWblCommunication& message);
   void handleApp2BtsGetWblCommunicationStatus(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetWblCommunicationStatus& message);
   void handleApp2BtsGetWblServiceAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetWblServiceAvailability& message);
   void handleApp2BtsGetWblLastIntendedMode(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetWblLastIntendedMode& message);
   void handleApp2BtsGetWblSupportedRestrictions(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetWblSupportedRestrictions& message);
   void handleApp2BtsGetWblWiFiSetupStatus(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetWblWiFiSetupStatus& message);
   void handleApp2BtsGetWblHealthinessIndicator(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetWblHealthinessIndicator& message);

   /**
    * Handle received Ipc2Bts message.
    * Handle any error within this function.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in,out] messageItem: message item
    * @param[in] message: received Ipc2Bts message
    */
   void handleIpc2BtsServiceAvailabilityWbl(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ServiceAvailabilityWbl& message);
   void handleIpc2BtsGetManagedObjectsWbl(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetManagedObjectsWbl& message);
   void handleIpc2BtsAddWifiSetupObjectPath(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AddWifiSetupObjectPath& message);
   void handleIpc2BtsDelWifiSetupObjectPath(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DelWifiSetupObjectPath& message);
   void handleIpc2BtsLastIntendedModeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LastIntendedModeUpdate& message);
   void handleIpc2BtsSupportedFrequenciesUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SupportedFrequenciesUpdate& message);
   void handleIpc2BtsWifiModeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_WifiModeUpdate& message);
   void handleIpc2BtsWifiPowerStateUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_WifiPowerStateUpdate& message);
   void handleIpc2BtsWifiFrequencyUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_WifiFrequencyUpdate& message);
   void handleIpc2BtsAccessPointConfigTypeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AccessPointConfigTypeUpdate& message);
   void handleIpc2BtsWifiConnectionStatusUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, INOUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_WifiConnectionStatusUpdate& message);
};

} //wbl
} //btstackif

#endif //_WBL_DBUS_IF_HANDLER_H_
