/**
 * @file SwitchBluetoothGenivi.cpp
 *
 * @par SW-Component
 * State machine for switch Bluetooth on/off
 *
 * @brief Implementation of Genivi switch Bluetooth on/off state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Genivi switch Bluetooth on/off state machine.
 */

#include "SwitchBluetoothGenivi.h"
#include "ISwitchBluetoothCallback.h"
#include "IBasicControl.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "EvolutionGeniviUtils.h"
#include "FwStringUtils.h"
#include "FwErrmemPrint.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#include <errno.h>
#include <cstring>
#include <cstdio>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/SwitchBluetoothGenivi.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace genivi {

SwitchBluetoothGenivi::SwitchBluetoothGenivi() :
_callback(0),
_controlIf(0),
_adapterDataReceived(false),
_mibDataFileName("/var/opt/bosch/dynamic/connectivity/bt_module/default_mib.txt"),
_defaultIdValue("Not Supported by BT"),
_defaultFwValue("0.0.0.p0"),
_chipIdName("ChipID"),
_buildIdName("BuildID"),
_fwVersionName("PtestFirmwareVersion"),
_assignCmd(" ::= "),
_evoVersionFileName("/etc/alps/evolution/version.txt"),
_code(0),
_geniviVersion(0),
_evoVersion(0),
_moduleIdList(),
_vendorIdList()
{
   _moduleIdList["104a"] = BTS_BT_MODULE_UGXZE_104A;
   _moduleIdList["304a"] = BTS_BT_MODULE_UGXZE_304A;
   _moduleIdList["1404"] = BTS_BT_MODULE_UGKZ2_H;
   _moduleIdList["1089"] = BTS_BT_MODULE_UGKZ2_E;
   _moduleIdList["1029"] = BTS_BT_MODULE_UGKZ2_2;
   _moduleIdList["0735"] = BTS_BT_MODULE_UGKZ7;

   _vendorIdList[BTS_BT_MODULE_UGXZE_104A] = BTS_HCI_CHIP_VENDOR_CSR;
   _vendorIdList[BTS_BT_MODULE_UGXZE_304A] = BTS_HCI_CHIP_VENDOR_CSR;
   _vendorIdList[BTS_BT_MODULE_UGKZ2_H] = BTS_HCI_CHIP_VENDOR_CSR;
   _vendorIdList[BTS_BT_MODULE_UGKZ2_E] = BTS_HCI_CHIP_VENDOR_CSR;
   _vendorIdList[BTS_BT_MODULE_UGKZ2_2] = BTS_HCI_CHIP_VENDOR_CSR;
   _vendorIdList[BTS_BT_MODULE_UGKZ7] = BTS_HCI_CHIP_VENDOR_MARVELL;
}

SwitchBluetoothGenivi::~SwitchBluetoothGenivi()
{
   _callback = 0;
   _controlIf = 0;
}

void SwitchBluetoothGenivi::reset(void)
{
   _adapterDataReceived = false;
}

void SwitchBluetoothGenivi::setCallback(IN ISwitchBluetoothCallback* callback)
{
   _callback = callback;

   FW_ERRMEM_ASSERT(0 != _callback);
}

void SwitchBluetoothGenivi::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);
}

void SwitchBluetoothGenivi::setPowered(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enabled)
{
   (void)(bts2AppMsgList);

   createSetAdapterPoweredMsg(bts2IpcMsgList, enabled);
}

void SwitchBluetoothGenivi::sendVirtualPoweredUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enabled, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   Ipc2Bts_AdapterPoweredUpdate* msg = ptrNew_Ipc2Bts_AdapterPoweredUpdate();
   if(0 != msg)
   {
      msg->setPowered(enabled);
      msg->setIpcCommonErrorCode(errorCode);
   }
   _controlIf->sendInternalIpc2BtsMessage(msg);
}

void SwitchBluetoothGenivi::getHwVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   createGetHwVersionMsg(bts2IpcMsgList);
}

ISwitchBluetoothRequest* SwitchBluetoothGenivi::getRequestIf(void)
{
   return this;
}

void SwitchBluetoothGenivi::handleAdapterProperties(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusPropertyList& properties, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   // is triggered by a signal or method return => check error code
   if(BTS_IPC_SUCCESS == errorCode)
   {
      if(true == _adapterDataReceived)
      {
         // adapter data was already received => ignore this message
      }
      else
      {
         // first reception of adapter data => process
         _adapterDataReceived = true;

         // store information
         for(size_t i = 0; i < properties.size(); i++)
         {
            const ::ccdbusif::evolution::AdapterProperty property = (::ccdbusif::evolution::AdapterProperty)properties[i].propEnum;

            switch(property)
            {
               case ::ccdbusif::evolution::ADAPTER_ADDRESS:
               {
                  setStackLocalBDAddress(properties[i].propData.getString());
                  break;
               }
               case ::ccdbusif::evolution::ADAPTER_POWERED:
               {
                  setStackPoweredMode(properties[i].propData.getBool());
                  break;
               }
               case ::ccdbusif::evolution::ADAPTER_HCIMODE:
               {
                  setStackHciMode(properties[i].propData.getBool());
                  break;
               }
               case ::ccdbusif::evolution::ADAPTER_VERSION_CODE:
               {
                  setVersionInfoCode(properties[i].propData.getUInt16());
                  break;
               }
               case ::ccdbusif::evolution::ADAPTER_VERSION_GENIVI_VERSION:
               {
                  setVersionInfoGeniviVersion(properties[i].propData.getUInt16());
                  break;
               }
               case ::ccdbusif::evolution::ADAPTER_VERSION_EVO_VERSION:
               {
                  setVersionInfoEvoVersion(properties[i].propData.getUInt16());
                  break;
               }
               default:
               {
                  break;
               }
            }
         }
      }
   }
}

void SwitchBluetoothGenivi::handleGetHwVersionConfirmation(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSStatusCode status, IN const BTSBtCoreVersion btVersion, IN const BTSManufacturerNameController manufacturerName, IN const BTSSubVersionController subVersion, IN const BTSExtensionInfo& extInfo)
{
   (void)(btVersion);
   (void)(manufacturerName);
   (void)(subVersion);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if(BTS_STATUS_CODE_SUCCESS != status)
   {
      return;
   }

   /*
    * following data has to be written to Bosch registry:
    * - "Bluetooth_HW_Version"         : "ChipID" value from "default_mib.txt"
    * - "Bluetooth_HW_LoaderVersion"   : "BuildID" value from "default_mib.txt"
    * - "Bluetooth_SW_Version"         : "PtestFirmwareVersion" value from "default_mib.txt" _or_
    * - "Bluetooth_SW_Version"         : UGKZ7 firmware version (as hex): <patch number><minor revision><major revision><release number>
    */
   ::std::string chipId;
   ::std::string buildId;
   ::std::string fwVersion;
   ::std::string fwExtInfo;
   BTSBtModuleId moduleId(BTS_BT_MODULE_UNKNOWN);
   BTSHciChipVendorId vendorId(BTS_HCI_CHIP_VENDOR_UNKNOWN);

   getDefaultMibData(chipId, buildId, fwVersion);

   moduleId = getModuleId(buildId);
   vendorId = getVendorId(moduleId);

   convertExtInfo(fwExtInfo, extInfo);

   addStackVersion(fwVersion, fwExtInfo);

   ETG_TRACE_USR3((" handleGetHwVersionConfirmation(): chipId=%s", chipId.c_str()));
   ETG_TRACE_USR3((" handleGetHwVersionConfirmation(): buildId=%s", buildId.c_str()));
   ETG_TRACE_USR3((" handleGetHwVersionConfirmation(): fwVersion=%s", fwVersion.c_str()));
   ETG_TRACE_USR3((" handleGetHwVersionConfirmation(): fwExtInfo=%s", fwExtInfo.c_str()));
   ETG_TRACE_USR3((" handleGetHwVersionConfirmation(): moduleId=%d", moduleId));
   ETG_TRACE_USR3((" handleGetHwVersionConfirmation(): vendorId=%d", vendorId));

   _callback->reportVersionInfo(bts2IpcMsgList, bts2AppMsgList, messageItem, chipId, buildId, ((0 < fwVersion.length()) ? fwVersion : fwExtInfo), vendorId, moduleId);
}

void SwitchBluetoothGenivi::handleHciModeUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool hciMode, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if(BTS_IPC_SUCCESS == errorCode)
   {
      _callback->updateHciMode(bts2IpcMsgList, bts2AppMsgList, messageItem, hciMode);
   }
}

void SwitchBluetoothGenivi::handlePoweredUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool powered, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force /*= false*/)
{
   (void)(response);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   _callback->updatePowered(bts2IpcMsgList, bts2AppMsgList, messageItem, powered, errorCode, force);
}

void SwitchBluetoothGenivi::getDefaultMibData(OUT ::std::string& chipId, OUT ::std::string& buildId, OUT ::std::string& fwVersion)
{
   FILE* file = fopen(_mibDataFileName.c_str(), "r");

   if(0 == file)
   {
      const int errNumber(errno);
      ETG_TRACE_ERR((" getDefaultMibData(): fopen of %100s failed: ERROR=%d (%s)", _mibDataFileName.c_str(), errNumber, strerror(errNumber)));
      FW_ERRMEM_ASSERT_ALWAYS();
      chipId = _defaultIdValue;
      buildId = _defaultIdValue;
      return;
   }

   bool chipIdFound = false;
   bool buildIdFound = false;
   bool fwVersionFound = false;
   const unsigned int bufferSize = 250;
   char buffer[bufferSize];

   while(((false == chipIdFound) || (false == buildIdFound) || (false == fwVersionFound)) &&
         (0 != fgets(buffer, sizeof(buffer), file)))
   {
      // remove CR LF
      char* c = buffer;
      while((*c != '\0') && (*c != '\r') && (*c != '\n'))
      {
         ++c;
      }
      *c = '\0';

      // check for comment
      if('#' == buffer[0])
      {
         continue;
      }

      if(false == buildIdFound)
      {
         buildIdFound = getDefaultMibDataItem(buildId, buffer, _buildIdName, 0);
      }
      if(false == chipIdFound)
      {
         chipIdFound = getDefaultMibDataItem(chipId, buffer, _chipIdName, 2);
      }
      if(false == fwVersionFound)
      {
         fwVersionFound = getDefaultMibDataItem(fwVersion, buffer, _fwVersionName, 0);
      }
   }

   (void)fclose(file);

   if(true == chipId.empty())
   {
      chipId = _defaultIdValue;
   }
   if(true == buildId.empty())
   {
      buildId = _defaultIdValue;
   }
}

bool SwitchBluetoothGenivi::getDefaultMibDataItem(OUT ::std::string& item, IN char* buffer, IN const ::std::string& key, IN const size_t offset)
{
   if(NULL == buffer)
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return false;
   }

   if(0 == key.length())
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return false;
   }

   char* subString = strstr(buffer, key.c_str());
   if(subString == buffer)
   {
      // key found at start of string
      subString = strstr(buffer, _assignCmd.c_str());
      if(NULL != subString)
      {
         // assignment command is available
         subString += _assignCmd.length();
         if(offset < strlen(subString))
         {
            subString += offset;
            item = subString;
         }
      }
   }

   if(0 < item.length())
   {
      return true;
   }
   else
   {
      return false;
   }
}

void SwitchBluetoothGenivi::convertExtInfo(OUT ::std::string& fwExtInfo, IN const BTSExtensionInfo& extInfo)
{
   // fwExtInfo format (as string):   <minor revision>.<major revision>.<release number>.p<patch number>
   // extInfo format (as byte array): <release number> <major revision> <minor revision> <patch number>

   if(4 > extInfo.size())
   {
      // given data is too small => different module than UGKZ7 is used
      fwExtInfo = _defaultFwValue;
      return;
   }

   const unsigned int bufferSize = 1 + strlen("255.255.255.p255");
   char buffer[bufferSize];
   snprintf(buffer, sizeof(buffer), "%u.%u.%u.p%u", extInfo[2], extInfo[1], extInfo[0], extInfo[3]);
   fwExtInfo = buffer;
}

void SwitchBluetoothGenivi::addStackVersion(OUT ::std::string& fwVersion, OUT ::std::string& fwExtInfo) const
{
   // _code
   // _geniviVersion
   const unsigned int bufferSize = 1 + strlen(" - 0601_C411 (LXC)");
   char buffer[bufferSize];

   if(true == isLxcVersion())
   {
      snprintf(buffer, sizeof(buffer), " - %04X_%04X (LXC)", _code, _geniviVersion);
   }
   else
   {
      snprintf(buffer, sizeof(buffer), " - %04X_%04X", _code, _geniviVersion);
   }

   if(0 < fwVersion.length())
   {
      fwVersion += buffer;
   }

   if(0 < fwExtInfo.length())
   {
      fwExtInfo += buffer;
   }
}

bool SwitchBluetoothGenivi::isLxcVersion(void) const
{
   FILE* file(fopen(_evoVersionFileName.c_str(), "r"));

   if(0 == file)
   {
      const int errNumber(errno);
      ETG_TRACE_ERRMEM((" #CONN: BtStackIf: isLxcVersion(): fopen of %100s failed: ERROR=%d (%s)", _evoVersionFileName.c_str(), errNumber, strerror(errNumber)));
      FW_ERRMEM_ASSERT_ALWAYS();
      return false;
   }

   bool isLxc(false);
   char buffer[51] = { 0 };

   if(0 == fgets(buffer, sizeof(buffer), file))
   {
      const int errNumber(errno);
      ETG_TRACE_ERRMEM((" #CONN: BtStackIf: isLxcVersion(): fgets of %100s failed: ERROR=%d (%s)", _evoVersionFileName.c_str(), errNumber, strerror(errNumber)));
      FW_ERRMEM_ASSERT_ALWAYS();
   }
   else
   {
      ::std::string versionString(buffer);

      isLxc = (::std::string::npos != versionString.find("LXC"));
   }

   (void)fclose(file);

   return isLxc;
}

BTSBtModuleId SwitchBluetoothGenivi::getModuleId(IN const ::std::string& buildId) const
{
   BTSBtModuleId moduleId(BTS_BT_MODULE_UNKNOWN);

   ::std::string working(buildId);
   ::fw::convertString2LowerCase(working);

   ::std::map< ::std::string, BTSBtModuleId >::const_iterator it = _moduleIdList.find(working);

   if(_moduleIdList.end() != it)
   {
      moduleId = it->second;
   }

   return moduleId;
}

BTSHciChipVendorId SwitchBluetoothGenivi::getVendorId(IN const BTSBtModuleId moduleId) const
{
   BTSHciChipVendorId vendorId(BTS_HCI_CHIP_VENDOR_UNKNOWN);

   ::std::map< BTSBtModuleId, BTSHciChipVendorId >::const_iterator it = _vendorIdList.find(moduleId);

   if(_vendorIdList.end() != it)
   {
      vendorId = it->second;
   }

   return vendorId;
}

void SwitchBluetoothGenivi::setStackLocalBDAddress(IN const BTSBDAddress& address)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if(false == address.empty())
   {
      BTSBDAddress tmpAddress;
      convertBdAddress2InternalValue(tmpAddress, address);

      _callback->setStackLocalBDAddress(tmpAddress);
   }
}

void SwitchBluetoothGenivi::setStackPoweredMode(IN const bool enable)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   _callback->setStackPoweredMode(enable);
}

void SwitchBluetoothGenivi::setStackHciMode(IN const bool enable)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   _callback->setStackHciMode(enable);
}

void SwitchBluetoothGenivi::setVersionInfoCode(IN const unsigned short int code)
{
   _code = code;
}

void SwitchBluetoothGenivi::setVersionInfoGeniviVersion(IN const unsigned short int geniviVersion)
{
   _geniviVersion = geniviVersion;
}

void SwitchBluetoothGenivi::setVersionInfoEvoVersion(IN const unsigned short int evoVersion)
{
   _evoVersion = evoVersion;
}

void SwitchBluetoothGenivi::createSetAdapterPoweredMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const bool enabled) const
{
   Bts2Ipc_SetAdapterPowered* msg = ptrNew_Bts2Ipc_SetAdapterPowered();
   if(0 != msg)
   {
      msg->setPowered(enabled);
      msg->setResponseMessageFlag(true);
      bts2IpcMsgList.push_back(msg);
   }
}

void SwitchBluetoothGenivi::createGetHwVersionMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList) const
{
   Bts2Ipc_BtApplGetHwVersionReq* msg = ptrNew_Bts2Ipc_BtApplGetHwVersionReq();
   if(0 != msg)
   {
      msg->setResponseMessageFlag(true);
      bts2IpcMsgList.push_back(msg);
   }
}

} //genivi
} //btstackif
