/**
 * @file StartupGenivi.cpp
 *
 * @par SW-Component
 * State machine for startup
 *
 * @brief Implementation of Genivi startup state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Genivi startup state machine.
 */

#include "StartupGenivi.h"
#include "IStartupCallback.h"
#include "IBasicControl.h"
#include "ITimerPool.h"
#include "IHfpDeviceCapabilitiesGenivi.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "FwErrmemPrint.h"
#include "FwUtils.h"

#include "cc_dbus_if/EvolutionGeniviDbusParser.h"
#include "cc_dbus_if/EvolutionGeniviUtility.h"
#include "cc_dbus_if/EvolutionGeniviDbusTypes.h"

#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/StartupGenivi.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace genivi {

StartupGenivi::StartupGenivi() :
_callback(0),
_controlIf(0),
_timerPoolIf(0),
_capability(BTS_AGENT_CAPABILITY_DISPLAY_YES_NO),
_hfpDeviceCapabilitiesGeniviIf(0),
_getAllReceived(false),
_getAllTimerUsed(false),
_getAllSent(false),
_timerGetAll(),
_initReceived(false),
_initTimerUsed(false),
_timerInit(),
_registerAgentSent(false),
_registerAgentTimer(),
_nmbRegisterAgentRequests(0),
_serviceGroupGeneral(),
_serviceStatusDisabledBtStackComm(),
_serviceStatusObjectManagerSystem(),
_serviceStatusObjectManagerSession(),
_serviceStatusIntrospectable(),
_serviceStatusPropertiesGetAll(),
_serviceStatusGenInitialization(),
_serviceStatusHliInitialization(),
_serviceStatusAdapter(),
_serviceStatusAgent(),
_serviceStatusAgentManager(),
_serviceStatusDM(),
_serviceStatusTrace(),
_serviceStatusGeniviTrace(),
_serviceStatusDID(),
_serviceStatusObexClient(),
_testSimulateDbusStartupError(false),
_testSimulateEvolutionStartupError(false)
{
   _serviceGroupGeneral.disableTimer();
   _serviceGroupGeneral.addServiceInfo("");
   _serviceGroupGeneral.setName("GroupGeneral");

   _serviceStatusDisabledBtStackComm.setName("DisabledBtStackComm");
   _serviceStatusObjectManagerSystem.setName("ObjectManagerSystem");
   _serviceStatusObjectManagerSession.setName("ObjectManagerSession");
   _serviceStatusIntrospectable.setName("Introspectable");
   _serviceStatusPropertiesGetAll.setName("PropertiesGetAll");
   _serviceStatusGenInitialization.setName("GenInitialization");
   _serviceStatusHliInitialization.setName("HliInitialization");
   _serviceStatusAdapter.setName("Adapter");
   _serviceStatusAgent.setName("Agent");
   _serviceStatusAgentManager.setName("AgentManager");
   _serviceStatusDM.setName("DM");
   _serviceStatusTrace.setName("Trace");
   _serviceStatusGeniviTrace.setName("GeniviTrace");
   _serviceStatusDID.setName("DID");
   _serviceStatusObexClient.setName("ObexClient");

   _serviceStatusDisabledBtStackComm.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_DISABLED_BT_STACK_COMM);
   _serviceStatusObjectManagerSystem.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SYSTEM);
   _serviceStatusObjectManagerSession.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SESSION);
   _serviceStatusIntrospectable.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_INTROSPECTABLE);
   _serviceStatusPropertiesGetAll.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_PROPERTIES_GET_ALL);
   _serviceStatusGenInitialization.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_GEN_INITIALIZATION);
   _serviceStatusHliInitialization.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_HLI_INITIALIZATION);
   _serviceStatusAdapter.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_ADAPTER);
   _serviceStatusAgent.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_AGENT);
   _serviceStatusAgentManager.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_AGENT_MANAGER);
   _serviceStatusDM.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_DM);
   _serviceStatusTrace.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_TRACE);
   _serviceStatusGeniviTrace.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_GENIVI_TRACE);
   _serviceStatusDID.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_DID);
   _serviceStatusObexClient.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_OBEX_CLIENT);
}

StartupGenivi::~StartupGenivi()
{
   _callback = 0;
   _controlIf = 0;
   _timerPoolIf = 0;
   _hfpDeviceCapabilitiesGeniviIf = 0;
}

void StartupGenivi::reset(void)
{
   _getAllReceived = false;
   _getAllTimerUsed = false;
   _getAllSent = false;
   stopTimer(_timerGetAll);
   _initReceived = false;
   _initTimerUsed = false;
   stopTimer(_timerInit);
   _registerAgentSent = false;
   stopTimer(_registerAgentTimer);
   _nmbRegisterAgentRequests = 0;

   if(true == _serviceStatusDisabledBtStackComm.getEnabled())
   {
      _serviceStatusDisabledBtStackComm.setAvailability(BTS_DBUS_SERVICE_NOT_AVAILABLE);
   }
   if(true == _serviceStatusObjectManagerSystem.getEnabled())
   {
      _serviceStatusObjectManagerSystem.setAvailability(BTS_DBUS_SERVICE_NOT_AVAILABLE);
   }
   if(true == _serviceStatusObjectManagerSession.getEnabled())
   {
      _serviceStatusObjectManagerSession.setAvailability(BTS_DBUS_SERVICE_NOT_AVAILABLE);
   }
   if(true == _serviceStatusIntrospectable.getEnabled())
   {
      // currently disabled; implement if necessary
   }
   if(true == _serviceStatusPropertiesGetAll.getEnabled())
   {
      ::ccdbusif::evolution::Interface evoInterface(::ccdbusif::evolution::IF_ADAPTER);
      ::ccdbusif::evolution::EvolutionGeniviDbusParser evoParser;
      _serviceStatusPropertiesGetAll.setAvailability(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface), BTS_DBUS_SERVICE_AVAILABLE); // Adapter, Trace
   }
   if(true == _serviceStatusGenInitialization.getEnabled())
   {
      _serviceStatusGenInitialization.setAvailability(BTS_DBUS_SERVICE_NOT_AVAILABLE);
   }
   if(true == _serviceStatusHliInitialization.getEnabled())
   {
      _serviceStatusHliInitialization.setAvailability(BTS_DBUS_SERVICE_NOT_AVAILABLE);
   }
   if(true == _serviceStatusAdapter.getEnabled())
   {
      _serviceStatusAdapter.setAvailability(BTS_DBUS_SERVICE_NOT_AVAILABLE);
   }
   if(true == _serviceStatusAgent.getEnabled())
   {
      _serviceStatusAgent.setAvailability(BTS_DBUS_SERVICE_NOT_AVAILABLE);
   }
   if(true == _serviceStatusAgentManager.getEnabled())
   {
      _serviceStatusAgentManager.setAvailability(BTS_DBUS_SERVICE_NOT_AVAILABLE);
   }
   if(true == _serviceStatusDM.getEnabled())
   {
      _serviceStatusDM.setAvailability(BTS_DBUS_SERVICE_NOT_AVAILABLE);
   }
   if(true == _serviceStatusTrace.getEnabled())
   {
      _serviceStatusTrace.setAvailability(BTS_DBUS_SERVICE_NOT_AVAILABLE);
   }
   if(true == _serviceStatusGeniviTrace.getEnabled())
   {
      _serviceStatusGeniviTrace.setAvailability(BTS_DBUS_SERVICE_NOT_AVAILABLE);
   }
   if(true == _serviceStatusDID.getEnabled())
   {
      _serviceStatusDID.setAvailability(BTS_DBUS_SERVICE_NOT_AVAILABLE);
   }
   if(true == _serviceStatusObexClient.getEnabled())
   {
      _serviceStatusObexClient.setAvailability(BTS_DBUS_SERVICE_NOT_AVAILABLE);
   }

   _serviceGroupGeneral.checkAvailability();

   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   _callback->startupSequenceStarted(_serviceGroupGeneral.getAvailability());
}

void StartupGenivi::setCallback(IN IStartupCallback* callback)
{
   _callback = callback;

   FW_ERRMEM_ASSERT(0 != _callback);
}

void StartupGenivi::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   _serviceStatusDisabledBtStackComm.setIpc2BtsSendIf(_controlIf);
   _serviceStatusObjectManagerSystem.setIpc2BtsSendIf(_controlIf);
   _serviceStatusObjectManagerSession.setIpc2BtsSendIf(_controlIf);
   _serviceStatusIntrospectable.setIpc2BtsSendIf(_controlIf);
   _serviceStatusPropertiesGetAll.setIpc2BtsSendIf(_controlIf);
   _serviceStatusGenInitialization.setIpc2BtsSendIf(_controlIf);
   _serviceStatusHliInitialization.setIpc2BtsSendIf(_controlIf);
   _serviceStatusAdapter.setIpc2BtsSendIf(_controlIf);
   _serviceStatusAgent.setIpc2BtsSendIf(_controlIf);
   _serviceStatusAgentManager.setIpc2BtsSendIf(_controlIf);
   _serviceStatusDM.setIpc2BtsSendIf(_controlIf);
   _serviceStatusTrace.setIpc2BtsSendIf(_controlIf);
   _serviceStatusGeniviTrace.setIpc2BtsSendIf(_controlIf);
   _serviceStatusDID.setIpc2BtsSendIf(_controlIf);
   _serviceStatusObexClient.setIpc2BtsSendIf(_controlIf);
}

void StartupGenivi::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_ERRMEM_ASSERT(0 != _timerPoolIf);
}

void StartupGenivi::setAgentCapability(IN const BTSAgentCapability capability)
{
   if(BTS_AGENT_CAPABILITY_LAST <= capability)
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   _capability = capability;
}

void StartupGenivi::checkAvailableServices(void)
{
   // ############################################################################################
   // collect data for analysis --- start
   BTSBitfield<unsigned int, BTSGenDbusServiceInterface, BTS_GEN_DBUS_SERVICE_LAST> enabledServices;
   BTSBitfield<unsigned int, BTSGenDbusServiceInterface, BTS_GEN_DBUS_SERVICE_LAST> availableServices;

   if(true == _serviceStatusObjectManagerSystem.getEnabled())
   {
      enabledServices.setBit(BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SYSTEM);
      if(BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusObjectManagerSystem.getAvailability())
      {
         availableServices.setBit(BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SYSTEM);
      }
      else
      {
         ETG_TRACE_FATAL((" ERROR: OBJECT_MANAGER_SYSTEM missing"));
      }
   }

   if(true == _serviceStatusObjectManagerSession.getEnabled())
   {
      enabledServices.setBit(BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SESSION);
      if(BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusObjectManagerSession.getAvailability())
      {
         availableServices.setBit(BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SESSION);
      }
      else
      {
         ETG_TRACE_FATAL((" ERROR: OBJECT_MANAGER_SESSION missing"));
      }
   }

   if(true == _serviceStatusGenInitialization.getEnabled())
   {
      enabledServices.setBit(BTS_GEN_DBUS_SERVICE_GEN_INITIALIZATION);
      if(BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusGenInitialization.getAvailability())
      {
         availableServices.setBit(BTS_GEN_DBUS_SERVICE_GEN_INITIALIZATION);
      }
      else
      {
         ETG_TRACE_FATAL((" ERROR: GEN_INITIALIZATION missing"));
      }
   }

   if(true == _serviceStatusHliInitialization.getEnabled())
   {
      enabledServices.setBit(BTS_GEN_DBUS_SERVICE_HLI_INITIALIZATION);
      if(BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusHliInitialization.getAvailability())
      {
         availableServices.setBit(BTS_GEN_DBUS_SERVICE_HLI_INITIALIZATION);
      }
      else
      {
         ETG_TRACE_FATAL((" ERROR: HLI_INITIALIZATION missing"));
      }
   }

   if(true == _serviceStatusAdapter.getEnabled())
   {
      enabledServices.setBit(BTS_GEN_DBUS_SERVICE_ADAPTER);
      if(BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusAdapter.getAvailability())
      {
         availableServices.setBit(BTS_GEN_DBUS_SERVICE_ADAPTER);
      }
      else
      {
         ETG_TRACE_FATAL((" ERROR: ADAPTER missing"));
      }
   }

   if(true == _serviceStatusAgentManager.getEnabled())
   {
      enabledServices.setBit(BTS_GEN_DBUS_SERVICE_AGENT_MANAGER);
      if(BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusAgentManager.getAvailability())
      {
         availableServices.setBit(BTS_GEN_DBUS_SERVICE_AGENT_MANAGER);
      }
      else
      {
         ETG_TRACE_FATAL((" ERROR: AGENT_MANAGER missing"));
      }
   }

   if(true == _serviceStatusDM.getEnabled())
   {
      enabledServices.setBit(BTS_GEN_DBUS_SERVICE_DM);
      if(BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusDM.getAvailability())
      {
         availableServices.setBit(BTS_GEN_DBUS_SERVICE_DM);
      }
      else
      {
         ETG_TRACE_FATAL((" ERROR: DM missing"));
      }
   }

   if(true == _serviceStatusTrace.getEnabled())
   {
      enabledServices.setBit(BTS_GEN_DBUS_SERVICE_TRACE);
      if(BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusTrace.getAvailability())
      {
         availableServices.setBit(BTS_GEN_DBUS_SERVICE_TRACE);
      }
      else
      {
         ETG_TRACE_FATAL((" ERROR: TRACE missing"));
      }
   }

   if(true == _serviceStatusGeniviTrace.getEnabled())
   {
      enabledServices.setBit(BTS_GEN_DBUS_SERVICE_GENIVI_TRACE);
      if(BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusGeniviTrace.getAvailability())
      {
         availableServices.setBit(BTS_GEN_DBUS_SERVICE_GENIVI_TRACE);
      }
      else
      {
         ETG_TRACE_FATAL((" ERROR: GENIVI_TRACE missing"));
      }
   }

   if(true == _serviceStatusDID.getEnabled())
   {
      enabledServices.setBit(BTS_GEN_DBUS_SERVICE_DID);
      if(BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusDID.getAvailability())
      {
         availableServices.setBit(BTS_GEN_DBUS_SERVICE_DID);
      }
      else
      {
         ETG_TRACE_FATAL((" ERROR: DID missing"));
      }
   }

   if(true == _serviceStatusObexClient.getEnabled())
   {
      enabledServices.setBit(BTS_GEN_DBUS_SERVICE_OBEX_CLIENT);
      if(BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusObexClient.getAvailability())
      {
         availableServices.setBit(BTS_GEN_DBUS_SERVICE_OBEX_CLIENT);
      }
      else
      {
         ETG_TRACE_FATAL((" ERROR: OBEX_CLIENT missing"));
      }
   }

   FW_ERRMEM_ASSERT(0 != _hfpDeviceCapabilitiesGeniviIf);
   if(0 != _hfpDeviceCapabilitiesGeniviIf)
   {
      IHfpDeviceCapabilitiesGenivi& geniviHfpDeviceCapabilities = *_hfpDeviceCapabilitiesGeniviIf;

      if(true == geniviHfpDeviceCapabilities.getEnabled())
      {
         enabledServices.setBit(BTS_GEN_DBUS_SERVICE_HFP_MANAGER);
         if(BTS_DBUS_SERVICE_AVAILABLE == geniviHfpDeviceCapabilities.getManagerAvailability())
         {
            availableServices.setBit(BTS_GEN_DBUS_SERVICE_HFP_MANAGER);
         }
         else
         {
            ETG_TRACE_FATAL((" ERROR: HFP_MANAGER missing"));
         }
      }
   }

   if(true == _serviceStatusIntrospectable.getEnabled())
   {
      enabledServices.setBit(BTS_GEN_DBUS_SERVICE_INTROSPECTABLE);
      if(BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusIntrospectable.getGroupAvailability())
      {
         availableServices.setBit(BTS_GEN_DBUS_SERVICE_INTROSPECTABLE);
      }
      else
      {
         ETG_TRACE_FATAL((" ERROR: INTROSPECTABLE missing"));
      }
   }

   if(true == _serviceStatusPropertiesGetAll.getEnabled())
   {
      enabledServices.setBit(BTS_GEN_DBUS_SERVICE_PROPERTIES_GET_ALL);
      if(BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusPropertiesGetAll.getGroupAvailability())
      {
         availableServices.setBit(BTS_GEN_DBUS_SERVICE_PROPERTIES_GET_ALL);
      }
      else
      {
         ETG_TRACE_FATAL((" ERROR: PROPERTIES_GET_ALL missing"));
      }
   }

   if(true == _serviceStatusAgent.getEnabled())
   {
      enabledServices.setBit(BTS_GEN_DBUS_SERVICE_AGENT);
      if(BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusAgent.getAvailability())
      {
         availableServices.setBit(BTS_GEN_DBUS_SERVICE_AGENT);
      }
      else
      {
         ETG_TRACE_FATAL((" ERROR: AGENT missing"));
      }
   }

   ETG_TRACE_ERRMEM((" #CONN: ERROR: ASF DBUS proxies failed: enabled=0x%08X available=0x%08X", enabledServices.getData(), availableServices.getData()));

   // collect data for analysis --- end
   // ############################################################################################
}

void StartupGenivi::getIntrospections(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   if(true == _serviceStatusIntrospectable.getEnabled())
   {
      createGetIntrospectionConnectionMsg(bts2IpcMsgList);
   }
}

void StartupGenivi::setHfpDeviceCapabilitiesGeniviIf(IN IHfpDeviceCapabilitiesGenivi* hfpDeviceCapabilitiesGeniviIf)
{
   _hfpDeviceCapabilitiesGeniviIf = hfpDeviceCapabilitiesGeniviIf;

   FW_ERRMEM_ASSERT(0 != _hfpDeviceCapabilitiesGeniviIf);
}

IStartupRequest* StartupGenivi::getRequestIf(void)
{
   return this;
}

void StartupGenivi::addToServiceGroup(IN const BTSUserMode userMode, IN DbusServiceStatus& status, IN const bool enabled)
{
   if((BTS_USER_MODE_CONNECTION == userMode) ||
      (BTS_USER_MODE_CONN_WO_AGENT == userMode) ||
      (BTS_USER_MODE_ALL_WI_AGENT == userMode) ||
      (BTS_USER_MODE_ALL_WO_AGENT == userMode))
   {
      if(true == enabled)
      {
         _serviceGroupGeneral.addServiceStatus(&status);
      }
   }
}

void StartupGenivi::addInterfaces(OUT ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSUserMode userMode)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   BTSDbusInterfaceItem dbusItem;
   ::ccdbusif::evolution::Interface evoInterface;
   ::ccdbusif::evolution::EvolutionGeniviDbusParser evoParser;

   if(BTS_USER_MODE_BT_DISABLED == userMode)
   {
      _serviceGroupGeneral.setEnabled(true);
      _serviceGroupGeneral.addServiceStatus(&_serviceStatusDisabledBtStackComm);

      _serviceStatusDisabledBtStackComm.setEnabled(true);

      _serviceStatusDisabledBtStackComm.addServiceInfo("DisabledBtStackComm");

      _serviceStatusDisabledBtStackComm.setAvailability(BTS_DBUS_SERVICE_WAITING);

      // create event for service not available to continue with normal handling
      _controlIf->createDbusServiceAvailabilityMessage((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_DISABLED_BT_STACK_COMM, BTS_DBUS_SERVICE_NOT_AVAILABLE);
   }

   if((BTS_USER_MODE_CONNECTION == userMode) ||
      (BTS_USER_MODE_CONN_WO_AGENT == userMode) ||
      (BTS_USER_MODE_ALL_WI_AGENT == userMode) ||
      (BTS_USER_MODE_ALL_WO_AGENT == userMode))
   {
      _serviceGroupGeneral.setEnabled(true);
      _serviceGroupGeneral.addServiceStatus(&_serviceStatusObjectManagerSystem);
      _serviceGroupGeneral.addServiceStatus(&_serviceStatusObjectManagerSession);
      _serviceGroupGeneral.addServiceStatus(&_serviceStatusGenInitialization);
      _serviceGroupGeneral.addServiceStatus(&_serviceStatusHliInitialization);
      _serviceGroupGeneral.addServiceStatus(&_serviceStatusAdapter);
      _serviceGroupGeneral.addServiceStatus(&_serviceStatusAgentManager);
      _serviceGroupGeneral.addServiceStatus(&_serviceStatusDM);
      _serviceGroupGeneral.addServiceStatus(&_serviceStatusTrace);
      _serviceGroupGeneral.addServiceStatus(&_serviceStatusGeniviTrace);
      _serviceGroupGeneral.addServiceStatus(&_serviceStatusDID);
      _serviceGroupGeneral.addServiceStatus(&_serviceStatusObexClient);

      _serviceStatusObjectManagerSystem.setEnabled(true);
      _serviceStatusObjectManagerSession.setEnabled(true);
      _serviceStatusIntrospectable.setEnabled(false); // HINT: set to false if not needed
      _serviceStatusPropertiesGetAll.setEnabled(true);
      _serviceStatusGenInitialization.setEnabled(true);
      _serviceStatusHliInitialization.setEnabled(true);
      _serviceStatusAdapter.setEnabled(true);
      _serviceStatusAgentManager.setEnabled(true);
      _serviceStatusDM.setEnabled(true);
      _serviceStatusTrace.setEnabled(true);
      _serviceStatusGeniviTrace.setEnabled(true);
      _serviceStatusDID.setEnabled(true);
      _serviceStatusObexClient.setEnabled(true);

      _serviceStatusObjectManagerSystem.addServiceInfo(evoParser.getRootObjectPath());
      _serviceStatusObjectManagerSession.addServiceInfo(evoParser.getRootObjectPath());
      _serviceStatusGenInitialization.addServiceInfo("<gen_initialization>");
      _serviceStatusHliInitialization.addServiceInfo("<hli_initialization>");
      _serviceStatusAdapter.addServiceInfo(evoParser.getInterface2ObjectPath(::ccdbusif::evolution::IF_ADAPTER));
      _serviceStatusAgentManager.addServiceInfo(evoParser.getInterface2ObjectPath(::ccdbusif::evolution::IF_AGENT_MANAGER));
      _serviceStatusDM.addServiceInfo(evoParser.getInterface2ObjectPath(::ccdbusif::evolution::IF_DM));
      _serviceStatusTrace.addServiceInfo(evoParser.getInterface2ObjectPath(::ccdbusif::evolution::IF_TRACE));
      _serviceStatusGeniviTrace.addServiceInfo(evoParser.getInterface2ObjectPath(::ccdbusif::evolution::IF_GENIVI_TRACE));
      _serviceStatusDID.addServiceInfo(evoParser.getInterface2ObjectPath(::ccdbusif::evolution::IF_DID));
      _serviceStatusObexClient.addServiceInfo(evoParser.getInterface2ObjectPath(::ccdbusif::evolution::IF_CLIENT));

      _serviceStatusObjectManagerSystem.setAvailability(BTS_DBUS_SERVICE_WAITING);
      _serviceStatusObjectManagerSession.setAvailability(BTS_DBUS_SERVICE_WAITING);
      _serviceStatusGenInitialization.setAvailability(BTS_DBUS_SERVICE_WAITING);
      _serviceStatusHliInitialization.setAvailability(BTS_DBUS_SERVICE_WAITING);
      _serviceStatusAdapter.setAvailability(BTS_DBUS_SERVICE_WAITING);
      _serviceStatusAgentManager.setAvailability(BTS_DBUS_SERVICE_WAITING);
      _serviceStatusDM.setAvailability(BTS_DBUS_SERVICE_WAITING);
      _serviceStatusTrace.setAvailability(BTS_DBUS_SERVICE_WAITING);
      _serviceStatusGeniviTrace.setAvailability(BTS_DBUS_SERVICE_WAITING);
      _serviceStatusDID.setAvailability(BTS_DBUS_SERVICE_WAITING);
      _serviceStatusObexClient.setAvailability(BTS_DBUS_SERVICE_WAITING);

      dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SYSTEM;
      dbusItem.busType = (BTSCommonEnumClass)evoParser.getInterface2BusType(::ccdbusif::evolution::IF_ADAPTER);
      dbusItem.busName = evoParser.getInterface2BusName(::ccdbusif::evolution::IF_ADAPTER);
      dbusItem.objPath = evoParser.getRootObjectPath();
      dbusInterfaces.push_back(dbusItem);

      dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SESSION;
      dbusItem.busType = (BTSCommonEnumClass)evoParser.getInterface2BusType(::ccdbusif::evolution::IF_CLIENT);
      dbusItem.busName = evoParser.getInterface2BusName(::ccdbusif::evolution::IF_CLIENT);
      dbusItem.objPath = evoParser.getRootObjectPath();
      dbusInterfaces.push_back(dbusItem);

      dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_ADAPTER;
      dbusInterfaces.push_back(dbusItem);
      dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_AGENT_MANAGER;
      dbusInterfaces.push_back(dbusItem);
      dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_DM;
      dbusInterfaces.push_back(dbusItem);
      dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_TRACE;
      dbusInterfaces.push_back(dbusItem);
      dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_GENIVI_TRACE;
      dbusInterfaces.push_back(dbusItem);
      dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_DID;
      dbusInterfaces.push_back(dbusItem);
      dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_OBEX_CLIENT;
      dbusInterfaces.push_back(dbusItem);

      // initial dynamic services/interfaces
      if(true == _serviceStatusIntrospectable.getEnabled())
      {
         _serviceGroupGeneral.addServiceStatus(&_serviceStatusIntrospectable);

         evoInterface = ::ccdbusif::evolution::IF_ADAPTER;
         _serviceStatusIntrospectable.addServiceInfo(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface)); // Adapter, Trace
         evoInterface = ::ccdbusif::evolution::IF_AGENT_MANAGER;
         _serviceStatusIntrospectable.addServiceInfo(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface)); // AgentManager
         evoInterface = ::ccdbusif::evolution::IF_DM;
         _serviceStatusIntrospectable.addServiceInfo(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface)); // DM, CTN, OPPC, OPPS, SPP, DID, DUN, PAN
         evoInterface = ::ccdbusif::evolution::IF_CLIENT;
         _serviceStatusIntrospectable.addServiceInfo(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface)); // Client
         evoInterface = ::ccdbusif::evolution::IF_ADAPTER;
         _serviceStatusIntrospectable.addServiceInfo(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getRootObjectPath()); // root
         evoInterface = ::ccdbusif::evolution::IF_CLIENT;
         _serviceStatusIntrospectable.addServiceInfo(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getRootObjectPath()); // root

         evoInterface = ::ccdbusif::evolution::IF_ADAPTER;
         _serviceStatusIntrospectable.setAvailability(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface), BTS_DBUS_SERVICE_WAITING); // Adapter, Trace
         evoInterface = ::ccdbusif::evolution::IF_AGENT_MANAGER;
         _serviceStatusIntrospectable.setAvailability(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface), BTS_DBUS_SERVICE_WAITING); // AgentManager
         evoInterface = ::ccdbusif::evolution::IF_DM;
         _serviceStatusIntrospectable.setAvailability(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface), BTS_DBUS_SERVICE_WAITING); // DM, CTN, OPPC, OPPS, SPP, DID, DUN, PAN
         evoInterface = ::ccdbusif::evolution::IF_CLIENT;
         _serviceStatusIntrospectable.setAvailability(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface), BTS_DBUS_SERVICE_WAITING); // Client
         evoInterface = ::ccdbusif::evolution::IF_ADAPTER;
         _serviceStatusIntrospectable.setAvailability(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getRootObjectPath(), BTS_DBUS_SERVICE_WAITING); // root
         evoInterface = ::ccdbusif::evolution::IF_CLIENT;
         _serviceStatusIntrospectable.setAvailability(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getRootObjectPath(), BTS_DBUS_SERVICE_WAITING); // root

         evoInterface = ::ccdbusif::evolution::IF_ADAPTER;
         dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_INTROSPECTABLE;
         dbusItem.busType = (BTSCommonEnumClass)evoParser.getInterface2BusType(evoInterface);
         dbusItem.busName = evoParser.getInterface2BusName(evoInterface);
         dbusItem.objPath = evoParser.getInterface2ObjectPath(evoInterface);
         dbusInterfaces.push_back(dbusItem);

         evoInterface = ::ccdbusif::evolution::IF_AGENT_MANAGER;
         dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_INTROSPECTABLE;
         dbusItem.busType = (BTSCommonEnumClass)evoParser.getInterface2BusType(evoInterface);
         dbusItem.busName = evoParser.getInterface2BusName(evoInterface);
         dbusItem.objPath = evoParser.getInterface2ObjectPath(evoInterface);
         dbusInterfaces.push_back(dbusItem);

         evoInterface = ::ccdbusif::evolution::IF_DM;
         dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_INTROSPECTABLE;
         dbusItem.busType = (BTSCommonEnumClass)evoParser.getInterface2BusType(evoInterface);
         dbusItem.busName = evoParser.getInterface2BusName(evoInterface);
         dbusItem.objPath = evoParser.getInterface2ObjectPath(evoInterface);
         dbusInterfaces.push_back(dbusItem);

         evoInterface = ::ccdbusif::evolution::IF_CLIENT;
         dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_INTROSPECTABLE;
         dbusItem.busType = (BTSCommonEnumClass)evoParser.getInterface2BusType(evoInterface);
         dbusItem.busName = evoParser.getInterface2BusName(evoInterface);
         dbusItem.objPath = evoParser.getInterface2ObjectPath(evoInterface);
         dbusInterfaces.push_back(dbusItem);

         evoInterface = ::ccdbusif::evolution::IF_ADAPTER;
         dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_INTROSPECTABLE;
         dbusItem.busType = (BTSCommonEnumClass)evoParser.getInterface2BusType(evoInterface);
         dbusItem.busName = evoParser.getInterface2BusName(evoInterface);
         dbusItem.objPath = evoParser.getRootObjectPath();
         dbusInterfaces.push_back(dbusItem);

         evoInterface = ::ccdbusif::evolution::IF_CLIENT;
         dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_INTROSPECTABLE;
         dbusItem.busType = (BTSCommonEnumClass)evoParser.getInterface2BusType(evoInterface);
         dbusItem.busName = evoParser.getInterface2BusName(evoInterface);
         dbusItem.objPath = evoParser.getRootObjectPath();
         dbusInterfaces.push_back(dbusItem);
      }

      _serviceGroupGeneral.addServiceStatus(&_serviceStatusPropertiesGetAll);

      evoInterface = ::ccdbusif::evolution::IF_ADAPTER;
      _serviceStatusPropertiesGetAll.addServiceInfo(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface)); // Adapter1, Trace
      // evoInterface = ::ccdbusif::evolution::IF_AGENT_MANAGER;
      // _serviceStatusPropertiesGetAll.addServiceInfo(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface)); --- AgentManager has no properties
      // evoInterface = ::ccdbusif::evolution::IF_DM;
      // _serviceStatusPropertiesGetAll.addServiceInfo(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface)); --- DM, CTN, OPPC, OPPS, SPP, DID, DUN, PAN have no properties
      // evoInterface = ::ccdbusif::evolution::IF_CLIENT;
      // _serviceStatusPropertiesGetAll.addServiceInfo(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface)); --- Client has no properties

      // availability is set to AVAILABLE due to issue in ASF
      evoInterface = ::ccdbusif::evolution::IF_ADAPTER;
      _serviceStatusPropertiesGetAll.setAvailability(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface), BTS_DBUS_SERVICE_AVAILABLE); // Adapter, Trace
      // evoInterface = ::ccdbusif::evolution::IF_AGENT_MANAGER;
      // _serviceStatusPropertiesGetAll.setAvailability(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface), BTS_DBUS_SERVICE_AVAILABLE); --- AgentManager has no properties
      // evoInterface = ::ccdbusif::evolution::IF_DM;
      // _serviceStatusPropertiesGetAll.setAvailability(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface), BTS_DBUS_SERVICE_AVAILABLE); --- DM, CTN, OPPC, OPPS, SPP, DID, DUN, PAN have no properties
      // evoInterface = ::ccdbusif::evolution::IF_CLIENT;
      // _serviceStatusPropertiesGetAll.setAvailability(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface), BTS_DBUS_SERVICE_AVAILABLE); --- Client has no properties

      evoInterface = ::ccdbusif::evolution::IF_ADAPTER;
      dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_PROPERTIES_GET_ALL;
      dbusItem.busType = (BTSCommonEnumClass)evoParser.getInterface2BusType(evoInterface);
      dbusItem.busName = evoParser.getInterface2BusName(evoInterface);
      dbusItem.objPath = evoParser.getInterface2ObjectPath(evoInterface);
      dbusInterfaces.push_back(dbusItem);

      // evoInterface = ::ccdbusif::evolution::IF_AGENT_MANAGER;
      // dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_PROPERTIES_GET_ALL;
      // dbusItem.busType = (BTSCommonEnumClass)evoParser.getInterface2BusType(evoInterface);
      // dbusItem.busName = evoParser.getInterface2BusName(evoInterface);
      // dbusItem.objPath = evoParser.getInterface2ObjectPath(evoInterface);
      // dbusInterfaces.push_back(dbusItem);

      // evoInterface = ::ccdbusif::evolution::IF_DM;
      // dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_PROPERTIES_GET_ALL;
      // dbusItem.busType = (BTSCommonEnumClass)evoParser.getInterface2BusType(evoInterface);
      // dbusItem.busName = evoParser.getInterface2BusName(evoInterface);
      // dbusItem.objPath = evoParser.getInterface2ObjectPath(evoInterface);
      // dbusInterfaces.push_back(dbusItem);

      // evoInterface = ::ccdbusif::evolution::IF_CLIENT;
      // dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_PROPERTIES_GET_ALL;
      // dbusItem.busType = (BTSCommonEnumClass)evoParser.getInterface2BusType(evoInterface);
      // dbusItem.busName = evoParser.getInterface2BusName(evoInterface);
      // dbusItem.objPath = evoParser.getInterface2ObjectPath(evoInterface);
      // dbusInterfaces.push_back(dbusItem);

      // nothing to do for _serviceStatusGenInitialization
      // nothing to do for _serviceStatusHliInitialization
   }

   if((BTS_USER_MODE_CONNECTION == userMode) ||
      (BTS_USER_MODE_ALL_WI_AGENT == userMode))
   {
      _serviceGroupGeneral.addServiceStatus(&_serviceStatusAgent);

      _serviceStatusAgent.setEnabled(true);

      _serviceStatusAgent.addServiceInfo(evoParser.getAgentObjectPath());

      _serviceStatusAgent.setAvailability(BTS_DBUS_SERVICE_WAITING);

      dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_AGENT;
      dbusItem.busType = (BTSCommonEnumClass)evoParser.getAgentBusType();
      dbusItem.busName = evoParser.getAgentBusName();
      dbusItem.objPath = evoParser.getAgentObjectPath();
      dbusInterfaces.push_back(dbusItem);
   }

   _serviceGroupGeneral.checkAvailability();

   _callback->startupSequenceStarted(_serviceGroupGeneral.getAvailability());
}

bool StartupGenivi::updateDbusServiceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem,
          IN const BTSGenDbusServiceInterface interface, IN const BTSDbusServiceAvailability availability, IN const BTSCommonEnumClass busType, IN const BTSBusName& busName, IN const BTSObjectPath& objPath)
{
   (void)(bts2AppMsgList);
   (void)(messageItem);

   if(BTS_GEN_DBUS_SERVICE_LAST <= interface)
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return false;
   }

   if(BTS_DBUS_SERVICE_LAST <= availability)
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return false;
   }

   bool checkGroupAvailability(false);

   switch(interface)
   {
      case BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SYSTEM:
         if(true == _serviceStatusObjectManagerSystem.getEnabled())
         {
            _serviceStatusObjectManagerSystem.setAvailability(availability);
            checkGroupAvailability = true;

            checkSendingGetAll(bts2IpcMsgList, (BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusObjectManagerSystem.getAvailability()), (BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusAdapter.getAvailability()));
         }
         break;
      case BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SESSION:
         if(true == _serviceStatusObjectManagerSession.getEnabled())
         {
            _serviceStatusObjectManagerSession.setAvailability(availability);
            checkGroupAvailability = true;
         }
         break;
      case BTS_GEN_DBUS_SERVICE_INTROSPECTABLE:
         if(true == _serviceStatusIntrospectable.getEnabled())
         {
            const ::ccdbusif::DbusBusType convertedBusType((::ccdbusif::DbusBusType)busType);
            if((::ccdbusif::BUS_TYPE_SYSTEM == convertedBusType) || (::ccdbusif::BUS_TYPE_SESSION == convertedBusType))
            {
               _serviceStatusIntrospectable.setAvailability(convertedBusType, busName, objPath, availability);
               checkGroupAvailability = true;
            }
            else
            {
               FW_ERRMEM_ASSERT_ALWAYS();
            }
         }
         break;
      case BTS_GEN_DBUS_SERVICE_PROPERTIES_GET_ALL:
         if(true == _serviceStatusPropertiesGetAll.getEnabled())
         {
            const ::ccdbusif::DbusBusType convertedBusType((::ccdbusif::DbusBusType)busType);
            if((::ccdbusif::BUS_TYPE_SYSTEM == convertedBusType) || (::ccdbusif::BUS_TYPE_SESSION == convertedBusType))
            {
               _serviceStatusPropertiesGetAll.setAvailability(convertedBusType, busName, objPath, availability);
               checkGroupAvailability = true;
            }
            else
            {
               FW_ERRMEM_ASSERT_ALWAYS();
            }
         }
         break;
      case BTS_GEN_DBUS_SERVICE_GEN_INITIALIZATION:
         if(true == _serviceStatusGenInitialization.getEnabled())
         {
            _serviceStatusGenInitialization.setAvailability(availability);
            checkGroupAvailability = true;

            if(true == _serviceStatusAgent.getEnabled())
            {
               checkSendingRegisterAgent(bts2IpcMsgList, (BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusAgentManager.getAvailability()), (BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusGenInitialization.getAvailability()), (BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusHliInitialization.getAvailability()));
            }
         }
         break;
      case BTS_GEN_DBUS_SERVICE_HLI_INITIALIZATION:
         if(true == _serviceStatusHliInitialization.getEnabled())
         {
            _serviceStatusHliInitialization.setAvailability(availability);
            checkGroupAvailability = true;

            if(true == _serviceStatusAgent.getEnabled())
            {
               checkSendingRegisterAgent(bts2IpcMsgList, (BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusAgentManager.getAvailability()), (BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusGenInitialization.getAvailability()), (BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusHliInitialization.getAvailability()));
            }
         }
         break;
      case BTS_GEN_DBUS_SERVICE_ADAPTER:
         if(true == _serviceStatusAdapter.getEnabled())
         {
            _serviceStatusAdapter.setAvailability(availability);
            checkGroupAvailability = true;

            checkSendingGetAll(bts2IpcMsgList, (BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusObjectManagerSystem.getAvailability()), (BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusAdapter.getAvailability()));
         }
         break;
      case BTS_GEN_DBUS_SERVICE_AGENT:
         if(true == _serviceStatusAgent.getEnabled())
         {
            _serviceStatusAgent.setAvailability(availability);
            checkGroupAvailability = true;
            if((BTS_DBUS_SERVICE_AVAILABLE == availability) || (BTS_DBUS_SERVICE_NOT_AVAILABLE == availability))
            {
               stopTimer(_registerAgentTimer);
            }
         }
         break;
      case BTS_GEN_DBUS_SERVICE_AGENT_MANAGER:
         if(true == _serviceStatusAgentManager.getEnabled())
         {
            _serviceStatusAgentManager.setAvailability(availability);
            checkGroupAvailability = true;

            if(true == _serviceStatusAgent.getEnabled())
            {
               checkSendingRegisterAgent(bts2IpcMsgList, (BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusAgentManager.getAvailability()), (BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusGenInitialization.getAvailability()), (BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusHliInitialization.getAvailability()));
            }
         }
         break;
      case BTS_GEN_DBUS_SERVICE_DEVICE:
         break;
      case BTS_GEN_DBUS_SERVICE_DM:
         if(true == _serviceStatusDM.getEnabled())
         {
            _serviceStatusDM.setAvailability(availability);
            checkGroupAvailability = true;
         }
         break;
      case BTS_GEN_DBUS_SERVICE_SERVICE:
         break;
      case BTS_GEN_DBUS_SERVICE_TRACE:
         if(true == _serviceStatusTrace.getEnabled())
         {
            _serviceStatusTrace.setAvailability(availability);
            checkGroupAvailability = true;
         }
         break;
      case BTS_GEN_DBUS_SERVICE_GENIVI_TRACE:
         if(true == _serviceStatusGeniviTrace.getEnabled())
         {
            _serviceStatusGeniviTrace.setAvailability(availability);
            checkGroupAvailability = true;
         }
         break;
      case BTS_GEN_DBUS_SERVICE_DID:
         if(true == _serviceStatusDID.getEnabled())
         {
            _serviceStatusDID.setAvailability(availability);
            checkGroupAvailability = true;
         }
         break;
      case BTS_GEN_DBUS_SERVICE_SPP:
         break;
      case BTS_GEN_DBUS_SERVICE_OBEX_CLIENT:
         if(true == _serviceStatusObexClient.getEnabled())
         {
            _serviceStatusObexClient.setAvailability(availability);
            checkGroupAvailability = true;
         }
         break;
      case BTS_GEN_DBUS_SERVICE_SERIAL:
         break;
      case BTS_GEN_DBUS_SERVICE_CONNMAN_MANAGER:
         break;
      case BTS_GEN_DBUS_SERVICE_CONNMAN_TECHNOLOGY:
         break;
      case BTS_GEN_DBUS_SERVICE_CONNMAN_SERVICE:
         break;
      case BTS_GEN_DBUS_SERVICE_HFP_HANDSFREE:
         break;
      case BTS_GEN_DBUS_SERVICE_HFP_MANAGER:
         break;
      case BTS_GEN_DBUS_SERVICE_HFP_MODEM:
         break;
      case BTS_GEN_DBUS_SERVICE_DISABLED_BT_STACK_COMM:
         if(true == _serviceStatusDisabledBtStackComm.getEnabled())
         {
            _serviceStatusDisabledBtStackComm.setAvailability(availability);
            checkGroupAvailability = true;
         }
         break;
      case BTS_GEN_DBUS_SERVICE_LAST:
         break;
      default:
         break;
   }

   return checkGroupAvailability;
}

void StartupGenivi::checkNewGroupAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool checkGroupAvailability)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if(true == checkGroupAvailability)
   {
      const BTSDbusServiceAvailability oldAvailability(_serviceGroupGeneral.getAvailability());
      _serviceGroupGeneral.checkAvailability();
      const BTSDbusServiceAvailability newAvailability(_serviceGroupGeneral.getAvailability());

      if(oldAvailability != newAvailability)
      {
         if(BTS_DBUS_SERVICE_WAITING == newAvailability)
         {
            _callback->indicateAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, BTS_DBUS_SERVICE_WAITING);
         }
         else if((BTS_DBUS_SERVICE_NOT_AVAILABLE == newAvailability) || (true == _testSimulateDbusStartupError) || (true == _testSimulateEvolutionStartupError))
         {
            BTSBluetoothOffReason reason;

            if(true == _serviceStatusDisabledBtStackComm.getEnabled())
            {
               ETG_TRACE_ERRMEM((" #CONN: ERROR: BT STACK COMMUNICATION DISABLED"));
               reason = BTS_BT_OFF_REASON_DBUS_ERROR;
            }
            else
            {
               bool stackRunning(true);
               const ::std::string hliProcess("evo_hli_socket");
               const ::std::string genProcess("evo_genivi_bt");

               if((false == ::fw::isProcessRunning(hliProcess)) || (true == _testSimulateEvolutionStartupError))
               {
                  ETG_TRACE_ERRMEM((" #CONN: ERROR: process %s is not running", hliProcess.c_str()));
                  stackRunning = false;
               }
               if(false == ::fw::isProcessRunning(genProcess))
               {
                  ETG_TRACE_ERRMEM((" #CONN: ERROR: process %s is not running", genProcess.c_str()));
                  stackRunning = false;
               }

               if(true == stackRunning)
               {
                  reason = BTS_BT_OFF_REASON_DBUS_ERROR;
               }
               else
               {
                  reason = BTS_BT_OFF_REASON_STACK_NOT_RUNNING; // HINT: does not work with separated connectivity domain
               }
            }

            _callback->indicateAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, BTS_DBUS_SERVICE_NOT_AVAILABLE, reason);
         }
         else if(BTS_DBUS_SERVICE_AVAILABLE == newAvailability)
         {
            _callback->indicateAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, BTS_DBUS_SERVICE_AVAILABLE);
         }
         else
         {
            FW_ERRMEM_ASSERT_ALWAYS();
         }
      }
   }
}

void StartupGenivi::handleAdapterProperties(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusPropertyList& properties, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(properties);

   // is triggered by a signal or method return => check error code
   if(BTS_IPC_SUCCESS == errorCode)
   {
      // stop timer
      stopGetAllTimer();

      if(true == getDataReceived())
      {
         // adapter data was already received => ignore this message
      }
      else
      {
         // first reception of adapter data => process
         setDataReceived(true);

         // store information => done by handler of other SMs
      }
   }
   else
   {
      // error was returned => GetAll request failed
      if(false == getDataReceived())
      {
         // start timer
         startGetAllTimer();
      }
   }
}

void StartupGenivi::handleAdapterInitialization(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   if(true == getAdapterInitDone())
   {
      // adapter initialization was already received => ignore this message
   }
   else
   {
      // first reception of adapter initialization => process
      setAdapterInitDone(true);
   }
}

void StartupGenivi::handleRegisterAgentResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if(true == _serviceStatusAgent.getEnabled())
   {
      if(true == _callback->getStartupOngoingFlag())
      {
         if(BTS_IPC_SUCCESS == errorCode)
         {
            // create service availability message
            _controlIf->createDbusServiceAvailabilityMessage((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_AGENT, BTS_DBUS_SERVICE_AVAILABLE);
         }
         else
         {
            // retry again
            startRepeatTimer();
         }
      }
      else
      {
         // startup sequence is somehow completed; assumption is that startup failed; continue now with success (if possible)
         if(BTS_IPC_SUCCESS == errorCode)
         {
            // create service availability message
            _controlIf->createDbusServiceAvailabilityMessage((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_AGENT, BTS_DBUS_SERVICE_AVAILABLE);
         }
         else
         {
            // ignore
         }
      }
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
   }
}

void StartupGenivi::handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   // check timer id
   if(true == _timerGetAll.compare(timerId))
   {
      ::std::vector< Bts2Ipc_BaseMessage* > bts2IpcMsgList;
      createGetAllPropertiesConnectionMsg(bts2IpcMsgList);
      if(0 < bts2IpcMsgList.size())
      {
         _controlIf->sendInternalBts2IpcMessage(bts2IpcMsgList[0]);
      }
   }
   else if(true == _timerInit.compare(timerId))
   {
      // simulate AdapterInitialization signal and cross the fingers that we have waited enough
      Ipc2Bts_AdapterInitialization* msg = ptrNew_Ipc2Bts_AdapterInitialization();
      if(0 != msg)
      {
         msg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
         _controlIf->sendInternalIpc2BtsMessage(msg);
      }
   }
   else if(true == _registerAgentTimer.compare(timerId))
   {
      ::std::vector< Bts2Ipc_BaseMessage* > bts2IpcMsgList;
      _registerAgentSent = false;
      checkSendingRegisterAgent(bts2IpcMsgList, true, true, true);
      if(0 < bts2IpcMsgList.size())
      {
         _controlIf->sendInternalBts2IpcMessage(bts2IpcMsgList[0]);
      }
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
   }
}

void StartupGenivi::setDataReceived(IN const bool enable)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   if(true == enable)
   {
      startInitTimer();
   }

   _getAllReceived = enable;

   if(true == _getAllReceived)
   {
      _controlIf->createDbusServiceAvailabilityMessage((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_GEN_INITIALIZATION, BTS_DBUS_SERVICE_AVAILABLE);
   }
}

bool StartupGenivi::getDataReceived(void) const
{
   return _getAllReceived;
}

void StartupGenivi::setAdapterInitDone(IN const bool enable)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   if(true == enable)
   {
      stopInitTimer();
   }

   _initReceived = enable;

   if(true == _initReceived)
   {
      _controlIf->createDbusServiceAvailabilityMessage((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_HLI_INITIALIZATION, BTS_DBUS_SERVICE_AVAILABLE);
   }
}

bool StartupGenivi::getAdapterInitDone(void) const
{
   return _initReceived;
}

void StartupGenivi::startGetAllTimer(void)
{
   // start timer only once
   if(true == _getAllTimerUsed)
   {
      return;
   }

   _getAllTimerUsed = true;

   startTimer(_timerGetAll, TIMEOUT_GET_ALL_ADAPTER);
}

void StartupGenivi::stopGetAllTimer(void)
{
   _getAllTimerUsed = true;

   stopTimer(_timerGetAll);
}

void StartupGenivi::checkSendingGetAll(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const bool objectManagerAvailable, IN const bool adapterAvailable)
{
   // send request only once
   if(true == _getAllSent)
   {
      return;
   }

   if((true == objectManagerAvailable) && (true == adapterAvailable))
   {
      _getAllSent = true;
      createGetAllPropertiesConnectionMsg(bts2IpcMsgList);
   }
}

void StartupGenivi::startInitTimer(void)
{
   // start timer only once
   if(true == _initTimerUsed)
   {
      return;
   }

   _initTimerUsed = true;

   startTimer(_timerInit, TIMEOUT_INIT_ADAPTER);
}

void StartupGenivi::stopInitTimer(void)
{
   _initTimerUsed = true;

   stopTimer(_timerInit);
}

void StartupGenivi::checkSendingRegisterAgent(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const bool agentManagerAvailable, IN const bool genInitializationAvailable, IN const bool hliInitializationAvailable)
{
   // send request only once
   if(true == _registerAgentSent)
   {
      return;
   }

   if((true == agentManagerAvailable) && (true == genInitializationAvailable) && (true == hliInitializationAvailable))
   {
      _registerAgentSent = true;
      createRegisterAgentMsg(bts2IpcMsgList);
   }
}

void StartupGenivi::startRepeatTimer(void)
{
   if(MAX_NUMBER_REGISTER_AGENT_REQUESTS <= _nmbRegisterAgentRequests)
   {
      // limit reached => timeout will happen
      return;
   }

   _nmbRegisterAgentRequests++;
   startTimer(_registerAgentTimer, TIMEOUT_REPEAT_REGISTER_AGENT);
}

void StartupGenivi::createGetAllPropertiesConnectionMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList) const
{
   // send request to get all properties of adapter interface (e.g. current Bluetooth on/off status)
   Bts2Ipc_GetAllPropertiesConnection* msg = ptrNew_Bts2Ipc_GetAllPropertiesConnection();
   if(0 != msg)
   {
      const ::ccdbusif::evolution::Interface evoInterface(::ccdbusif::evolution::IF_ADAPTER);
      ::ccdbusif::evolution::EvolutionGeniviDbusParser evoParser;

      msg->setBusType((BTSCommonEnumClass)evoParser.getInterface2BusType(evoInterface));
      msg->setBusName(evoParser.getInterface2BusName(evoInterface));
      msg->setObjPath(evoParser.getInterface2ObjectPath(evoInterface));
      msg->setInterface(evoInterface);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void StartupGenivi::createRegisterAgentMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList) const
{
   // register agent
   Bts2Ipc_RegisterAgent* msg = ptrNew_Bts2Ipc_RegisterAgent();
   if(0 != msg)
   {
      ::ccdbusif::evolution::EvolutionGeniviDbusParser evoParser;

      msg->setAgent(evoParser.getAgentObjectPath());
      msg->setCapability(_capability);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void StartupGenivi::createGetIntrospectionConnectionMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList) const
{
   Bts2Ipc_GetIntrospectionConnection* msg = ptrNew_Bts2Ipc_GetIntrospectionConnection();
   if(0 != msg)
   {
      const ::ccdbusif::evolution::Interface evoInterface(::ccdbusif::evolution::IF_ADAPTER);
      ::ccdbusif::evolution::EvolutionGeniviDbusParser evoParser;

      msg->setBusType((BTSCommonEnumClass)evoParser.getInterface2BusType(evoInterface));
      msg->setBusName(evoParser.getInterface2BusName(evoInterface));
      msg->setObjPath(evoParser.getInterface2ObjectPath(evoInterface));
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void StartupGenivi::startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout)
{
   // ETG_TRACE_USR3((" startTimer: timeout=%u", timeout));

   FW_ERRMEM_IF_NULL_PTR_RETURN(_timerPoolIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   timer.setTimerPool(_timerPoolIf);
   timer.start(_controlIf, this, timeout);
}

void StartupGenivi::stopTimer(IN ExtendedTimerEntry& timer) const
{
   // ETG_TRACE_USR3((" stopTimer"));

   timer.stop();

   // do not release timer
}

void StartupGenivi::releaseTimer(IN ExtendedTimerEntry& timer) const
{
   // ETG_TRACE_USR3((" releaseTimer"));

   timer.release();
}

} //genivi
} //btstackif
