/**
 * @file SppPoolGenivi.cpp
 *
 * @par SW-Component
 * State machine for SPP pool
 *
 * @brief Implementation of Genivi SPP pool state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Genivi SPP pool state machine.
 */

#include "SppPoolGenivi.h"
#include "ISppPoolCallback.h"
#include "IBasicControl.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"

#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/SppPoolGenivi.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace genivi {

SppPoolGenivi::SppPoolGenivi() :
_callback(0),
_controlIf(0),
_timerPoolIf(0),
_maxSppConnections(30),
_maxLocalSppServices(23)
{
}

SppPoolGenivi::~SppPoolGenivi()
{
   _callback = 0;
   _controlIf = 0;
   _timerPoolIf = 0;
}

void SppPoolGenivi::reset(void)
{
}

void SppPoolGenivi::setCallback(IN ISppPoolCallback* callback)
{
   _callback = callback;

   FW_NORMAL_ASSERT(0 != _callback);
}

void SppPoolGenivi::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);
}

void SppPoolGenivi::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_NORMAL_ASSERT(0 != _timerPoolIf);
}

BTSSppInstanceId SppPoolGenivi::getMaxNumberSppConnections(void) const
{
   return _maxSppConnections;
}

BTSSppInstanceId SppPoolGenivi::getMaxNumberLocalSppServices(void) const
{
   return _maxLocalSppServices;
}

void SppPoolGenivi::setUuid(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSSppInstanceId instance, IN const BTSUuid& outgoingUuid, IN const BTSUuid& incomingUuid)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(instance);
   (void)(outgoingUuid);
   (void)(incomingUuid);
}

void SppPoolGenivi::setLocalSppUuids(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSUuidList& uuidList)
{
   (void)(bts2AppMsgList);

   createSetAdapterAvailableSppServersMsg(bts2IpcMsgList, uuidList);
}

void SppPoolGenivi::sendVirtualLocalSppUuidsUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSUuidList& uuidList, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_controlIf);

   Ipc2Bts_AdapterAvailableSppServersUpdate* msg = ptrNew_Ipc2Bts_AdapterAvailableSppServersUpdate();
   if(0 != msg)
   {
      msg->setAvailableSppServers(uuidList);
      msg->setIpcCommonErrorCode(errorCode);
   }
   _controlIf->sendInternalIpc2BtsMessage(msg);
}

ISppPoolRequest* SppPoolGenivi::getRequestIf(void)
{
   return this;
}

void SppPoolGenivi::handleLocalSppUuidsUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSUuidList& uuidList, IN const BTSIpcCommonErrorCode errorCode, IN const bool response)
{
   (void)(response);

   FW_IF_NULL_PTR_RETURN(_callback);

   _callback->updateLocalSppUuids(bts2IpcMsgList, bts2AppMsgList, messageItem, uuidList, (BTS_IPC_SUCCESS == errorCode));
}

void SppPoolGenivi::createSetAdapterAvailableSppServersMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSUuidList& uuidList) const
{
   Bts2Ipc_SetAdapterAvailableSppServers* msg = ptrNew_Bts2Ipc_SetAdapterAvailableSppServers();
   if(0 != msg)
   {
      msg->setAvailableSppServers(uuidList);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

} //genivi
} //btstackif
