/**
 * @file ServiceSearchGenivi.cpp
 *
 * @par SW-Component
 * State machine for service search
 *
 * @brief Implementation of Genivi service search state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Genivi service search state machine.
 */

#include "ServiceSearchGenivi.h"
#include "IServiceSearchCallback.h"
#include "IBasicControl.h"
#include "IObjectPathManagerGenivi.h"
#include "ServiceSearchData.h"
#include "FwAssert.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"

namespace btstackif {
namespace genivi {

ServiceSearchGenivi::ServiceSearchGenivi() :
_callback(0),
_controlIf(0),
_objectPathManagerIf(0)
{
}

ServiceSearchGenivi::~ServiceSearchGenivi()
{
   _callback = 0;
   _controlIf = 0;
   _objectPathManagerIf = 0;
}

void ServiceSearchGenivi::reset(void)
{
}

void ServiceSearchGenivi::setCallback(IN IServiceSearchCallback* callback)
{
   _callback = callback;

   FW_NORMAL_ASSERT(0 != _callback);
}

void ServiceSearchGenivi::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);
}

void ServiceSearchGenivi::startSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSearchType searchType)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManagerIf->getObjectPath4Address(objPath, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createDiscoverServicesRequest(bts2IpcMsgList, address, searchType, objPath, true);
}

void ServiceSearchGenivi::cancelSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManagerIf->getObjectPath4Address(objPath, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createCancelDiscoveryRequest(bts2IpcMsgList, address, objPath, true);
}

void ServiceSearchGenivi::sendVirtualFailedResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_controlIf);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManagerIf->getObjectPath4Address(objPath, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   Ipc2Bts_DiscoverServicesResult* msg = ptrNew_Ipc2Bts_DiscoverServicesResult();
   if(0 != msg)
   {
      msg->setDevice(objPath);
      msg->setApp2BtsCompareItem(_callback->getDataEntry(address).requestItem.item);
      msg->setIpcCommonErrorCode(BTS_IPC_RETRY_ABORTED);
   }

   _controlIf->sendInternalIpc2BtsMessage(msg, true);
}

void ServiceSearchGenivi::getDeviceUuids(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManagerIf->getObjectPath4Address(objPath, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createGetUuidRequest(bts2IpcMsgList, address, objPath, true);
}

void ServiceSearchGenivi::getServiceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManagerIf->getObjectPath4Protocol(objPath, address, protocol))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createGetServiceVersionRequest(bts2IpcMsgList, address, protocol, objPath, true);
}

void ServiceSearchGenivi::getDeviceVendorId(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManagerIf->getObjectPath4Address(objPath, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createGetDeviceVendorIdRequest(bts2IpcMsgList, address, objPath, true);
}

void ServiceSearchGenivi::getDeviceVendorIdSource(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManagerIf->getObjectPath4Address(objPath, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createGetDeviceVendorIdSourceRequest(bts2IpcMsgList, address, objPath, true);
}

void ServiceSearchGenivi::getDeviceProductId(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManagerIf->getObjectPath4Address(objPath, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createGetDeviceProductIdRequest(bts2IpcMsgList, address, objPath, true);
}

void ServiceSearchGenivi::getDeviceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManagerIf->getObjectPath4Address(objPath, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createGetDeviceVersionRequest(bts2IpcMsgList, address, objPath, true);
}

void ServiceSearchGenivi::getPbapInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManagerIf->getObjectPath4Address(objPath, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createGetPbapInfoRequest(bts2IpcMsgList, address, objPath, true);
}

void ServiceSearchGenivi::getMapInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManagerIf->getObjectPath4Address(objPath, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createGetMapInfoRequest(bts2IpcMsgList, address, objPath, true);
}

void ServiceSearchGenivi::setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager)
{
   _objectPathManagerIf = objectPathManager;

   FW_NORMAL_ASSERT(0 != _objectPathManagerIf);
}

void ServiceSearchGenivi::handleDiscoverServicesResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& address, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      return;
   }

   _callback->serviceSearchResult(bts2IpcMsgList, bts2AppMsgList, messageItem, connectFailed, address, errorCode);
}

void ServiceSearchGenivi::handleCancelDiscoveryResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      return;
   }

   _callback->cancelServiceSearchResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, errorCode);
}

void ServiceSearchGenivi::setSdpSearchActiveAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool enable, IN const bool success)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   _callback->setSdpSearchActiveAfterPairing(bts2IpcMsgList, bts2AppMsgList, messageItem, address, enable, success);
}

bool ServiceSearchGenivi::startSdpSearchAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool success)
{
   FW_IF_NULL_PTR_RETURN_FALSE(_callback);
   FW_IF_NULL_PTR_RETURN_FALSE(_objectPathManagerIf);

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }

   return _callback->startSdpSearchAfterPairing(bts2IpcMsgList, bts2AppMsgList, messageItem, address, success);
}

bool ServiceSearchGenivi::startSdpSearchAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device)
{
   FW_IF_NULL_PTR_RETURN_FALSE(_callback);
   FW_IF_NULL_PTR_RETURN_FALSE(_objectPathManagerIf);

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }

   return _callback->startSdpSearchAfterPairing(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
}

void ServiceSearchGenivi::setPairingStatus(IN const BTSObjectPath& device, IN const bool success)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   _callback->setPairingStatus(address, success);
}

void ServiceSearchGenivi::updateDeviceUuids(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSSupportedServices& supportedServices, IN const BTSUuidList& uuidList)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      return;
   }

   _callback->updateDeviceUuids(bts2IpcMsgList, bts2AppMsgList, messageItem, result, response, address, supportedServices, uuidList);
}

void ServiceSearchGenivi::updateServiceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSProfileVersion version)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSBDAddress address;
   BTSProtocolId protocol(BTS_PROTO_HFP);
   if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, device))
   {
      // should never happen
      return;
   }

   _callback->updateServiceVersion(bts2IpcMsgList, bts2AppMsgList, messageItem, result, response, address, protocol, version);
}

void ServiceSearchGenivi::updateDeviceVendorId(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSVendorId vendorId)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      return;
   }

   _callback->updateDeviceVendorId(bts2IpcMsgList, bts2AppMsgList, messageItem, result, response, address, vendorId);
}

void ServiceSearchGenivi::updateDeviceVendorIdSource(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSVendorIdSource vendorIdSource)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      return;
   }

   _callback->updateDeviceVendorIdSource(bts2IpcMsgList, bts2AppMsgList, messageItem, result, response, address, vendorIdSource);
}

void ServiceSearchGenivi::updateDeviceProductId(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSProductId productId)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      return;
   }

   _callback->updateDeviceProductId(bts2IpcMsgList, bts2AppMsgList, messageItem, result, response, address, productId);
}

void ServiceSearchGenivi::updateDeviceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSVersion version)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      return;
   }

   _callback->updateDeviceVersion(bts2IpcMsgList, bts2AppMsgList, messageItem, result, response, address, version);
}

void ServiceSearchGenivi::updatePbapInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSPbapSdpRecord& record, IN const bool valid)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      return;
   }

   _callback->updatePbapInfo(bts2IpcMsgList, bts2AppMsgList, messageItem, result, response, address, record, valid);
}

void ServiceSearchGenivi::updateMapInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSMapSdpRecordList& recordList, IN const bool valid)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      return;
   }

   _callback->updateMapInfo(bts2IpcMsgList, bts2AppMsgList, messageItem, result, response, address, recordList, valid);
}

void ServiceSearchGenivi::createDiscoverServicesRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSSearchType searchType, IN const BTSObjectPath& device, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_DiscoverServices* msg = ptrNew_Bts2Ipc_DiscoverServices();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setSearchType(searchType);

      ::ccdbusif::evolution::EvolutionGeniviDbusParser evoParser;
      BTSUuid uuid;
      switch(searchType)
      {
         case BTS_SEARCH_ALL:
            // empty uuid; using Evolution we cannot send a separate search request for SPP => same request for ALL and SPP
            break;
         case BTS_SEARCH_SPP:
            // empty uuid; using Evolution we cannot send a separate search request for SPP => same request for ALL and SPP
            break;
         case BTS_SEARCH_DID:
            uuid = evoParser.getUuid2String(::ccdbusif::evolution::UUID_PNPINFORMATION);
            break;
         case BTS_SEARCH_PBAP:
            uuid = evoParser.getUuid2String(::ccdbusif::evolution::UUID_PHONEBOOK_ACCESS_PSE);
            break;
         case BTS_SEARCH_MAP:
            uuid = evoParser.getUuid2String(::ccdbusif::evolution::UUID_MESSAGE_ACCESS_SERVER);
            break;
         case BTS_SEARCH_PAN:
            // empty uuid; using Evolution we cannot send a separate search request for PAN because BT device can support PANU, NAP and GN roles
            break;
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }
      msg->setUuid(uuid);

      msg->setDevice(device);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void ServiceSearchGenivi::createCancelDiscoveryRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_CancelDiscovery* msg = ptrNew_Bts2Ipc_CancelDiscovery();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setDevice(device);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void ServiceSearchGenivi::createGetUuidRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_GetDeviceUuids* msg = ptrNew_Bts2Ipc_GetDeviceUuids();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setDevice(device);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void ServiceSearchGenivi::createGetServiceVersionRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSObjectPath& service, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_GetServiceVersion* msg = ptrNew_Bts2Ipc_GetServiceVersion();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setProtocol(protocol);
      msg->setService(service);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void ServiceSearchGenivi::createGetDeviceVendorIdRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_GetDeviceVendorId* msg = ptrNew_Bts2Ipc_GetDeviceVendorId();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setDevice(device);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void ServiceSearchGenivi::createGetDeviceVendorIdSourceRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_GetDeviceVendorIdSource* msg = ptrNew_Bts2Ipc_GetDeviceVendorIdSource();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setDevice(device);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void ServiceSearchGenivi::createGetDeviceProductIdRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_GetDeviceProductId* msg = ptrNew_Bts2Ipc_GetDeviceProductId();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setDevice(device);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void ServiceSearchGenivi::createGetDeviceVersionRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_GetDeviceVersion* msg = ptrNew_Bts2Ipc_GetDeviceVersion();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setDevice(device);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void ServiceSearchGenivi::createGetPbapInfoRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_GetDevicePbapInfo* msg = ptrNew_Bts2Ipc_GetDevicePbapInfo();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setDevice(device);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void ServiceSearchGenivi::createGetMapInfoRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_GetDeviceMapInfo* msg = ptrNew_Bts2Ipc_GetDeviceMapInfo();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setDevice(device);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

} //genivi
} //btstackif
