/**
 * @file PanConnectDisconnectGenivi.cpp
 *
 * @par SW-Component
 * State machine for PAN connect/disconnect
 *
 * @brief Implementation of Genivi PAN connect/disconnect state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Genivi PAN connect/disconnect state machine.
 */

#include "PanConnectDisconnectGenivi.h"
#include "IProtocolManagerCallback.h"
#include "ProtocolManagerData.h"
#include "IBasicControl.h"
#include "IConfiguration.h"
#include "IDeviceManager.h"
#include "IObjectPathManagerGenivi.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "BtsUtils.h"
#include "FwErrmemPrint.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/PanConnectDisconnectGenivi.cpp.trc.h"
#endif
#endif

// if enabled connect and disconnect SMs shall be updated and tested
// #define ENABLE_BUSY
#undef ENABLE_BUSY

namespace btstackif {
namespace genivi {

PanConnectDisconnectGenivi::PanConnectDisconnectGenivi() :
_callback(0),
_controlIf(0),
_timerPoolIf(0),
_configurationIf(0),
_deviceManagerIf(0),
_objectPathManagerIf(0),
_serviceStatusConnManManager(),
_serviceStatusConnManService(),
_connectDisconnectDataList(),
_serviceConnManManagerTimeout(10000),
_serviceConnManServiceTimeout(10000)
{
   _serviceStatusConnManManager.setEnabled(false); // initial setting is false
   _serviceStatusConnManManager.setName("ConnManManager");
   _serviceStatusConnManManager.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_CONNMAN_MANAGER);
   _serviceStatusConnManService.setEnabled(false); // initial setting is false
   _serviceStatusConnManService.setName("ConnManService");
   _serviceStatusConnManService.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_CONNMAN_SERVICE);
}

PanConnectDisconnectGenivi::~PanConnectDisconnectGenivi()
{
   _callback = 0;
   _controlIf = 0;
   _timerPoolIf = 0;
   _configurationIf = 0;
   _deviceManagerIf = 0;
   _objectPathManagerIf = 0;
}

void PanConnectDisconnectGenivi::reset(void)
{
   // keep _serviceStatusConnManManager unchanged
   _serviceStatusConnManService.removeAllServiceInfos(); // TODO: check during testing if object path will be removed by ConnMan
   // stop and release all timer
   for(::std::map< BTSBDAddress, PanConnectDisconnectDataGenivi >::iterator it = _connectDisconnectDataList.begin(); it != _connectDisconnectDataList.end(); ++it)
   {
      releaseTimer(it->second.serviceConnManManagerTimer);
      releaseTimer(it->second.serviceConnManServiceTimer);
   }
   _connectDisconnectDataList.clear();

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);
   _objectPathManagerIf->removeAllProtocol2ObjectPathMappings();
}

void PanConnectDisconnectGenivi::setCallback(IN IProtocolManagerCallback* callback)
{
   _callback = callback;

   FW_ERRMEM_ASSERT(0 != _callback);
}

void PanConnectDisconnectGenivi::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   _serviceStatusConnManManager.setIpc2BtsSendIf(_controlIf);
   _serviceStatusConnManService.setIpc2BtsSendIf(_controlIf);
}

void PanConnectDisconnectGenivi::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_ERRMEM_ASSERT(0 != _timerPoolIf);
}

void PanConnectDisconnectGenivi::prepareProtocolConnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   // ensure that data entry is available
   (void)checkDataList(address);

   // simulate some states: protocol added + status updated
   // DBUS service availability is not needed
   BTSHandleIpc2BtsMessageItem messageItem;
   _callback->protocolAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);
   FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
   FW_ERRMEM_ASSERT(0 == messageItem.message);

   // initial status is always disconnected => no action required

   // check if all data is complete
   checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName);
   FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
   FW_ERRMEM_ASSERT(0 == messageItem.message);
}

void PanConnectDisconnectGenivi::setDeviceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool available)
{
   (void)(available);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName);
}

bool PanConnectDisconnectGenivi::isServiceAvailable(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   // check for availability of ConnMan service
   return (BTS_DBUS_SERVICE_AVAILABLE == _serviceStatusConnManManager.getAvailability());
}

void PanConnectDisconnectGenivi::waitForServiceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   PanConnectDisconnectDataGenivi& entry = checkDataList(address);

   // start timer if not active
   if(false == isTimerActive(entry.serviceConnManManagerTimer))
   {
      startTimer(entry.serviceConnManManagerTimer, _serviceConnManManagerTimeout);
   }
}

bool PanConnectDisconnectGenivi::isServiceSearchNeeded(OUT BTSSearchType& searchType, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   // needed
   searchType = BTS_SEARCH_PAN;
   return true;
}

bool PanConnectDisconnectGenivi::isSettingUuidNeeded(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   // not needed
   return false;
}

void PanConnectDisconnectGenivi::waitForProtocolAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   PanConnectDisconnectDataGenivi& entry = checkDataList(address);

   // start timer if not active
   if(false == isTimerActive(entry.serviceConnManServiceTimer))
   {
      startTimer(entry.serviceConnManServiceTimer, _serviceConnManServiceTimeout);
   }
}

void PanConnectDisconnectGenivi::connect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath service;
   if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   createConnManServiceConnectMsg(bts2IpcMsgList, address, service);
}

void PanConnectDisconnectGenivi::disconnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool pauseBtStreaming)
{
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);
   (void)(pauseBtStreaming);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath service;
   if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   createConnManServiceDisconnectMsg(bts2IpcMsgList, address, service);
}

void PanConnectDisconnectGenivi::cancel(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath service;
   if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   // sending of disconnect during ongoing connect disabled because extension of SM is necessary
   createConnManServiceDisconnectMsg(bts2IpcMsgList, address, service);
}

void PanConnectDisconnectGenivi::accept(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool accept)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);
   (void)(accept);

   // no remote connect for PAN
   FW_ERRMEM_ASSERT_ALWAYS();
}

void PanConnectDisconnectGenivi::sendVirtualFailedConnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSIpcCommonErrorCode errorCode) const
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath service;
   if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol))
   {
      // should never happen
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   Ipc2Bts_ConnManServiceConnectResult* msg = ptrNew_Ipc2Bts_ConnManServiceConnectResult();
   if(0 != msg)
   {
      msg->setDevice(service);
      msg->setIpcCommonErrorCode(errorCode);
   }

   _controlIf->sendInternalIpc2BtsMessage(msg, true);
}

void PanConnectDisconnectGenivi::sendVirtualFailedDisconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSIpcCommonErrorCode errorCode) const
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath service;
   if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol))
   {
      // should never happen
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   Ipc2Bts_ConnManServiceDisconnectResult* msg = ptrNew_Ipc2Bts_ConnManServiceDisconnectResult();
   if(0 != msg)
   {
      msg->setDevice(service);
      msg->setIpcCommonErrorCode(errorCode);
   }

   _controlIf->sendInternalIpc2BtsMessage(msg, true);
}

void PanConnectDisconnectGenivi::sendVirtualConnectedUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool connected, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath service;
   if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol))
   {
      // should never happen
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   Ipc2Bts_ConnManServiceStateUpdate* msg = ptrNew_Ipc2Bts_ConnManServiceStateUpdate();
   if(0 != msg)
   {
      if(true == connected)
      {
         msg->setValue(::ccdbusif::connman::SERVICE_STATE_READY);
      }
      else
      {
         msg->setValue(::ccdbusif::connman::SERVICE_STATE_IDLE);
      }
      msg->setDevice(service);
      msg->setIpcCommonErrorCode(errorCode);
   }

   _controlIf->sendInternalIpc2BtsMessage(msg, true);

   if(true == connected)
   {
      Ipc2Bts_ConnManServiceConnectResult* msg2 = ptrNew_Ipc2Bts_ConnManServiceConnectResult();
      if(0 != msg2)
      {
         msg2->setDevice(service);
         msg2->setIpcCommonErrorCode(errorCode);
      }

      _controlIf->sendInternalIpc2BtsMessage(msg2, true);
   }
   else
   {
      Ipc2Bts_ConnManServiceDisconnectResult* msg2 = ptrNew_Ipc2Bts_ConnManServiceDisconnectResult();
      if(0 != msg2)
      {
         msg2->setDevice(service);
         msg2->setIpcCommonErrorCode(errorCode);
      }

      _controlIf->sendInternalIpc2BtsMessage(msg2, true);
   }
}

IProtocolManagerRequest* PanConnectDisconnectGenivi::getRequestIf(void)
{
   return this;
}

void PanConnectDisconnectGenivi::setConfigurationIf(IN IConfiguration* configuration)
{
   _configurationIf = configuration;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_configurationIf);

   _serviceStatusConnManManager.setEnabled(_configurationIf->getConfiguration().supportedServices.getBit(BTS_SUPP_SRV_PAN));
   _serviceStatusConnManService.setEnabled(_configurationIf->getConfiguration().supportedServices.getBit(BTS_SUPP_SRV_PAN));
}

void PanConnectDisconnectGenivi::setDeviceManagerIf(IN IDeviceManager* deviceManagerIf)
{
   _deviceManagerIf = deviceManagerIf;

   FW_ERRMEM_ASSERT(0 != _deviceManagerIf);
}

void PanConnectDisconnectGenivi::setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager)
{
   _objectPathManagerIf = objectPathManager;

   FW_ERRMEM_ASSERT(0 != _objectPathManagerIf);
}

void PanConnectDisconnectGenivi::addInterfaces(OUT ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSUserMode userMode)
{
   if((BTS_USER_MODE_CONNECTION == userMode) ||
      (BTS_USER_MODE_CONN_WO_AGENT == userMode) ||
      (BTS_USER_MODE_ALL_WI_AGENT == userMode) ||
      (BTS_USER_MODE_ALL_WO_AGENT == userMode))
   {
      _serviceStatusConnManManager.addServiceInfo("/");

      if(true == _serviceStatusConnManManager.getEnabled())
      {
         BTSDbusInterfaceItem dbusItem;
         dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_CONNMAN_MANAGER;
         dbusInterfaces.push_back(dbusItem);

         // do not set BTS_DBUS_SERVICE_WAITING for _serviceStatusConnManManager because it will start a timer
      }
   }
}

void PanConnectDisconnectGenivi::protocolAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSObjectPath& protocolObjectPath, IN const BTSProtocolId protocol)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" protocolAdded: protocol=%d address=%20s protocolObjectPath=%s", protocol, address.c_str(), protocolObjectPath.c_str()));

   if(true == _serviceStatusConnManService.getEnabled())
   {
      // service object path was created
      // given: device address, service object, protocol

      // store mapping
      _objectPathManagerIf->setProtocol2ObjectPath(address, protocol, protocolObjectPath);

      // DBUS service information has to be added
      _serviceStatusConnManService.addServiceInfo(address, protocolObjectPath);
      // set to waiting because proxy was created
      _serviceStatusConnManService.setAvailability(address, BTS_DBUS_SERVICE_WAITING);

      // consider that this function can be called more than once
      // doubled calls are handled within protocolAdded()
      const BTSSppInstanceId sppInstance(0);
      const BTSUuid uuid;
      const BTSMasInstanceId masInstance(0);
      const BTSMasInstanceName masName;
      _callback->protocolAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);

      // check if all data is complete
      checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName);
   }
}

void PanConnectDisconnectGenivi::protocolRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& protocolObjectPath)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(true == _serviceStatusConnManService.getEnabled())
   {
      // service object path was removed
      BTSBDAddress address;
      BTSProtocolId protocol;
      if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, protocolObjectPath))
      {
         // ignore
         return;
      }

      if(BTS_PROTO_PAN != protocol)
      {
         // ignore
         return;
      }

      // do not check --- if(false == _callback->isPanViaConnMan())

      ETG_TRACE_USR3((" protocolRemoved: protocolObjectPath=%s", protocolObjectPath.c_str()));

      // DBUS service information has to be removed
      _serviceStatusConnManService.removeServiceInfo(address);
      // remove related entry in protocol manager
      const BTSSppInstanceId sppInstance(0);
      const BTSUuid uuid;
      const BTSMasInstanceId masInstance(0);
      const BTSMasInstanceName masName;
      _callback->protocolRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);

      _objectPathManagerIf->removeProtocol2ObjectPath(protocolObjectPath);
   }
}

void PanConnectDisconnectGenivi::updateDbusServiceAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& service, IN const BTSDbusServiceAvailability availability)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" updateDbusServiceAvailability: service=%s", service.c_str()));

   if(true == _serviceStatusConnManService.getEnabled())
   {
      BTSBDAddress address;
      BTSProtocolId protocol;
      if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, service))
      {
         // service object path is created before proxy is created => therefore this error should never happen
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }

      // set availability
      _serviceStatusConnManService.setAvailability(address, availability);
      const bool available(BTS_DBUS_SERVICE_AVAILABLE == availability);

      // TODO: [low]: check for BTS_DBUS_SERVICE_NOT_AVAILABLE needed

      const BTSSppInstanceId sppInstance(0);
      const BTSUuid uuid;
      const BTSMasInstanceId masInstance(0);
      const BTSMasInstanceName masName;

      // store availability
      setDbusServiceAvailability(*_callback, address, protocol, sppInstance, uuid, masInstance, masName, available);

      // check if all data is complete
      checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName);
   }
}

void PanConnectDisconnectGenivi::updateDbusServiceConnManAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& service, IN const BTSDbusServiceAvailability availability)
{
   (void)(service);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   ETG_TRACE_USR3((" updateDbusServiceConnManAvailability: service=%s", service.c_str()));

   if(true == _serviceStatusConnManManager.getEnabled())
   {
      _serviceStatusConnManManager.setAvailability(availability);

      if(BTS_DBUS_SERVICE_AVAILABLE == availability)
      {
         for(::std::map< BTSBDAddress, PanConnectDisconnectDataGenivi >::iterator it = _connectDisconnectDataList.begin(); it != _connectDisconnectDataList.end(); ++it)
         {
            PanConnectDisconnectDataGenivi& entry = it->second;

            // check for active timer
            if(true == isTimerActive(entry.serviceConnManManagerTimer))
            {
               stopTimer(entry.serviceConnManManagerTimer);

               // ConnMan manager service is available now
               const BTSSppInstanceId sppInstance(0);
               const BTSUuid uuid;
               const BTSMasInstanceId masInstance(0);
               const BTSMasInstanceName masName;

               _callback->waitingForServiceResult(bts2IpcMsgList, bts2AppMsgList, messageItem, it->first, BTS_PROTO_PAN, sppInstance, uuid, masInstance, masName, BTS_REQ_SUCCESS);
            }
         }

         // request ConnMan services
         createGetConnManServicesMsg(bts2IpcMsgList);
      }
   }
}

void PanConnectDisconnectGenivi::handleConnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& failedAddress, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" handleConnectResult: device=%s", device.c_str()));

   BTSBDAddress address;
   BTSProtocolId protocol;
   if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, device))
   {
      // should never happen
      return;
   }

   const BTSSppInstanceId sppInstance(0);
   const BTSUuid uuid;
   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;
   BTSRequestResult result(BTS_REQ_FAILED);

   // check error code
   if(BTS_IPC_SUCCESS == errorCode)
   {
      result = BTS_REQ_SUCCESS;
   }
   else if(BTS_IPC_BUSY == errorCode)
   {
#ifdef ENABLE_BUSY
      result = BTS_REQ_BUSY;
#endif
   }
   else if(BTS_IPC_RETRY_ABORTED == errorCode)
   {
      result = BTS_REQ_CONNECT_ABORTED;
   }
   else
   {
      // other error codes
   }

#ifdef PAN_OLD_INTERMEDIATE_STATE_UPD
   // update connecting before result in case of successful result
   if(BTS_IPC_SUCCESS == errorCode)
   {
      _callback->connecting(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);
   }
#endif

   // forward connect result
   if(true == _callback->connectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, result))
   {
      connectFailed = true;
      failedAddress = address;
   }
}

void PanConnectDisconnectGenivi::handleInterfaceName(IN const BTSObjectPath& device, IN const BTSDeviceName& interface)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" handleInterfaceName: device=%s", device.c_str()));

   if(true == _serviceStatusConnManService.getEnabled())
   {
      BTSBDAddress address;
      BTSProtocolId protocol;
      if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, device))
      {
         // should never happen
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }

      const BTSSppInstanceId sppInstance(0);
      const BTSUuid uuid;
      const BTSMasInstanceId masInstance(0);
      const BTSMasInstanceName masName;

      // forward device name
      ::std::vector< Bts2Ipc_BaseMessage* > tmpBts2IpcMsgList;
      ::std::vector< Bts2App_BaseMessage* > tmpBts2AppMsgList;
       BTSHandleIpc2BtsMessageItem messageItem;
      _callback->updateVirtualDeviceName(tmpBts2IpcMsgList, tmpBts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, interface);
   }
}

void PanConnectDisconnectGenivi::updateConnectStatus(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if(BTS_PROTO_PAN != protocol)
   {
      return;
   }

   if(false == _callback->isPanViaConnMan())
   {
      return;
   }

   ETG_TRACE_USR3((" updateConnectStatus: address=%s", address.c_str()));

   if(true == _serviceStatusConnManService.getEnabled())
   {
      // optional information, contains e.g. page timeout information

      // forward only in case of failed connect
      if(BTS_STATUS_CODE_SUCCESS == status)
      {
         return;
      }

      // forward failed connect reason
      const BTSSppInstanceId sppInstance(0);
      const BTSUuid uuid;
      const BTSMasInstanceId masInstance(0);
      const BTSMasInstanceName masName;
      BTSDisconnectReason reason(BTS_DISCONNECT_REASON_PROTO_CONNECT_FAILED);

      // check for page timeout, aborted, missing key
      if(BTS_STATUS_CODE_PAGE_TIMEOUT == status)
      {
         reason = BTS_DISCONNECT_REASON_OUT_OF_RANGE; // e.g. device is out of range
      }
      else if(BTS_STATUS_CODE_ABORTED == status)
      {
         reason = BTS_DISCONNECT_REASON_PROTO_CONNECT_ABORTED; // e.g. connect was aborted/cancelled
      }
      else if(BTS_STATUS_CODE_PIN_OR_KEY_MISSING == status)
      {
         reason = BTS_DISCONNECT_REASON_MISSING_LINK_KEY; // e.g. pairing information on BT device was removed
      }

      _callback->updateFailedConnectReason(address, protocol, sppInstance, uuid, masInstance, masName, reason);
   }
}

void PanConnectDisconnectGenivi::handleConnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if(BTS_PROTO_PAN != protocol)
   {
      return;
   }

   if(false == _callback->isPanViaConnMan())
   {
      return;
   }

   ETG_TRACE_USR3((" handleConnectInd: address=%s", address.c_str()));

   if(true == _serviceStatusConnManService.getEnabled())
   {
      // optional information

      // forward connect indication, use success or failed as result, reason for failed connect was forwarded via updateFailedConnectReason() before
      const BTSSppInstanceId sppInstance(0);
      const BTSUuid uuid;
      const BTSMasInstanceId masInstance(0);
      const BTSMasInstanceName masName;
      const BTSRequestResult result((BTS_STATUS_CODE_SUCCESS == status) ? BTS_REQ_SUCCESS : BTS_REQ_FAILED);

      _callback->connectIndication(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, result);
   }
}

void PanConnectDisconnectGenivi::handleServiceState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& failedAddress, IN const BTSObjectPath& device, IN const ::ccdbusif::connman::ServiceState serviceState)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" handleServiceState: device=%s", device.c_str()));

   if(true == _serviceStatusConnManService.getEnabled())
   {
      BTSBDAddress address;
      BTSProtocolId protocol;
      if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, device))
      {
         // should never happen
         return;
      }

      const BTSSppInstanceId sppInstance(0);
      const BTSUuid uuid;
      const BTSMasInstanceId masInstance(0);
      const BTSMasInstanceName masName;
      const ::ccdbusif::connman::ServiceState newState(serviceState);

      // get previous connection status
      const ::ccdbusif::connman::ServiceState prevState(getConnectionStatus(*_callback, address, protocol, sppInstance, uuid, masInstance, masName));

      // store connection status
      setConnectionStatus(*_callback, address, protocol, sppInstance, uuid, masInstance, masName, newState);

      // compare old and new service state
      if(prevState == newState)
      {
         // same state => ignore
         return;
      }

#ifdef PAN_OLD_INTERMEDIATE_STATE_UPD
      // get old connected state
      const bool oldConnected(getConnectedState(*_callback, address, protocol, sppInstance, uuid, masInstance, masName));
#endif

      // get new connected state
      bool newConnected(false);
      bool newConnecting(false);
      bool newDisconnecting(false);

#ifdef PAN_OLD_INTERMEDIATE_STATE_UPD
      if(false == isNewConnectionState(newConnected, newConnecting, newDisconnecting, newState))
      {
         // ignore because intermediate state
         return;
      }

      // compare old and new connected state
      if(oldConnected == newConnected)
      {
         // same state => ignore
         return;
      }
#else
      bool oldConnected(false);
      bool oldConnecting(false);
      bool oldDisconnecting(false);

      const bool newUpdateStatus(isNewConnectionState(newConnected, newConnecting, newDisconnecting, newState));
      const bool oldUpdateStatus(isNewConnectionState(oldConnected, oldConnecting, oldDisconnecting, prevState));

      if((true == newConnecting) && (false == oldConnecting))
      {
         // forward connecting
         forwardIntermediateState(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName, BTS_SERVICE_STATE_CONNECTING);
         // and return
         return;
      }
      else if((true == newDisconnecting) && (false == oldDisconnecting))
      {
         // forward disconnecting
         forwardIntermediateState(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName, BTS_SERVICE_STATE_DISCONNECTING);
         // and return
         return;
      }
      else
      {
         // check connected/disconnected states
         if((true == newUpdateStatus) && (true == oldUpdateStatus) && (newConnected != oldConnected))
         {
            // change from connected to disconnected or vice versa => continue and update status
         }
         else if((true == newUpdateStatus) && (false == oldUpdateStatus))
         {
            // change from intermediate state to connected or disconnected => continue and update status
         }
         else
         {
            // state did not change => ignore
            return;
         }
      }
#endif

      // forward new connection status
      const BTSConnectionStatus connStatus((true == newConnected) ? BTS_CONN_CONNECTED : BTS_CONN_DISCONNECTED);
      const BTSDisconnectReason reason(BTS_DISCONNECT_REASON_LAST);
      if(true == _callback->updateStatus(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, connStatus, reason))
      {
         connectFailed = true;
         failedAddress = address;
      }
   }
}

void PanConnectDisconnectGenivi::handleDisconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" handleDisconnectResult: device=%s", device.c_str()));

   BTSBDAddress address;
   BTSProtocolId protocol;
   if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, device))
   {
      // should never happen
      return;
   }

   const BTSSppInstanceId sppInstance(0);
   const BTSUuid uuid;
   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;
   BTSRequestResult result(BTS_REQ_FAILED);

   // check error code
   if((BTS_IPC_SUCCESS == errorCode) || (BTS_IPC_NOT_CONNECTED == errorCode))
   {
      result = BTS_REQ_SUCCESS;
   }
   else if(BTS_IPC_BUSY == errorCode)
   {
#ifdef ENABLE_BUSY
      result = BTS_REQ_BUSY;
#endif
   }
   else if(BTS_IPC_RETRY_ABORTED == errorCode)
   {
      result = BTS_REQ_DISCONNECT_ABORTED;
   }
   else
   {
      // other error codes
   }

#ifdef PAN_OLD_INTERMEDIATE_STATE_UPD
   // update disconnecting before result in case of successful result (but not in cancel connect case)
   if((BTS_IPC_SUCCESS == errorCode) && (false == isConnectOngoing(*_callback, address, protocol, sppInstance, uuid, masInstance, masName)))
   {
      _callback->disconnecting(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);
   }
#else
   // check for successful disconnect and cancel situation
   if((BTS_IPC_SUCCESS == errorCode) && (true == isConnectOngoing(*_callback, address, protocol, sppInstance, uuid, masInstance, masName)))
   {
      // get previous connection status
      const ::ccdbusif::connman::ServiceState prevState(getConnectionStatus(*_callback, address, protocol, sppInstance, uuid, masInstance, masName));
      bool oldConnected(false);
      bool oldConnecting(false);
      bool oldDisconnecting(false);
      const bool oldUpdateStatus(isNewConnectionState(oldConnected, oldConnecting, oldDisconnecting, prevState));

      // check previous connection state
      if((true == oldConnecting) || ((true == oldUpdateStatus) && (true == oldConnected)))
      {
         // last indicated state is connecting or connected, disconnected state will follow, indicate disconnecting to upper layer, this will force connect SM to wait for disconnected state
         forwardIntermediateState(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName, BTS_SERVICE_STATE_DISCONNECTING);
      }
   }
   // check for successful disconnect and disconnect situation
   else if((BTS_IPC_SUCCESS == errorCode) && (true == isDisconnectOngoing(*_callback, address, protocol, sppInstance, uuid, masInstance, masName)))
   {
      // get previous connection status
      const ::ccdbusif::connman::ServiceState prevState(getConnectionStatus(*_callback, address, protocol, sppInstance, uuid, masInstance, masName));
      bool oldConnected(false);
      bool oldConnecting(false);
      bool oldDisconnecting(false);
      const bool oldUpdateStatus(isNewConnectionState(oldConnected, oldConnecting, oldDisconnecting, prevState));

      // check previous connection state
      if((true == oldUpdateStatus) && (true == oldConnected))
      {
         // last indicated state is connected, disconnecting and disconnected states will follow, indicate disconnecting to upper layer, this will force disconnect SM to wait for disconnected state
         forwardIntermediateState(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName, BTS_SERVICE_STATE_DISCONNECTING);
      }
   }
#endif

   // forward disconnect result
   _callback->disconnectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, result);
}

void PanConnectDisconnectGenivi::updateDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSInternalDisconnectReason reason)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if(BTS_PROTO_PAN != protocol)
   {
      return;
   }

   if(false == _callback->isPanViaConnMan())
   {
      return;
   }

   ETG_TRACE_USR3((" updateDisconnectReason: address=%s", address.c_str()));

   if(true == _serviceStatusConnManService.getEnabled())
   {
      // optional information, only called in case of successful disconnect

      // forward disconnect reason
      const BTSSppInstanceId sppInstance(0);
      const BTSUuid uuid;
      const BTSMasInstanceId masInstance(0);
      const BTSMasInstanceName masName;

      _callback->updateDisconnectReason(address, protocol, sppInstance, uuid, masInstance, masName, convertInternalDisconnectReason2Public(reason));
   }
}

void PanConnectDisconnectGenivi::handleDisconnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if(BTS_PROTO_PAN != protocol)
   {
      return;
   }

   if(false == _callback->isPanViaConnMan())
   {
      return;
   }

   ETG_TRACE_USR3((" handleDisconnectInd: address=%s", address.c_str()));

   if(true == _serviceStatusConnManService.getEnabled())
   {
      // optional information

      // forward disconnect indication, use success or failed as result, disconnect reason was forwarded via updateDisconnectReason() before
      const BTSSppInstanceId sppInstance(0);
      const BTSUuid uuid;
      const BTSMasInstanceId masInstance(0);
      const BTSMasInstanceName masName;
      const BTSRequestResult result((BTS_STATUS_CODE_SUCCESS == status) ? BTS_REQ_SUCCESS : BTS_REQ_FAILED);

      _callback->disconnectIndication(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, result);
   }
}

void PanConnectDisconnectGenivi::handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   // check timer id
   for(::std::map< BTSBDAddress, PanConnectDisconnectDataGenivi >::iterator it = _connectDisconnectDataList.begin(); it != _connectDisconnectDataList.end(); ++it)
   {
      PanConnectDisconnectDataGenivi& entry = it->second;

      // check for active timer
      if(true == entry.serviceConnManManagerTimer.compare(timerId))
      {
         // timeout happened for availability of ConnMan manager service
         const BTSSppInstanceId sppInstance(0);
         const BTSUuid uuid;
         const BTSMasInstanceId masInstance(0);
         const BTSMasInstanceName masName;

         _callback->waitingForServiceResult(bts2IpcMsgList, bts2AppMsgList, messageItem, it->first, BTS_PROTO_PAN, sppInstance, uuid, masInstance, masName, BTS_REQ_FAILED);

         return;
      }
      else if(true == entry.serviceConnManServiceTimer.compare(timerId))
      {
         // timeout happened for availability of ConnMan service service
         const BTSSppInstanceId sppInstance(0);
         const BTSUuid uuid;
         const BTSMasInstanceId masInstance(0);
         const BTSMasInstanceName masName;

         _callback->waitingForProtocolResult(bts2IpcMsgList, bts2AppMsgList, messageItem, it->first, BTS_PROTO_PAN, sppInstance, uuid, masInstance, masName, BTS_REQ_FAILED);

         return;
      }
   }
}

void PanConnectDisconnectGenivi::createGetConnManServicesMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList) const
{
   Bts2Ipc_GetConnManServices* msg = ptrNew_Bts2Ipc_GetConnManServices();
   if(0 != msg)
   {
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void PanConnectDisconnectGenivi::createConnManServiceConnectMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& deviceAddress, IN const BTSObjectPath& service) const
{
   Bts2Ipc_ConnManServiceConnect* msg = ptrNew_Bts2Ipc_ConnManServiceConnect();
   if(0 != msg)
   {
      msg->setBDAddress(deviceAddress);
      msg->setDevice(service);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void PanConnectDisconnectGenivi::createConnManServiceDisconnectMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& deviceAddress, IN const BTSObjectPath& service) const
{
   Bts2Ipc_ConnManServiceDisconnect* msg = ptrNew_Bts2Ipc_ConnManServiceDisconnect();
   if(0 != msg)
   {
      msg->setBDAddress(deviceAddress);
      msg->setDevice(service);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void PanConnectDisconnectGenivi::checkForDataComplete(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   // get data entry
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   // get stored availability information
   const bool oldAvailableL1(entry.info.getBit(ProtocolManagerData::PROTOCOL_AVAILABLE));
   const bool oldAvailableL2(entry.info.getBit(ProtocolManagerData::PROTOCOL_AVAILABLE_L2));

   // get new availability information
   bool newAvailableL1(true);
   bool newAvailableL2(true);

   if(false == isDeviceAvailable(address))
   {
      newAvailableL1 = false;
      newAvailableL2 = false;
   }
   else if(false == entry.info.getBit(ProtocolManagerData::PROTOCOL_ADDED))
   {
      newAvailableL1 = false;
      newAvailableL2 = false;
   }
   else if(false == getDbusServiceAvailability(entry.secondaryInfo))
   {
      newAvailableL2 = false;
   }

   // indicate only changed states
   if(oldAvailableL1 != newAvailableL1)
   {
      ProtocolManagerData::ProtocolAvailableInfo protocolAvailableInfo;
      protocolAvailableInfo.setBit(ProtocolManagerData::PROTOCOL_AVAILABLE_LEVEL_1);

      if(true == newAvailableL1)
      {
         callback.protocolAvailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, protocolAvailableInfo);
      }
      else
      {
         callback.protocolUnavailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, protocolAvailableInfo);
      }
   }

   // indicate only changed states
   if(oldAvailableL2 != newAvailableL2)
   {
      ProtocolManagerData::ProtocolAvailableInfo protocolAvailableInfo;
      protocolAvailableInfo.setBit(ProtocolManagerData::PROTOCOL_AVAILABLE_LEVEL_2);

      if(true == newAvailableL2)
      {
         callback.protocolAvailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, protocolAvailableInfo);

         PanConnectDisconnectDataGenivi& entry = checkDataList(address);

         // check for active timer
         if(true == isTimerActive(entry.serviceConnManServiceTimer))
         {
            stopTimer(entry.serviceConnManServiceTimer);

            // ConnMan service service is available now
            const BTSSppInstanceId sppInstance(0);
            const BTSUuid uuid;
            const BTSMasInstanceId masInstance(0);
            const BTSMasInstanceName masName;

            callback.waitingForProtocolResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, BTS_REQ_SUCCESS);
         }
      }
      else
      {
         callback.protocolUnavailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, protocolAvailableInfo);
      }
   }
}

bool PanConnectDisconnectGenivi::isNewConnectionState(OUT bool& connected, OUT bool& connecting, OUT bool& disconnecting, IN const ::ccdbusif::connman::ServiceState state) const
{
   /*
    * according connman documentation (service-api.txt + overview-api.txt):
    * - idle: The basic state of every service is "idle". This means that this service is not in use at all at the moment. It also is not attempting to connect or do anything else. => disconnected
    * - association: The "association" state indicates that this service tries to establish a low-level connection to the network. For example associating/connecting with a WiFi access point. => connecting
    * - configuration: With the "configuration" state the service indicates that it is trying to retrieve/configure IP settings. => connecting
    * - ready: The "ready" state signals a successfully connected device. => connected
    * - online: "online" signals that an Internet connection is available and has been verified. => connected
    * - disconnect: With the "disconnect" state a service indicates that it is going to terminate the current connection and will return to the "idle" state. => disconnecting
    * - failure: In addition a "failure" state indicates a wrong behavior. It is similar to the "idle" state since the service is not connected. => disconnected
    */

   bool process(false);

   // check given state
   switch(state)
   {
      case ::ccdbusif::connman::SERVICE_STATE_IDLE:
         process = true;
         connected = false;
         break;
      case ::ccdbusif::connman::SERVICE_STATE_FAILURE:
         process = true;
         connected = false;
         break;
      case ::ccdbusif::connman::SERVICE_STATE_ASSOCIATION:
         connecting = true;
         break;
      case ::ccdbusif::connman::SERVICE_STATE_CONFIGURATION:
         connecting = true;
         break;
      case ::ccdbusif::connman::SERVICE_STATE_READY:
         process = true;
         connected = true;
         break;
      case ::ccdbusif::connman::SERVICE_STATE_DISCONNECT:
         disconnecting = true;
         break;
      case ::ccdbusif::connman::SERVICE_STATE_ONLINE:
         process = true;
         connected = true;
         break;
      case ::ccdbusif::connman::SERVICE_STATE_UNKNOWN:
         FW_ERRMEM_ASSERT_ALWAYS();
         break;
      default:
         FW_ERRMEM_ASSERT_ALWAYS();
         break;
   }

   return process;
}

void PanConnectDisconnectGenivi::setConnectionStatus(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const ::ccdbusif::connman::ServiceState state)
{
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   setConnectionStatus(entry.secondaryInfo, state);

   // mark connection status as available
   entry.info.setBit(ProtocolManagerData::STATUS_UPDATED);
}

void PanConnectDisconnectGenivi::setConnectionStatus(OUT unsigned int& secondaryData, IN const ::ccdbusif::connman::ServiceState state) const
{
   ::ccdbusif::connman::ServiceState tmpState(state);

   if(::ccdbusif::connman::SERVICE_STATE_UNKNOWN <= tmpState)
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      tmpState = ::ccdbusif::connman::SERVICE_STATE_IDLE;
   }

   // use bits 0 .. 3
   const unsigned int stateMask(0x0F);
   const unsigned int stateValue((const unsigned int)tmpState);

   secondaryData &= ~stateMask;
   secondaryData |= (stateValue & stateMask);
}

::ccdbusif::connman::ServiceState PanConnectDisconnectGenivi::getConnectionStatus(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   return getConnectionStatus(entry.secondaryInfo);
}

::ccdbusif::connman::ServiceState PanConnectDisconnectGenivi::getConnectionStatus(IN const unsigned int secondaryData) const
{
   // use bits 0 .. 3
   const unsigned int stateMask(0x0F);

   return (::ccdbusif::connman::ServiceState)(secondaryData & stateMask);
}

void PanConnectDisconnectGenivi::setDbusServiceAvailability(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool available)
{
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   setDbusServiceAvailability(entry.secondaryInfo, available);
}

void PanConnectDisconnectGenivi::setDbusServiceAvailability(OUT unsigned int& secondaryData, IN const bool available) const
{
   // use bit 4
   const unsigned int stateMask(0x010);
   const unsigned int stateValue((const unsigned int)available);

   secondaryData &= ~stateMask;
   secondaryData |= ((stateValue << 4) & stateMask);
}

bool PanConnectDisconnectGenivi::getDbusServiceAvailability(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   return getDbusServiceAvailability(entry.secondaryInfo);
}

bool PanConnectDisconnectGenivi::getDbusServiceAvailability(IN const unsigned int secondaryData) const
{
   // use bit 4
   const unsigned int stateMask(0x010);

   return (stateMask == (secondaryData & stateMask));
}

bool PanConnectDisconnectGenivi::getConnectedState(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   const ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   return entry.connected;
}

bool PanConnectDisconnectGenivi::isConnectOngoing(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   const ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   return (0 != entry.sm.getConnectSm());
}

bool PanConnectDisconnectGenivi::isDisconnectOngoing(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   const ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   return (0 != entry.sm.getDisconnectSm());
}

bool PanConnectDisconnectGenivi::isDeviceAvailable(IN const BTSBDAddress& address) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_deviceManagerIf);

   return _deviceManagerIf->isDeviceAvailable(address);
}

void PanConnectDisconnectGenivi::startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout)
{
   ETG_TRACE_USR3((" startTimer: timeout=%u", timeout));

   FW_ERRMEM_IF_NULL_PTR_RETURN(_timerPoolIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   timer.setTimerPool(_timerPoolIf);
   timer.start(_controlIf, this, timeout);
}

void PanConnectDisconnectGenivi::stopTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" stopTimer"));

   timer.stop();

   // do not release timer
}

void PanConnectDisconnectGenivi::releaseTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" releaseTimer"));

   timer.release();
}

bool PanConnectDisconnectGenivi::isTimerActive(IN const ExtendedTimerEntry& timer) const
{
   return timer.isActive();
}

PanConnectDisconnectDataGenivi& PanConnectDisconnectGenivi::checkDataList(IN const BTSBDAddress& address)
{
   return _connectDisconnectDataList[address];
}

void PanConnectDisconnectGenivi::forwardIntermediateState(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSServiceState serviceState) const
{
   if(BTS_SERVICE_STATE_CONNECTING == serviceState)
   {
      // check for existing SM
      const ProtocolManagerData& protocolData = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);
      if(0 != protocolData.sm.getConnectSm())
      {
         callback.connecting(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);
      }
   }
   else if(BTS_SERVICE_STATE_DISCONNECTING == serviceState)
   {
      // it can happen that SM does not exist during this moment e.g. connection loss
      const ProtocolManagerData& protocolData = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);
      if((0 != protocolData.sm.getDisconnectSm()) || (0 != protocolData.sm.getConnectSm()))
      {
         callback.disconnecting(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);
      }
   }
}

} //genivi
} //btstackif
