/**
 * @file LocalAdapterModesGenivi.cpp
 *
 * @par SW-Component
 * State machine for local adapter modes
 *
 * @brief Implementation of Genivi local adapter modes state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Genivi local adapter modes state machine.
 */

#include "LocalAdapterModesGenivi.h"
#include "ILocalAdapterModesCallback.h"
#include "IBasicControl.h"
#include "FwAssert.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"

namespace btstackif {
namespace genivi {

LocalAdapterModesGenivi::LocalAdapterModesGenivi() :
_callback(0),
_controlIf(0),
_adapterDataReceived(false)
{
}

LocalAdapterModesGenivi::~LocalAdapterModesGenivi()
{
   _callback = 0;
   _controlIf = 0;
}

void LocalAdapterModesGenivi::reset(void)
{
   _adapterDataReceived = false;
}

void LocalAdapterModesGenivi::setCallback(IN ILocalAdapterModesCallback* callback)
{
   _callback = callback;

   FW_NORMAL_ASSERT(0 != _callback);
}

void LocalAdapterModesGenivi::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);
}

void LocalAdapterModesGenivi::setDiscoverable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enabled)
{
   (void)(bts2AppMsgList);

   createSetAdapterDiscoverableMsg(bts2IpcMsgList, enabled);
}

void LocalAdapterModesGenivi::setPairable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enabled)
{
   (void)(bts2AppMsgList);

   createSetAdapterPairableMsg(bts2IpcMsgList, enabled);
}

void LocalAdapterModesGenivi::setConnectable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enabled)
{
   (void)(bts2AppMsgList);

   createSetAdapterConnectableMsg(bts2IpcMsgList, enabled);
}

void LocalAdapterModesGenivi::setDiscoverableTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSDiscoverableTimeout timeout)
{
   (void)(bts2AppMsgList);

   createSetAdapterDiscoverabletimeoutMsg(bts2IpcMsgList, timeout);
}

void LocalAdapterModesGenivi::setPairableTimout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSDiscoverableTimeout timeout)
{
   (void)(bts2AppMsgList);

   createSetAdapterPairabletimeoutMsg(bts2IpcMsgList, timeout);
}

void LocalAdapterModesGenivi::getDiscoverable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   createGetAdapterDiscoverableMsg(bts2IpcMsgList);
}

void LocalAdapterModesGenivi::getPairable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   createGetAdapterPairableMsg(bts2IpcMsgList);
}

void LocalAdapterModesGenivi::getConnectable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   createGetAdapterConnectableMsg(bts2IpcMsgList);
}

void LocalAdapterModesGenivi::sendVirtualDiscoverableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enabled, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_controlIf);

   Ipc2Bts_AdapterDiscoverableUpdate* msg = ptrNew_Ipc2Bts_AdapterDiscoverableUpdate();
   if(0 != msg)
   {
      msg->setDiscoverable(enabled);
      msg->setIpcCommonErrorCode(errorCode);
   }
   _controlIf->sendInternalIpc2BtsMessage(msg);
}

void LocalAdapterModesGenivi::sendVirtualPairableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enabled, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_controlIf);

   Ipc2Bts_AdapterPairableUpdate* msg = ptrNew_Ipc2Bts_AdapterPairableUpdate();
   if(0 != msg)
   {
      msg->setPairable(enabled);
      msg->setIpcCommonErrorCode(errorCode);
   }
   _controlIf->sendInternalIpc2BtsMessage(msg);
}

void LocalAdapterModesGenivi::sendVirtualConnectableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enabled, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_controlIf);

   Ipc2Bts_AdapterConnectableUpdate* msg = ptrNew_Ipc2Bts_AdapterConnectableUpdate();
   if(0 != msg)
   {
      msg->setConnectable(enabled);
      msg->setIpcCommonErrorCode(errorCode);
   }
   _controlIf->sendInternalIpc2BtsMessage(msg);
}

void LocalAdapterModesGenivi::handleDiscoverableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force /*= false*/)
{
   (void)(response);

   FW_IF_NULL_PTR_RETURN(_callback);

   _callback->updateDiscoverable(bts2IpcMsgList, bts2AppMsgList, messageItem, enabled, (BTS_IPC_SUCCESS == errorCode), force);
}

void LocalAdapterModesGenivi::handlePairableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force /*= false*/)
{
   (void)(response);

   FW_IF_NULL_PTR_RETURN(_callback);

   _callback->updatePairable(bts2IpcMsgList, bts2AppMsgList, messageItem, enabled, (BTS_IPC_SUCCESS == errorCode), force);
}

void LocalAdapterModesGenivi::handleConnectableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force /*= false*/)
{
   (void)(response);

   FW_IF_NULL_PTR_RETURN(_callback);

   _callback->updateConnectable(bts2IpcMsgList, bts2AppMsgList, messageItem, enabled, (BTS_IPC_SUCCESS == errorCode), force);
}

void LocalAdapterModesGenivi::handleDiscoverableTimeoutUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoverableTimeout timeout, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force /*= false*/)
{
   (void)(response);

   FW_IF_NULL_PTR_RETURN(_callback);

   _callback->updateDiscoverableTimeout(bts2IpcMsgList, bts2AppMsgList, messageItem, timeout, (BTS_IPC_SUCCESS == errorCode), force);
}

void LocalAdapterModesGenivi::handlePairableTimeoutUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoverableTimeout timeout, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force /*= false*/)
{
   (void)(response);

   FW_IF_NULL_PTR_RETURN(_callback);

   _callback->updatePairableTimeout(bts2IpcMsgList, bts2AppMsgList, messageItem, timeout, (BTS_IPC_SUCCESS == errorCode), force);
}

void LocalAdapterModesGenivi::handleAdapterProperties(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusPropertyList& properties, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   // is triggered by a signal or method return => check error code
   if(BTS_IPC_SUCCESS == errorCode)
   {
      if(true == _adapterDataReceived)
      {
         // adapter data was already received => ignore this message
      }
      else
      {
         // first reception of adapter data => process
         _adapterDataReceived = true;

         // store information
         for(size_t i = 0; i < properties.size(); i++)
         {
            const ::ccdbusif::evolution::AdapterProperty property = (::ccdbusif::evolution::AdapterProperty)properties[i].propEnum;

            switch(property)
            {
               case ::ccdbusif::evolution::ADAPTER_DISCOVERABLE:
               {
                  setInitialDiscoverableMode(properties[i].propData.getBool());
                  break;
               }
               case ::ccdbusif::evolution::ADAPTER_DISCOVERABLETIMEOUT:
               {
                  break;
               }
               case ::ccdbusif::evolution::ADAPTER_PAIRABLE:
               {
                  setInitialPairableMode(properties[i].propData.getBool());
                  break;
               }
               case ::ccdbusif::evolution::ADAPTER_PAIRABLETIMEOUT:
               {
                  break;
               }
               case ::ccdbusif::evolution::ADAPTER_CONNECTABLE:
               {
                  setInitialConnectableMode(properties[i].propData.getBool());
                  break;
               }
               default:
               {
                  break;
               }
            }
         }
      }
   }
}

void LocalAdapterModesGenivi::setInitialDiscoverableMode(IN const bool enabled)
{
   FW_IF_NULL_PTR_RETURN(_callback);

   _callback->setInitialStackDiscoverableMode(enabled);
}

void LocalAdapterModesGenivi::setInitialPairableMode(IN const bool enabled)
{
   FW_IF_NULL_PTR_RETURN(_callback);

   _callback->setInitialStackPairableMode(enabled);
}

void LocalAdapterModesGenivi::setInitialConnectableMode(IN const bool enabled)
{
   FW_IF_NULL_PTR_RETURN(_callback);

   _callback->setInitialStackConnectableMode(enabled);
}

void LocalAdapterModesGenivi::createSetAdapterDiscoverableMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const bool enabled) const
{
   Bts2Ipc_SetAdapterDiscoverable* msg = ptrNew_Bts2Ipc_SetAdapterDiscoverable();
   if(0 != msg)
   {
      msg->setDiscoverable(enabled);
      msg->setResponseMessageFlag(true);
      bts2IpcMsgList.push_back(msg);
   }
}

void LocalAdapterModesGenivi::createSetAdapterPairableMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const bool enabled) const
{
   Bts2Ipc_SetAdapterPairable* msg = ptrNew_Bts2Ipc_SetAdapterPairable();
   if(0 != msg)
   {
      msg->setPairable(enabled);
      msg->setResponseMessageFlag(true);
      bts2IpcMsgList.push_back(msg);
   }
}

void LocalAdapterModesGenivi::createSetAdapterConnectableMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const bool enabled) const
{
   Bts2Ipc_SetAdapterConnectable* msg = ptrNew_Bts2Ipc_SetAdapterConnectable();
   if(0 != msg)
   {
      msg->setConnectable(enabled);
      msg->setResponseMessageFlag(true);
      bts2IpcMsgList.push_back(msg);
   }
}

void LocalAdapterModesGenivi::createSetAdapterDiscoverabletimeoutMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSDiscoverableTimeout timeout) const
{
   Bts2Ipc_SetAdapterDiscoverabletimeout* msg = ptrNew_Bts2Ipc_SetAdapterDiscoverabletimeout();
   if(0 != msg)
   {
      msg->setDiscoverableTimeout(timeout);
      msg->setResponseMessageFlag(true);
      bts2IpcMsgList.push_back(msg);
   }
}

void LocalAdapterModesGenivi::createSetAdapterPairabletimeoutMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSDiscoverableTimeout timeout) const
{
   Bts2Ipc_SetAdapterPairabletimeout* msg = ptrNew_Bts2Ipc_SetAdapterPairabletimeout();
   if(0 != msg)
   {
      msg->setPairableTimeout(timeout);
      msg->setResponseMessageFlag(true);
      bts2IpcMsgList.push_back(msg);
   }
}

void LocalAdapterModesGenivi::createGetAdapterDiscoverableMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList) const
{
   Bts2Ipc_GetAdapterDiscoverable* msg = ptrNew_Bts2Ipc_GetAdapterDiscoverable();
   if(0 != msg)
   {
      msg->setResponseMessageFlag(true);
      bts2IpcMsgList.push_back(msg);
   }
}

void LocalAdapterModesGenivi::createGetAdapterPairableMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList) const
{
   Bts2Ipc_GetAdapterPairable* msg = ptrNew_Bts2Ipc_GetAdapterPairable();
   if(0 != msg)
   {
      msg->setResponseMessageFlag(true);
      bts2IpcMsgList.push_back(msg);
   }
}

void LocalAdapterModesGenivi::createGetAdapterConnectableMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList) const
{
   Bts2Ipc_GetAdapterConnectable* msg = ptrNew_Bts2Ipc_GetAdapterConnectable();
   if(0 != msg)
   {
      msg->setResponseMessageFlag(true);
      bts2IpcMsgList.push_back(msg);
   }
}

} //genivi
} //btstackif
