/**
 * @file LinkKeyGenivi.cpp
 *
 * @par SW-Component
 * State machine for link key
 *
 * @brief Implementation of Genivi link key state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Genivi link key state machine.
 */

#include "LinkKeyGenivi.h"
#include "ILinkKeyCallback.h"
#include "IObjectPathManagerGenivi.h"
#include "FwAssert.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"

namespace btstackif {
namespace genivi {

LinkKeyGenivi::LinkKeyGenivi() :
_callback(0),
_objectPathManagerIf(0),
_tokenList(),
_pendingList()
{
}

LinkKeyGenivi::~LinkKeyGenivi()
{
   _callback = 0;
   _objectPathManagerIf = 0;
}

void LinkKeyGenivi::reset(void)
{
   _tokenList.clear();
   _pendingList.clear();
}

void LinkKeyGenivi::setCallback(IN ILinkKeyCallback* callback)
{
   _callback = callback;

   FW_NORMAL_ASSERT(0 != _callback);
}

void LinkKeyGenivi::setLinkKey(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSLinkKeyType linkKeyType, IN const BTSLinkKey& linkKey, IN const BTSDLinkKey& dLinkKey)
{
   (void)(bts2AppMsgList);
   (void)(linkKeyType);
   (void)(dLinkKey);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath device;
   if(false == _objectPathManagerIf->getObjectPath4Address(device, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   ::std::map< BTSObjectPath, act_t >::iterator it = _tokenList.find(device);
   if(_tokenList.end() != it)
   {
      // entry is available
      createRequestLinkkeyResMsg(bts2IpcMsgList, address, linkKey, it->second);

      _tokenList.erase(it);
   }
   else
   {
      // entry is not available
   }

   // check if given device is in pending list (could happen if first link key request is not answered by application); if found remove
   removeDevice(device);
}

void LinkKeyGenivi::setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager)
{
   _objectPathManagerIf = objectPathManager;

   FW_NORMAL_ASSERT(0 != _objectPathManagerIf);
}

void LinkKeyGenivi::handleRequestLinkkey(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const act_t token)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // store token
   ::std::map< BTSObjectPath, act_t >::iterator it = _tokenList.find(device);
   if(_tokenList.end() != it)
   {
      // entry is already stored => should never happen
      _tokenList[device] = token; // update and continue with latest request
   }
   else
   {
      // add entry
      _tokenList[device] = token;
   }

   // check if mapping is available
   BTSBDAddress address;
   if(true == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // we can directly forward
      _callback->indicateLinkKeyRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
   }
   else
   {
      // mapping is not available => add to pending list
      if(true == isDevicePending(device))
      {
         // entry is already stored => should never happen
      }
      else
      {
         // add entry
         _pendingList.push_back(device);
      }
   }
}

bool LinkKeyGenivi::handleCancelRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool sent, IN const act_t token)
{
   (void)(bts2AppMsgList);
   (void)(messageItem);

   /*
    * Cancel request will be triggered by Evolution in following scenarios:
    * - timeout RequestLinkkey: 25s => will be handled by failed connect
    */

   // but ensure that answer is sent
   if(false == sent)
   {
      createCancelRes(bts2IpcMsgList, token, true);
   }

   return true;
}

bool LinkKeyGenivi::handleCancelRequestRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool sent, IN const act_t token)
{
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(device);

   /*
    * Cancel request will be triggered by Evolution in following scenarios:
    * - timeout RequestLinkkey: 25s => will be handled by failed connect
    */

   // but ensure that answer is sent
   if(false == sent)
   {
      createCancelRequestRes(bts2IpcMsgList, token, true);
   }

   return true;
}

void LinkKeyGenivi::deviceAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath device;
   if(false == _objectPathManagerIf->getObjectPath4Address(device, address))
   {
      // should never happen
      return;
   }

   // check if given device is in pending list
   if(true == isDevicePending(device))
   {
      // entry is available => forward request and remove
      _callback->indicateLinkKeyRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
      removeDevice(device);
   }
}

void LinkKeyGenivi::deviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath device;
   if(false == _objectPathManagerIf->getObjectPath4Address(device, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // check if given device is in pending list; if found remove
   removeDevice(device);
}

void LinkKeyGenivi::deviceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
}

void LinkKeyGenivi::deviceUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
}

void LinkKeyGenivi::deviceConnectionStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSStatusTransition aclTransition, IN const bool aclConnected, IN const BTSStatusTransition anyProfileTransition, IN const bool anyProfileConnected)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
   (void)(aclTransition);
   (void)(aclConnected);
   (void)(anyProfileTransition);
   (void)(anyProfileConnected);
}

void LinkKeyGenivi::deviceCreationFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
   (void)(result);
}

void LinkKeyGenivi::deviceRemovalFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
   (void)(result);
}

void LinkKeyGenivi::createRequestLinkkeyResMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSLinkKey& linkKey, IN const act_t token) const
{
   Bts2Ipc_RequestLinkkeyRes* msg = ptrNew_Bts2Ipc_RequestLinkkeyRes();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setLinkKey(linkKey);
      msg->setDbusToken(token);

      bts2IpcMsgList.push_back(msg);
   }
}

void LinkKeyGenivi::createCancelRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const act_t token, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_CancelRes* msg = ptrNew_Bts2Ipc_CancelRes();
   if(0 != msg)
   {
      msg->setDbusToken(token);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void LinkKeyGenivi::createCancelRequestRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const act_t token, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_CancelRequestRes* msg = ptrNew_Bts2Ipc_CancelRequestRes();
   if(0 != msg)
   {
      msg->setDbusToken(token);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

bool LinkKeyGenivi::isDevicePending(IN const BTSObjectPath& device) const
{
   for(::std::vector< BTSObjectPath >::const_iterator it = _pendingList.begin(); it != _pendingList.end(); ++it)
   {
      if(*it == device)
      {
         return true;
      }
   }

   return false;
}

void LinkKeyGenivi::removeDevice(IN const BTSObjectPath& device)
{
   for(::std::vector< BTSObjectPath >::iterator it = _pendingList.begin(); it != _pendingList.end(); ++it)
   {
      if(*it == device)
      {
         _pendingList.erase(it);
         break;
      }
   }
}

} //genivi
} //btstackif
