/**
 * @file InquiryGenivi.cpp
 *
 * @par SW-Component
 * State machine for inquiry
 *
 * @brief Implementation of Genivi inquiry state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Genivi inquiry state machine.
 */

#include "InquiryGenivi.h"
#include "IInquiryCallback.h"
#include "IObjectPathManagerGenivi.h"
#include "FwAssert.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"

namespace btstackif {
namespace genivi {

InquiryGenivi::InquiryGenivi() :
_callback(0),
_objectPathManagerIf(0)
{
}

InquiryGenivi::~InquiryGenivi()
{
   _callback = 0;
   _objectPathManagerIf = 0;
}

void InquiryGenivi::reset(void)
{
}

void InquiryGenivi::setCallback(IN IInquiryCallback* callback)
{
   _callback = callback;

   FW_NORMAL_ASSERT(0 != _callback);
}

void InquiryGenivi::startDiscovery(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   // all necessary checks are done before

   createStartDiscoveryMsg(bts2IpcMsgList);
}

void InquiryGenivi::stopDiscovery(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   // all necessary checks are done before

   createStopDiscoveryMsg(bts2IpcMsgList);
}

void InquiryGenivi::setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager)
{
   _objectPathManagerIf = objectPathManager;

   FW_NORMAL_ASSERT(0 != _objectPathManagerIf);
}

void InquiryGenivi::handleStartDiscoveryResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_IF_NULL_PTR_RETURN(_callback);

   BTSRequestResult result(BTS_REQ_FAILED);

   if(BTS_IPC_SUCCESS == errorCode)
   {
      // start discovery request is successful
      result = BTS_REQ_SUCCESS;
   }

   _callback->startDiscoveryResult(bts2IpcMsgList, bts2AppMsgList, messageItem, result);
}

void InquiryGenivi::handleStopDiscoveryResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_IF_NULL_PTR_RETURN(_callback);

   BTSRequestResult result(BTS_REQ_FAILED);

   if(BTS_IPC_SUCCESS == errorCode)
   {
      // stop discovery request is successful
      result = BTS_REQ_SUCCESS;
   }

   _callback->stopDiscoveryResult(bts2IpcMsgList, bts2AppMsgList, messageItem, result);
}

void InquiryGenivi::handleAdapterDiscoveringUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool discovering, IN const BTSIpcCommonErrorCode errorCode, IN const bool response)
{
   (void)(response);

   FW_IF_NULL_PTR_RETURN(_callback);

   if(BTS_IPC_SUCCESS != errorCode)
   {
      // we have to ignore
      return;
   }

   BTSDiscoveringStatus status(BTS_DISCOVERING_OFF);
   if(true == discovering)
   {
      status = BTS_DISCOVERING_ON;
   }

   _callback->updateStatus(bts2IpcMsgList, bts2AppMsgList, messageItem, status);
}

void InquiryGenivi::handleDeviceFound(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoveredDeviceInfo& device)
{
   FW_IF_NULL_PTR_RETURN(_callback);

   _callback->deviceFound(bts2IpcMsgList, bts2AppMsgList, messageItem, device);
}

void InquiryGenivi::createStartDiscoveryMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList) const
{
   Bts2Ipc_StartDiscovery* msg = ptrNew_Bts2Ipc_StartDiscovery();
   if(0 != msg)
   {
      bts2IpcMsgList.push_back(msg);
   }
}

void InquiryGenivi::createStopDiscoveryMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList) const
{
   Bts2Ipc_StopDiscovery* msg = ptrNew_Bts2Ipc_StopDiscovery();
   if(0 != msg)
   {
      bts2IpcMsgList.push_back(msg);
   }
}

} //genivi
} //btstackif
