/**
 * @file SwitchBluetoothGenivi.h
 *
 * @par SW-Component
 * State machine for switch Bluetooth on/off
 *
 * @brief Implementation of Genivi switch Bluetooth on/off state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of Genivi switch Bluetooth on/off state machine.
 */

#ifndef _SWITCH_BLUETOOTH_GENIVI_H_
#define _SWITCH_BLUETOOTH_GENIVI_H_

#include "ISwitchBluetoothRequest.h"
#include "ISwitchBluetoothGenivi.h"
#include "EvolutionGeniviStackTypes.h"

namespace btstackif {
namespace genivi {

/**
 * SwitchBluetooth class.
 */
class SwitchBluetoothGenivi : public ISwitchBluetoothRequest, public ISwitchBluetoothGenivi
{
public:
   /**
    * Default constructor.
    */
   SwitchBluetoothGenivi();

   /**
    * Destructor.
    */
   virtual ~SwitchBluetoothGenivi();

   virtual void reset(void);

   virtual void setCallback(IN ISwitchBluetoothCallback* callback);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setPowered(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enabled);

   virtual void sendVirtualPoweredUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enabled, IN const BTSIpcCommonErrorCode errorCode);

   virtual void getHwVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual ISwitchBluetoothRequest* getRequestIf(void);

   virtual void handleAdapterProperties(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusPropertyList& properties, IN const BTSIpcCommonErrorCode errorCode);

   virtual void handleGetHwVersionConfirmation(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSStatusCode status, IN const BTSBtCoreVersion btVersion, IN const BTSManufacturerNameController manufacturerName, IN const BTSSubVersionController subVersion, IN const BTSExtensionInfo& extInfo);

   virtual void handleHciModeUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool hciMode, IN const BTSIpcCommonErrorCode errorCode);

   virtual void handlePoweredUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool powered, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force = false);

private:
   ISwitchBluetoothCallback* _callback; /**< callback interface */
   IBasicControl* _controlIf; /**< basic control interface */
   bool _adapterDataReceived; /**< flag for data received */
   const ::std::string _mibDataFileName; /**< path for file containing HW information */
   const ::std::string _defaultIdValue; /**< default id value */
   const ::std::string _defaultFwValue; /**< default FW version value */
   const ::std::string _chipIdName; /**< HW information entry name for chip id */
   const ::std::string _buildIdName; /**< HW information entry name for build id */
   const ::std::string _fwVersionName; /**< HW information entry name for FW version */
   const ::std::string _assignCmd; /**< assignment command in HW information */
   const ::std::string _evoVersionFileName; /**< path for file containing Evolution version */
   unsigned short int _code; /**< version info: code */
   unsigned short int _geniviVersion; /**< version info: Genivi version */
   unsigned short int _evoVersion; /**< version info: Evo version */
   ::std::map< ::std::string, BTSBtModuleId > _moduleIdList; /**< map containing module ids */
   ::std::map< BTSBtModuleId, BTSHciChipVendorId > _vendorIdList; /**< map containing vendor ids */

   void getDefaultMibData(OUT ::std::string& chipId, OUT ::std::string& buildId, OUT ::std::string& fwVersion);

   bool getDefaultMibDataItem(OUT ::std::string& item, IN char* buffer, IN const ::std::string& key, IN const size_t offset);

   void convertExtInfo(OUT ::std::string& fwExtInfo, IN const BTSExtensionInfo& extInfo);

   void addStackVersion(OUT ::std::string& fwVersion, OUT ::std::string& fwExtInfo) const;

   bool isLxcVersion(void) const;

   BTSBtModuleId getModuleId(IN const ::std::string& buildId) const;

   BTSHciChipVendorId getVendorId(IN const BTSBtModuleId moduleId) const;

   void setStackLocalBDAddress(IN const BTSBDAddress& address);

   void setStackPoweredMode(IN const bool enable);

   void setStackHciMode(IN const bool enable);

   void setVersionInfoCode(IN const unsigned short int code);

   void setVersionInfoGeniviVersion(IN const unsigned short int geniviVersion);

   void setVersionInfoEvoVersion(IN const unsigned short int evoVersion);

   void createSetAdapterPoweredMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const bool enabled) const;

   void createGetHwVersionMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList) const;
};

} //genivi
} //btstackif

#endif //_SWITCH_BLUETOOTH_GENIVI_H_
