/**
 * @file PanConnectDisconnectGenivi.h
 *
 * @par SW-Component
 * State machine for PAN connect/disconnect
 *
 * @brief Implementation of Genivi PAN connect/disconnect state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of Genivi PAN connect/disconnect state machine.
 */

#ifndef _PAN_CONNECT_DISCONNECT_GENIVI_H_
#define _PAN_CONNECT_DISCONNECT_GENIVI_H_

#include "IProtocolManagerRequest.h"
#include "IPanConnectDisconnectGenivi.h"
#include "IExtendedTimeoutHandler.h"
#include "EvolutionGeniviStackTypes.h"
#include "DbusServiceAvailability.h"
#include "PanConnectDisconnectDataGenivi.h"

namespace btstackif {
namespace genivi {

// class forward declarations
class IObjectPathManagerGenivi;

/**
 * PAN connect/disconnect class.
 */
class PanConnectDisconnectGenivi : public IProtocolManagerRequest, public IPanConnectDisconnectGenivi, public IExtendedTimeoutHandler
{
public:
   /**
    * Default constructor.
    */
   PanConnectDisconnectGenivi();

   /**
    * Destructor.
    */
   virtual ~PanConnectDisconnectGenivi();

   virtual void reset(void);

   virtual void setCallback(IN IProtocolManagerCallback* callback);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setTimerPoolIf(IN ITimerPool* timerPool);

   virtual void prepareProtocolConnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual void setDeviceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool available);

   virtual bool isServiceAvailable(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const;

   virtual void waitForServiceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual bool isServiceSearchNeeded(OUT BTSSearchType& searchType, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const;

   virtual bool isSettingUuidNeeded(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const;

   virtual void waitForProtocolAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual void connect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual void disconnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool pauseBtStreaming);

   virtual void cancel(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual void accept(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool accept);

   virtual void sendVirtualFailedConnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSIpcCommonErrorCode errorCode) const;

   virtual void sendVirtualFailedDisconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSIpcCommonErrorCode errorCode) const;

   virtual void sendVirtualConnectedUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool connected, IN const BTSIpcCommonErrorCode errorCode);

   virtual IProtocolManagerRequest* getRequestIf(void);

   virtual void setConfigurationIf(IN IConfiguration* configuration);

   virtual void setDeviceManagerIf(IN IDeviceManager* deviceManagerIf);

   virtual void setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager);

   virtual void addInterfaces(OUT ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSUserMode userMode);

   virtual void protocolAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSObjectPath& protocolObjectPath, IN const BTSProtocolId protocol);

   virtual void protocolRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& protocolObjectPath);

   virtual void updateDbusServiceAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& service, IN const BTSDbusServiceAvailability availability);

   virtual void updateDbusServiceConnManAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& service, IN const BTSDbusServiceAvailability availability);

   virtual void handleConnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& failedAddress, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode);

   virtual void handleInterfaceName(IN const BTSObjectPath& device, IN const BTSDeviceName& interface);

   virtual void updateConnectStatus(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status);

   virtual void handleConnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status);

   virtual void handleServiceState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& failedAddress, IN const BTSObjectPath& device, IN const ::ccdbusif::connman::ServiceState serviceState);

   virtual void handleDisconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode);

   virtual void updateDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSInternalDisconnectReason reason);

   virtual void handleDisconnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status);

   virtual void handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId);

private:
   IProtocolManagerCallback* _callback; /**< callback interface */
   IBasicControl* _controlIf; /**< basic control interface */
   ITimerPool* _timerPoolIf; /**< timer pool interface */
   IConfiguration* _configurationIf; /**< configuration interface */
   IDeviceManager* _deviceManagerIf; /**< device manager interface */
   IObjectPathManagerGenivi* _objectPathManagerIf; /**< object path manager interface */
   DbusServiceStatus _serviceStatusConnManManager; /**< ConnMan manager service status */
   DbusServiceStatus _serviceStatusConnManService; /**< ConnMan service service status */
   ::std::map< BTSBDAddress, PanConnectDisconnectDataGenivi > _connectDisconnectDataList; /**< map containing data information */
   const BTSTimeValue _serviceConnManManagerTimeout; /**< timeout in ms for availability of ConnMan manager service */
   const BTSTimeValue _serviceConnManServiceTimeout; /**< timeout in ms for availability of ConnMan service service */

   void createGetConnManServicesMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList) const;

   void createConnManServiceConnectMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& deviceAddress, IN const BTSObjectPath& service) const;

   void createConnManServiceDisconnectMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& deviceAddress, IN const BTSObjectPath& service) const;

   void checkForDataComplete(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   bool isNewConnectionState(OUT bool& connected, OUT bool& connecting, OUT bool& disconnecting, IN const ::ccdbusif::connman::ServiceState state) const;

   void setConnectionStatus(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const ::ccdbusif::connman::ServiceState state);

   void setConnectionStatus(OUT unsigned int& secondaryData, IN const ::ccdbusif::connman::ServiceState state) const;

   ::ccdbusif::connman::ServiceState getConnectionStatus(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const;

   ::ccdbusif::connman::ServiceState getConnectionStatus(IN const unsigned int secondaryData) const;

   void setDbusServiceAvailability(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool available);

   void setDbusServiceAvailability(OUT unsigned int& secondaryData, IN const bool available) const;

   bool getDbusServiceAvailability(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const;

   bool getDbusServiceAvailability(IN const unsigned int secondaryData) const;

   bool getConnectedState(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const;

   bool isConnectOngoing(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const;

   bool isDisconnectOngoing(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const;

   bool isDeviceAvailable(IN const BTSBDAddress& address) const;

   void startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout);

   void stopTimer(IN ExtendedTimerEntry& timer) const;

   void releaseTimer(IN ExtendedTimerEntry& timer) const;

   bool isTimerActive(IN const ExtendedTimerEntry& timer) const;

   PanConnectDisconnectDataGenivi& checkDataList(IN const BTSBDAddress& address);

   void forwardIntermediateState(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSServiceState serviceState) const;
};

} //genivi
} //btstackif

#endif //_PAN_CONNECT_DISCONNECT_GENIVI_H_
