/**
 * @file ObexConnectDisconnectGenivi.h
 *
 * @par SW-Component
 * State machine for OBEX connect/disconnect
 *
 * @brief Implementation of Genivi OBEX connect/disconnect state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of Genivi OBEX connect/disconnect state machine.
 */

#ifndef _OBEX_CONNECT_DISCONNECT_GENIVI_H_
#define _OBEX_CONNECT_DISCONNECT_GENIVI_H_

#include "IProtocolManagerRequest.h"
#include "IObexConnectDisconnectGenivi.h"
#include "IExtendedTimeoutHandler.h"
#include "EvolutionGeniviStackTypes.h"
#include "ObexConnectDisconnectDataGenivi.h"

namespace btstackif {
namespace genivi {

// class forward declarations
class IObjectPathManagerGenivi;

/**
 * OBEX connect/disconnect class.
 */
class ObexConnectDisconnectGenivi : public IProtocolManagerRequest, public IObexConnectDisconnectGenivi, public IExtendedTimeoutHandler
{
public:
   /**
    * Default constructor.
    */
   ObexConnectDisconnectGenivi();

   /**
    * Destructor.
    */
   virtual ~ObexConnectDisconnectGenivi();

   virtual void reset(void);

   virtual void setCallback(IN IProtocolManagerCallback* callback);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setTimerPoolIf(IN ITimerPool* timerPool);

   virtual void prepareProtocolConnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual void setDeviceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool available);

   virtual bool isServiceAvailable(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const;

   virtual void waitForServiceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual bool isServiceSearchNeeded(OUT BTSSearchType& searchType, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const;

   virtual bool isSettingUuidNeeded(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const;

   virtual void waitForProtocolAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual void connect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual void disconnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool pauseBtStreaming);

   virtual void cancel(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual void accept(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool accept);

   virtual void sendVirtualFailedConnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSIpcCommonErrorCode errorCode) const;

   virtual void sendVirtualFailedDisconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSIpcCommonErrorCode errorCode) const;

   virtual void sendVirtualConnectedUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool connected, IN const BTSIpcCommonErrorCode errorCode);

   virtual IProtocolManagerRequest* getRequestIf(void);

   virtual void setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetoothIf);

   virtual void setDeviceManagerIf(IN IDeviceManager* deviceManagerIf);

   virtual void setServiceSearchIf(IN IServiceSearch* serviceSearchIf);

   virtual void setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager);

   virtual void protocolAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSObjectPath& protocolObjectPath, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance);

   virtual void protocolRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& protocolObjectPath);

   virtual void handleCreateSessionResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& failedAddress, IN const BTSBDAddress& remoteAddress, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance, IN const BTSObjectPath& session, IN const BTSIpcCommonErrorCode errorCode);

   virtual void updateConnectStatus(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status);

   virtual void handleConnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status);

   virtual void handleRemoveSessionResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& remoteAddress, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance, IN const BTSIpcCommonErrorCode errorCode);

   virtual void handleCancelSessionResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& remoteAddress, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance, IN const BTSIpcCommonErrorCode errorCode);

   virtual void updateDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSInternalDisconnectReason reason);

   virtual void handleDisconnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status);

   virtual void setTriggerIgnoreProtocolAdded(IN const bool enable);

   virtual void handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId);

private:
   IProtocolManagerCallback* _callback; /**< callback interface */
   IBasicControl* _controlIf; /**< basic control interface */
   ITimerPool* _timerPoolIf; /**< timer pool interface */
   ISwitchBluetooth* _switchBluetoothIf; /**< switch Bluetooth interface */
   IDeviceManager* _deviceManagerIf; /**< device manager interface */
   IServiceSearch* _serviceSearchIf; /**< service search interface */
   IObjectPathManagerGenivi* _objectPathManagerIf; /**< object path manager interface */
   ::std::map< BTSProtocolBaseEntry, ObexConnectDisconnectDataGenivi > _connectDisconnectDataList; /**< map containing data information */
   const BTSTimeValue _statusUpdateTimeout; /**< timeout in ms for availability of SPP service service */

   bool _testTriggerIgnoreProtocolAdded; /**< flag for ignore protocol added */

   void createCreateSessionRequestMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& remoteAddress, IN const BTSBDAddress& localAddress, IN const BTSProtocolId protocolId, IN const BTSMasInstanceId instance) const;

   void createRemoveSessionRequestMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& remoteAddress, IN const BTSProtocolId protocolId, IN const BTSMasInstanceId instance, IN const BTSObjectPath& session) const;

   void createCancelSessionRequestMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& remoteAddress, IN const BTSBDAddress& localAddress, IN const BTSProtocolId protocolId, IN const BTSMasInstanceId instance) const;

   void checkForDataComplete(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const;

   bool isDeviceAvailable(IN const BTSBDAddress& address) const;

   bool findMatchingConnectMasInstance(OUT BTSMasInstanceId& masInstance, OUT BTSMasInstanceName& masName, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol) const;

   bool findMatchingDisconnectMasInstance(OUT BTSMasInstanceId& masInstance, OUT BTSMasInstanceName& masName, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol) const;

   void getMasInstanceName(OUT BTSMasInstanceName& instanceName, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId instanceId) const;

   bool isAnyMasInstanceConnected(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address) const;

   void checkForStartingStatusTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance);

   void startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout);

   void stopTimer(IN ExtendedTimerEntry& timer) const;

   void releaseTimer(IN ExtendedTimerEntry& timer) const;

   bool isTimerActive(IN const ExtendedTimerEntry& timer) const;

   ObexConnectDisconnectDataGenivi& checkDataList(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance);
};

} //genivi
} //btstackif

#endif //_OBEX_CONNECT_DISCONNECT_GENIVI_H_
