/**
 * @file IServiceSearchGenivi.h
 *
 * @par SW-Component
 * State machine for service search
 *
 * @brief Interface definition for Genivi service search state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the Genivi interface for service search state machine.
 */

#ifndef _I_SERVICE_SEARCH_GENIVI_H_
#define _I_SERVICE_SEARCH_GENIVI_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

namespace genivi {

// class forward declarations
class IObjectPathManagerGenivi;

/**
 * Interface definition.
 */
class IServiceSearchGenivi
{
public:
   /**
    * Destructor.
    */
   virtual ~IServiceSearchGenivi() {}

   /**
    * Set object path manager interface.
    *
    * @param[in] objectPathManager: object path manager interface
    */
   virtual void setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager) = 0;

   /**
    * Handle result for start service search.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[out] connectFailed: flag indicating that ACL connect failed
    * @param[out] address: address of device the ACL connect failed to
    * @param[in] device: device object path
    * @param[in] errorCode: error code (result)
    */
   virtual void handleDiscoverServicesResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& address, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Handle result for cancel service search.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: device object path
    * @param[in] errorCode: error code (result)
    */
   virtual void handleCancelDiscoveryResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Setter for active SDP search after successful pairing (automatic search started by Bluetooth stack).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: device object path
    * @param[in] enable: enable flag
    * @param[in] success: success flag
    */
   virtual void setSdpSearchActiveAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool enable, IN const bool success) = 0;

   /**
    * Start SDP search after pairing (BtStackIf internal search).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: device object path
    * @param[in] success: success flag (result)
    *
    * @return = true: manual SDP search was triggered,
    * @return = false: no SDP search was triggered
    */
   virtual bool startSdpSearchAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool success) = 0;

   /**
    * Start SDP search after pairing (BtStackIf internal search).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: device object path
    *
    * @return = true: manual SDP search was triggered,
    * @return = false: no SDP search was triggered
    */
   virtual bool startSdpSearchAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device) = 0;

   /**
    * Set current pairing status.
    *
    * @param[in] device: device object path
    * @param[in] success: pairing status
    */
   virtual void setPairingStatus(IN const BTSObjectPath& device, IN const bool success) = 0;

   /**
    * Update uuids data.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result (needed for get requests)
    * @param[in] response: response flag
    * @param[in] device: device object path
    * @param[in] supportedServices: supported services
    * @param[in] uuidList: list with UUIDs (SPP capabilities)
    */
   virtual void updateDeviceUuids(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSSupportedServices& supportedServices, IN const BTSUuidList& uuidList) = 0;

   /**
    * Update protocol version.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result (needed for get requests)
    * @param[in] response: response flag
    * @param[in] device: device object path
    * @param[in] version: protocol version
    */
   virtual void updateServiceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSProfileVersion version) = 0;

   /**
    * Update vendor id (DID).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result (needed for get requests)
    * @param[in] response: response flag
    * @param[in] device: device object path
    * @param[in] vendorId: vendor id
    */
   virtual void updateDeviceVendorId(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSVendorId vendorId) = 0;

   /**
    * Update vendor id source (DID).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result (needed for get requests)
    * @param[in] response: response flag
    * @param[in] device: device object path
    * @param[in] vendorIdSource: vendor id source
    */
   virtual void updateDeviceVendorIdSource(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSVendorIdSource vendorIdSource) = 0;

   /**
    * Update product id (DID).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result (needed for get requests)
    * @param[in] response: response flag
    * @param[in] device: device object path
    * @param[in] productId: product id
    */
   virtual void updateDeviceProductId(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSProductId productId) = 0;

   /**
    * Update version (DID).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result (needed for get requests)
    * @param[in] response: response flag
    * @param[in] device: device object path
    * @param[in] version: version
    */
   virtual void updateDeviceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSVersion version) = 0;

   /**
    * Update PBAP info.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result (needed for get requests)
    * @param[in] response: response flag
    * @param[in] device: device object path
    * @param[in] record: SDP record
    * @param[in] valid: valid flag
    */
   virtual void updatePbapInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSPbapSdpRecord& record, IN const bool valid) = 0;

   /**
    * Update MAP info.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result (needed for get requests)
    * @param[in] response: response flag
    * @param[in] device: device object path
    * @param[in] recordList: SDP record list
    * @param[in] valid: valid flag
    */
   virtual void updateMapInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSMapSdpRecordList& recordList, IN const bool valid) = 0;
};

} //genivi
} //btstackif

#endif //_I_SERVICE_SEARCH_GENIVI_H_
