/**
 * @file IPanConnectDisconnectGenivi.h
 *
 * @par SW-Component
 * State machine for PAN connect/disconnect
 *
 * @brief Interface definition for Genivi PAN connect/disconnect state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the Genivi interface for PAN connect/disconnect state machine.
 */

#ifndef _I_PAN_CONNECT_DISCONNECT_GENIVI_H_
#define _I_PAN_CONNECT_DISCONNECT_GENIVI_H_

#include "EvolutionGeniviStackTypes.h"

namespace btstackif {

// class forward declarations
class IProtocolManagerRequest;
class IConfiguration;
class IDeviceManager;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

namespace genivi {

// class forward declarations
class IObjectPathManagerGenivi;

/**
 * Interface definition.
 */
class IPanConnectDisconnectGenivi
{
public:
   /**
    * Destructor.
    */
   virtual ~IPanConnectDisconnectGenivi() {}

   /**
    * Get request interface.
    *
    * @return = request interface
    */
   virtual IProtocolManagerRequest* getRequestIf(void) = 0;

   /**
    * Set configuration interface.
    *
    * @param[in] configuration: configuration interface
    */
   virtual void setConfigurationIf(IN IConfiguration* configuration) = 0;

   /**
    * Set device manager interface.
    *
    * @param[in] deviceManagerIf: device manager interface
    */
   virtual void setDeviceManagerIf(IN IDeviceManager* deviceManagerIf) = 0;

   /**
    * Set object path manager interface.
    *
    * @param[in] objectPathManager: object path manager interface
    */
   virtual void setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager) = 0;

   /**
    * Add DBus interface.
    *
    * @param[out] dbusInterfaces: DBus interfaces
    * @param[in] userMode: user mode
    */
   virtual void addInterfaces(OUT ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSUserMode userMode) = 0;

   /**
    * Signal for protocol added.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocolObjectPath: protocol object path
    * @param[in] protocol: protocol
    */
   virtual void protocolAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSObjectPath& protocolObjectPath, IN const BTSProtocolId protocol) = 0;

   /**
    * Signal for protocol removed.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] protocolObjectPath: protocol object path
    */
   virtual void protocolRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& protocolObjectPath) = 0;

   /**
    * Update service availability.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] service: service object path
    * @param[in] availability: DBUS service available flag
    */
   virtual void updateDbusServiceAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& service, IN const BTSDbusServiceAvailability availability) = 0;

   /**
    * Update service availability (ConnMan manager).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] service: service object path
    * @param[in] availability: DBUS service available flag
    */
   virtual void updateDbusServiceConnManAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& service, IN const BTSDbusServiceAvailability availability) = 0;

   /**
    * Handle result for start connect.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[out] connectFailed: flag indicating that ACL connect failed
    * @param[out] failedAddress: address of device the ACL connect failed to
    * @param[in] device: object path
    * @param[in] errorCode: error code (result)
    */
   virtual void handleConnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& failedAddress, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Handle interface name.
    *
    * @param[in] device: object path
    * @param[in] interface: interface name
    */
   virtual void handleInterfaceName(IN const BTSObjectPath& device, IN const BTSDeviceName& interface) = 0;

   /**
    * Set connect status.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] status: status
    */
   virtual void updateConnectStatus(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status) = 0;

   /**
    * Handle connect indication.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] status: status
    */
   virtual void handleConnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status) = 0;

   /**
    * Handle update for service state.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[out] connectFailed: flag indicating that ACL connect failed
    * @param[out] failedAddress: address of device the ACL connect failed to
    * @param[in] device: object path
    * @param[in] serviceState: service state
    */
   virtual void handleServiceState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& failedAddress, IN const BTSObjectPath& device, IN const ::ccdbusif::connman::ServiceState serviceState) = 0;

   /**
    * Handle result for start disconnect.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: object path
    * @param[in] errorCode: error code (result)
    */
   virtual void handleDisconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Set disconnect reason.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] reason: disconnect reason
    */
   virtual void updateDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSInternalDisconnectReason reason) = 0;

   /**
    * Handle disconnect indication.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] status: status
    */
   virtual void handleDisconnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status) = 0;
};

} //genivi
} //btstackif

#endif //_I_PAN_CONNECT_DISCONNECT_GENIVI_H_
