/**
 * @file IObexConnectDisconnectGenivi.h
 *
 * @par SW-Component
 * State machine for OBEX connect/disconnect
 *
 * @brief Interface definition for Genivi OBEX connect/disconnect state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the Genivi interface for OBEX connect/disconnect state machine.
 */

#ifndef _I_OBEX_CONNECT_DISCONNECT_GENIVI_H_
#define _I_OBEX_CONNECT_DISCONNECT_GENIVI_H_

#include "EvolutionGeniviStackTypes.h"

namespace btstackif {

// class forward declarations
class IProtocolManagerRequest;
class ISwitchBluetooth;
class IDeviceManager;
class IServiceSearch;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

namespace genivi {

// class forward declarations
class IObjectPathManagerGenivi;

/**
 * Interface definition.
 */
class IObexConnectDisconnectGenivi
{
public:
   /**
    * Destructor.
    */
   virtual ~IObexConnectDisconnectGenivi() {}

   /**
    * Get request interface.
    *
    * @return = request interface
    */
   virtual IProtocolManagerRequest* getRequestIf(void) = 0;

   /**
    * Set switch Bluetooth interface.
    *
    * @param[in] switchBluetoothIf: switch Bluetooth interface
    */
   virtual void setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetoothIf) = 0;

   /**
    * Set device manager interface.
    *
    * @param[in] deviceManagerIf: device manager interface
    */
   virtual void setDeviceManagerIf(IN IDeviceManager* deviceManagerIf) = 0;

   /**
    * Set service search interface.
    *
    * @param[in] serviceSearchIf: service search interface
    */
   virtual void setServiceSearchIf(IN IServiceSearch* serviceSearchIf) = 0;

   /**
    * Set object path manager interface.
    *
    * @param[in] objectPathManager: object path manager interface
    */
   virtual void setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager) = 0;

   /**
    * Signal for protocol added.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocolObjectPath: protocol object path
    * @param[in] protocol: protocol
    * @param[in] masInstance: MAS instance ID (valid for MAP)
    */
   virtual void protocolAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSObjectPath& protocolObjectPath, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance) = 0;

   /**
    * Signal for protocol removed.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] protocolObjectPath: protocol object path
    */
   virtual void protocolRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& protocolObjectPath) = 0;

   /**
    * Handle result for create session.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[out] connectFailed: flag indicating that ACL connect failed
    * @param[out] failedAddress: address of device the ACL connect failed to
    * @param[in] remoteAddress: remote device address
    * @param[in] protocol: protocol
    * @param[in] masInstance: MAS instance ID (valid for MAP)
    * @param[in] session: session object path (set if creation was successful)
    * @param[in] errorCode: error code (result)
    */
   virtual void handleCreateSessionResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& failedAddress, IN const BTSBDAddress& remoteAddress, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance, IN const BTSObjectPath& session, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Set connect status.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] status: status
    */
   virtual void updateConnectStatus(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status) = 0;

   /**
    * Handle connect indication.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] status: status
    */
   virtual void handleConnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status) = 0;

   /**
    * Handle result for remove session.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] remoteAddress: remote device address
    * @param[in] protocol: protocol
    * @param[in] masInstance: MAS instance ID (valid for MAP)
    * @param[in] errorCode: error code (result)
    */
   virtual void handleRemoveSessionResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& remoteAddress, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Handle result for cancel session.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] remoteAddress: remote device address
    * @param[in] protocol: protocol
    * @param[in] masInstance: MAS instance ID (valid for MAP)
    * @param[in] errorCode: error code (result)
    */
   virtual void handleCancelSessionResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& remoteAddress, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Set disconnect reason.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] reason: disconnect reason
    */
   virtual void updateDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSInternalDisconnectReason reason) = 0;

   /**
    * Handle disconnect indication.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] status: status
    */
   virtual void handleDisconnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTriggerIgnoreProtocolAdded(IN const bool enable) = 0;
};

} //genivi
} //btstackif

#endif //_I_OBEX_CONNECT_DISCONNECT_GENIVI_H_
