/**
 * @file ILocalAdapterModesGenivi.h
 *
 * @par SW-Component
 * State machine for local adapter modes
 *
 * @brief Interface definition for Genivi local adapter modes state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the Genivi interface for local adapter modes state machine.
 */

#ifndef _I_LOCAL_ADAPTER_MODES_GENIVI_H_
#define _I_LOCAL_ADAPTER_MODES_GENIVI_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

namespace genivi {

/**
 * Interface definition.
 */
class ILocalAdapterModesGenivi
{
public:
   /**
    * Destructor.
    */
   virtual ~ILocalAdapterModesGenivi() {}

   /**
    * Handle discoverable mode update message.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] enabled: discoverable mode
    * @param[in] errorCode: error code
    * @param[in] response: response flag (update or return)
    * @param[in] force: force flag
    */
   virtual void handleDiscoverableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force = false) = 0;

   /**
    * Handle pairable mode update message.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] enabled: pairable mode
    * @param[in] errorCode: error code
    * @param[in] response: response flag (update or return)
    * @param[in] force: force flag
    */
   virtual void handlePairableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force = false) = 0;

   /**
    * Handle connectable mode update message.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] enabled: connectable mode
    * @param[in] errorCode: error code
    * @param[in] response: response flag (update or return)
    * @param[in] force: force flag
    */
   virtual void handleConnectableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force = false) = 0;

   /**
    * Handle discoverable timeout update message.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] timeout: discoverable timeout
    * @param[in] errorCode: error code
    * @param[in] response: response flag (update or return)
    * @param[in] force: force flag
    */
   virtual void handleDiscoverableTimeoutUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoverableTimeout timeout, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force = false) = 0;

   /**
    * Handle pairable timeout update message.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] timeout: pairable timeout
    * @param[in] errorCode: error code
    * @param[in] response: response flag (update or return)
    * @param[in] force: force flag
    */
   virtual void handlePairableTimeoutUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoverableTimeout timeout, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force = false) = 0;

   /**
    * Handle received adapter properties.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] properties: properties
    * @param[in] errorCode: error code
    */
   virtual void handleAdapterProperties(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusPropertyList& properties, IN const BTSIpcCommonErrorCode errorCode) = 0;
};

} //genivi
} //btstackif

#endif //_I_LOCAL_ADAPTER_MODES_GENIVI_H_
