/**
 * @file IHfpDeviceCapabilitiesGenivi.h
 *
 * @par SW-Component
 * State machine for HFP device capabilities
 *
 * @brief Interface definition for Genivi HFP device capabilities state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the Genivi interface for HFP device capabilities state machine.
 */

#ifndef _I_HFP_DEVICE_CAPABILITIES_GENIVI_H_
#define _I_HFP_DEVICE_CAPABILITIES_GENIVI_H_

#include "EvolutionGeniviStackTypes.h"

namespace btstackif {

// class forward declarations
class IHfpDeviceCapabilitiesRequest;
class DbusServiceStatus;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

namespace genivi {

// class forward declarations
class IObjectPathManagerGenivi;

/**
 * Interface definition.
 */
class IHfpDeviceCapabilitiesGenivi
{
public:
   /**
    * Destructor.
    */
   virtual ~IHfpDeviceCapabilitiesGenivi() {}

   /**
    * Get request interface.
    *
    * @return = request interface
    */
   virtual IHfpDeviceCapabilitiesRequest* getRequestIf(void) = 0;

   /**
    * Set object path manager interface.
    *
    * @param[in] objectPathManager: object path manager interface
    */
   virtual void setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager) = 0;

   /**
    * Set enabled flag. If enabled HFP device capabilities will be requested else not.
    *
    * @param[in] enabled: enabled flag
    * @param[in] userMode: user mode
    */
   virtual void setEnabled(IN const bool enabled, IN const BTSUserMode userMode) = 0;

   /**
    * Get enabled flag.
    *
    * @return = true: enabled,
    * @return = false: disabled
    */
   virtual bool getEnabled(void) const = 0;

   /**
    * Get DBus service entry.
    *
    * @return = DBus service entry
    */
   virtual DbusServiceStatus& getServiceEntry(void) = 0;

   /**
    * Add DBus interface.
    *
    * @param[out] dbusInterfaces: DBus interfaces
    * @param[in] userMode: user mode
    */
   virtual void addInterfaces(OUT ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSUserMode userMode) = 0;

   /**
    * Get service availability for HFP manager interface.
    *
    * @return = service availability
    */
   virtual BTSDbusServiceAvailability getManagerAvailability(void) const = 0;

   /**
    * Update service availability for HFP manager interface.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] availability: DBUS service available flag
    */
   virtual void updateManagerAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusServiceAvailability availability) = 0;

   /**
    * Update service availability for HFP modem interface.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] objPath: object path
    * @param[in] availability: DBUS service available flag
    */
   virtual void updateModemAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& objPath, IN const BTSDbusServiceAvailability availability) = 0;

   /**
    * Update service availability for HFP handsfree interface.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] objPath: object path
    * @param[in] availability: DBUS service available flag
    */
   virtual void updateHandsfreeAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& objPath, IN const BTSDbusServiceAvailability availability) = 0;

   /**
    * Handle added HFP interface.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] objPath: object path
    * @param[in] interface: interface
    */
   virtual void handleAddedHfpInterface(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& objPath, IN const ::ccdbusif::evolution::Interface interface) = 0;

   /**
    * Handle removed HFP interface.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] objPath: object path
    * @param[in] interface: interface
    */
   virtual void handleRemovedHfpInterface(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& objPath, IN const ::ccdbusif::evolution::Interface interface) = 0;

   /**
    * Handle update for handsfree features.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] modem: object path of modem device
    * @param[in] features: handsfree features
    */
   virtual void handleHandsfreeFeaturesUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& modem, IN const BTSHfpSupportedFeatures& features) = 0;

   /**
    * Handle update for handsfree inband ringing enabled.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] modem: object path of modem device
    * @param[in] inbandRingingEnabled: inband ringing enabled status
    */
   virtual void handleHandsfreeInbandRingingUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& modem, IN const bool inbandRingingEnabled) = 0;

   /**
    * Handle result for GetModems().
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] errorCode: error code (result)
    */
   virtual void handleGetModemsResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Handle result for GetHandsfreeProperties().
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] modem: object path of modem device
    * @param[in] errorCode: error code (result)
    */
   virtual void handleGetHandsfreePropertiesResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& modem, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Handle result for GetModemProperties().
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] modem: object path of modem device
    * @param[in] errorCode: error code (result)
    */
   virtual void handleGetModemPropertiesResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& modem, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Handle modem added information.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] modem: object path of modem device
    */
   virtual void handleModemAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSObjectPath& modem) = 0;

   /**
    * Handle modem removed information.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] modem: object path of modem device
    */
   virtual void handleModemRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& modem) = 0;

   /**
    * Handle update for modem interfaces.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] modem: object path of modem device
    * @param[in] interfaces: modem interfaces
    */
   virtual void handleModemInterfacesUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& modem, IN const BTSDbusInterfaceList& interfaces) = 0;
};

} //genivi
} //btstackif

#endif //_I_HFP_DEVICE_CAPABILITIES_GENIVI_H_
