/**
 * @file IHfpAvpConnectDisconnectGenivi.h
 *
 * @par SW-Component
 * State machine for HFP/AVP connect/disconnect
 *
 * @brief Interface definition for Genivi HFP/AVP connect/disconnect state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the Genivi interface for HFP/AVP connect/disconnect state machine.
 */

#ifndef _I_HFP_AVP_CONNECT_DISCONNECT_GENIVI_H_
#define _I_HFP_AVP_CONNECT_DISCONNECT_GENIVI_H_

#include "EvolutionGeniviStackTypes.h"

namespace btstackif {

// class forward declarations
class IProtocolManagerRequest;
class IDeviceManager;
class IServiceSearch;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

namespace genivi {

// class forward declarations
class IObjectPathManagerGenivi;

/**
 * Interface definition.
 */
class IHfpAvpConnectDisconnectGenivi
{
public:
   /**
    * Destructor.
    */
   virtual ~IHfpAvpConnectDisconnectGenivi() {}

   /**
    * Get request interface.
    *
    * @return = request interface
    */
   virtual IProtocolManagerRequest* getRequestIf(void) = 0;

   /**
    * Set device manager interface.
    *
    * @param[in] deviceManagerIf: device manager interface
    */
   virtual void setDeviceManagerIf(IN IDeviceManager* deviceManagerIf) = 0;

   /**
    * Set service search interface.
    *
    * @param[in] serviceSearchIf: service search interface
    */
   virtual void setServiceSearchIf(IN IServiceSearch* serviceSearchIf) = 0;

   /**
    * Set object path manager interface.
    *
    * @param[in] objectPathManager: object path manager interface
    */
   virtual void setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager) = 0;

   /**
    * Signal for protocol added.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] properties: device properties
    */
   virtual void protocolAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusPropertyList& properties) = 0;

   /**
    * Signal for protocol added.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] deviceObjectPath: device object path
    * @param[in] protocolObjectPath: protocol object path
    * @param[in] protocol: protocol
    * @param[in] uuid: SPP UUID
    */
   virtual void protocolAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& deviceObjectPath, IN const BTSObjectPath& protocolObjectPath, IN const BTSProtocolId protocol, IN const BTSUuid& uuid) = 0;

   /**
    * Signal for protocol added.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol
    * @param[in] uuid: SPP UUID
    * @param[in] state: connection state
    */
   virtual void protocolAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSServiceState state) = 0;

   /**
    * Signal for protocol removed.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] protocolObjectPath: protocol object path
    */
   virtual void protocolRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& protocolObjectPath) = 0;

   /**
    * Update service availability.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] service: service object path
    * @param[in] availability: DBUS service available flag
    */
   virtual void updateDbusServiceAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& service, IN const BTSDbusServiceAvailability availability) = 0;

   /**
    * Handle cancel request from stack.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] sent: flag indicating that response message was already sent
    * @param[in] token: DBUS token
    *
    * @return = true: request was answered,
    * @return = false: request was not answered
    */
   virtual bool handleCancelRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool sent, IN const act_t token) = 0;

   /**
    * Handle cancel request from stack.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: object path of device
    * @param[in] sent: flag indicating that response message was already sent
    * @param[in] token: DBUS token
    *
    * @return = true: request was answered,
    * @return = false: request was not answered
    */
   virtual bool handleCancelRequestRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool sent, IN const act_t token) = 0;

   /**
    * Set connected service.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: object path of device
    * @param[in] connectedService: connected service
    */
   virtual void updateConnectedService(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSServiceFunction& connectedService) = 0;

   /**
    * Remove connected service.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: object path of device
    */
   virtual void removeConnectedService(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device) = 0;

   /**
    * Handle remote connect request.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: object path of device
    * @param[in] protocol: protocol id
    * @param[in] uuid: uuid
    * @param[in] token: DBUS token
    *
    * @return = true: request was handled,
    * @return = false: request was not handled
    */
   virtual bool handleRemoteConnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const act_t token) = 0;

   /**
    * Create message for authorize service response.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[in] device: object path of device
    * @param[in] token: token
    * @param[in] mode: confirmation mode
    * @param[in] responseFlag: response flag
    */
   virtual void createAuthorizeServiceResponse(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSObjectPath& device, IN const act_t token, IN const BTSConfirmationMode mode, IN const bool responseFlag) const = 0;

   /**
    * Handle result for start connect.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[out] connectFailed: flag indicating that ACL connect failed
    * @param[out] failedAddress: address of device the ACL connect failed to
    * @param[in] service: service object path
    * @param[in] errorCode: error code (result)
    */
   virtual void handleConnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& failedAddress, IN const BTSObjectPath& service, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Set connect status.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] status: status
    */
   virtual void updateConnectStatus(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status) = 0;

   /**
    * Handle connect indication.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] status: status
    */
   virtual void handleConnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status) = 0;

   /**
    * Handle update for service state.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[out] connectFailed: flag indicating that ACL connect failed
    * @param[out] failedAddress: address of device the ACL connect failed to
    * @param[in] service: service object path
    * @param[in] serviceState: service state
    * @param[in] errorCode: error code (result)
    */
   virtual void handleServiceState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& failedAddress, IN const BTSObjectPath& service, IN const BTSServiceState serviceState, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Handle result for start disconnect.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] service: service object path
    * @param[in] errorCode: error code (result)
    */
   virtual void handleDisconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& service, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Set disconnect reason.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] reason: disconnect reason
    */
   virtual void updateDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSInternalDisconnectReason reason) = 0;

   /**
    * Handle disconnect indication.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] status: status
    */
   virtual void handleDisconnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status) = 0;

   /**
    * Signal for serial added.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] serialObjectPath: serial object path
    */
   virtual void serialAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& serialObjectPath) = 0;

   /**
    * Signal for serial removed.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] serialObjectPath: serial object path
    */
   virtual void serialRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& serialObjectPath) = 0;

   /**
    * Update service availability for serial interface.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] service: service object path
    * @param[in] availability: DBUS service available flag
    */
   virtual void updateDbusServiceSerialAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& service, IN const BTSDbusServiceAvailability availability) = 0;

   /**
    * Set character device name.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] service: object path of service
    * @param[in] deviceName: character device name
    */
   virtual void updateCharacterDeviceName(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& service, IN const BTSDeviceName& deviceName) = 0;

   /**
    * Inform about added/removed character device (TTY monitor).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] deviceName: character device name
    * @param[in] added: added/removed flag
    */
   virtual void characterDeviceAddedRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDeviceName& deviceName, IN const bool added) = 0;
};

} //genivi
} //btstackif

#endif //_I_HFP_AVP_CONNECT_DISCONNECT_GENIVI_H_
