/**
 * @file TrcIpc2Bts_GEN.cpp
 *
 * @par SW-Component
 * Trace
 *
 * @brief Traces for Ipc2Bts messages.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of traces for Ipc2Bts messages.
 */

#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "TraceClasses.h"
#include "FwTrace.h"
#include "BtsUtils.h"
#include "EvolutionGeniviStackTypes.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"
#include "cc_dbus_if/ConnManDbusParser.h"
#include "TrcIpc2BtsProperty_GEN.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_IPC2BTS
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/TrcIpc2Bts_GEN.cpp.trc.h"
#else
#include "BtStackIfTypesTrace.h"
#include "BtStackInternalTypesTrace.h"
#include "EvolutionGeniviStackTypesTrace.h"
#include "EvolutionGeniviStackOpcodesTrace.h"
#endif
#endif

namespace btstackif {
namespace genivi {

void Ipc2Bts_ServiceAvailabilityConnection::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d interface=%d availabilityEvent=%d busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSGenDbusServiceInterface, _interface),
            ETG_ENUM(TRC_BTSDbusServiceAvailability, _availabilityEvent),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s interface=%s availabilityEvent=%s busType=%d busName=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getGenDbusServiceInterface2String(_interface),
            getDbusServiceAvailability2String(_availabilityEvent),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_GetAllPropertiesConnection::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d interface=%d properties.size()=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_CCDBUSIFInterface, _interface),
            _properties.size() /*DONE: format*/));
   doPropertyTrace(getMessageTraceId(), opcode, _interface, _properties);

#else

   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s interface=%s properties.size()=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            parser.getInterface2StringTrace(_interface).c_str(),
            _properties.size() /*DONE: format*/));
   doPropertyTrace(getMessageTraceId(), opcode, _interface, _properties);

#endif
}

void Ipc2Bts_AddDeviceObjectPathMapping::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _objPath.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_DelDeviceObjectPathMapping::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_AddProtocolObjectPathMapping::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;

   if((BTS_PROTO_HFP == _protocol) || (BTS_PROTO_AVP == _protocol))
   {
      ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d protocol=%d device=%s",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
               ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
               ETG_ENUM(TRC_BTSProtocolId, _protocol),
               _device.c_str() /*DONE: format*/));
   }
   else if(BTS_PROTO_SPP == _protocol)
   {
      ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d protocol=%d uuid=%40s device=%s",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
               ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
               ETG_ENUM(TRC_BTSProtocolId, _protocol),
               _uuid.c_str() /*DONE: format*/,
               _device.c_str() /*DONE: format*/));
   }
   else if(BTS_PROTO_MSG == _protocol)
   {
      BTSBDAddressTrace traceAddr(_deviceAddress);

      ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d protocol=%d instanceId=%u btAddr=%08X%04X",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
               ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
               ETG_ENUM(TRC_BTSProtocolId, _protocol),
               _instanceId,
               traceAddr.getUpper(), traceAddr.getLower()));
   }
   else
   {
      BTSBDAddressTrace traceAddr(_deviceAddress);

      ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d protocol=%d btAddr=%08X%04X",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
               ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
               ETG_ENUM(TRC_BTSProtocolId, _protocol),
               traceAddr.getUpper(), traceAddr.getLower()));
   }

   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;

   if((BTS_PROTO_HFP == _protocol) || (BTS_PROTO_AVP == _protocol))
   {
      BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s protocol=%s device=%s",
               getMessageTraceId(),
               getGenIpc2BtsOpcode2String(opcode),
               getIpcCommonErrorCode2String(_commonErrorCode),
               getProtocolId2String(_protocol),
               _device.c_str() /*DONE: format*/));
   }
   else if(BTS_PROTO_SPP == _protocol)
   {
      BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s protocol=%s uuid=%40s device=%s",
               getMessageTraceId(),
               getGenIpc2BtsOpcode2String(opcode),
               getIpcCommonErrorCode2String(_commonErrorCode),
               getProtocolId2String(_protocol),
               _uuid.c_str() /*DONE: format*/,
               _device.c_str() /*DONE: format*/));
   }
   else if(BTS_PROTO_MSG == _protocol)
   {
      BTSBDAddressTrace traceAddr(_deviceAddress);

      BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s protocol=%s instanceId=%u btAddr=%08X%04X",
               getMessageTraceId(),
               getGenIpc2BtsOpcode2String(opcode),
               getIpcCommonErrorCode2String(_commonErrorCode),
               getProtocolId2String(_protocol),
               _instanceId,
               traceAddr.getUpper(), traceAddr.getLower()));
   }
   else
   {
      BTSBDAddressTrace traceAddr(_deviceAddress);

      BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s protocol=%s btAddr=%08X%04X",
               getMessageTraceId(),
               getGenIpc2BtsOpcode2String(opcode),
               getIpcCommonErrorCode2String(_commonErrorCode),
               getProtocolId2String(_protocol),
               traceAddr.getUpper(), traceAddr.getLower()));
   }

   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_DelProtocolObjectPathMapping::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_AddSerialObjectPathMapping::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_DelSerialObjectPathMapping::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_Release::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_RequestPinCode::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_DisplayPinCode::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d passkey=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _passkey.c_str()));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _device.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s passkey=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _passkey.c_str()));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _device.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_RequestPasskey::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_RequestConfirmation::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d passkey=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _passkey.c_str()));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _device.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s passkey=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _passkey.c_str()));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _device.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_Cancel::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_CancelRequest::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_AuthorizeService::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d protocol=%d uuid=%32s device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSProtocolId, _protocol),
            _uuid.c_str(),
            _device.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s protocol=%s uuid=%32s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getProtocolId2String(_protocol),
            _uuid.c_str(),
            _device.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_RequestAuthorization::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_RequestLinkkey::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_DisplayPasskey::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d entered=%u passkey=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _entered,
            _passkey.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _device.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s entered=%u passkey=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _entered,
            _passkey.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _device.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_StartDiscoveryResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_StopDiscoveryResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_RemoveDeviceResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_CreateDeviceResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Ipc2Bts_SendHCICmdResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_AdapterInitialization::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_DeviceCreated::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_DeviceRemoved::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_BtStackLog::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int binSize = (unsigned int)_logData.size();
   unsigned char* binData = &(_logData[0]);
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d status=%d logOpcode=0x%04X logLength=%u logData=%02X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSLogStatus, _status),
            _logOpcode,
            binSize,
            ETG_LIST(binSize, ETG_T8, binData) /*DONE: _logData format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int binSize = (unsigned int)_logData.size();
   unsigned char* binData = &(_logData[0]);
   char tmpBuffer[1024] = { 0 };
   for(unsigned int i = 0; (i < binSize) && (i < ((sizeof(tmpBuffer) - 1) / 3)); i++)
   {
      sprintf(&tmpBuffer[3 * i], "%02X ", binData[i]);
   }
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s status=%s logOpcode=0x%04X logLength=%u logData=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getLogStatus2String(_status),
            _logOpcode,
            binSize,
            tmpBuffer /*DONE: _logData format*/));

#endif
}

void Ipc2Bts_HCIEvent::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d length=%u data=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _length,
            "<TODO>" /*TODO: _data format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s length=%u data=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _length,
            "<TODO>" /*TODO: _data format*/));

#endif
}

void Ipc2Bts_DeviceFound::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_device.address);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X rssi=%d majorServiceClass=0x%04X majorDeviceClass=%d minorDeviceClass=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.rssi,
            _device.majorServiceClass.getData(),
            ETG_ENUM(TRC_BTSMajorDeviceClass, _device.majorDeviceClass),
            ETG_ENUM(TRC_BTSMinorDeviceClass, _device.minorDeviceClass)));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): services=0x%04X sppCapabilities.size()=%u name=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _device.services.getData(),
            _device.sppCapabilities.size(),
            _device.name.c_str()));
   for(size_t i = 0; i < _device.sppCapabilities.size(); i++)
   {
      ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): sppCapabilities[%u]: uuidList.size()=%u serviceName=%s",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
               i,
               _device.sppCapabilities[i].uuidList.size(),
               _device.sppCapabilities[i].serviceName.c_str() /*DONE: _sppCapabilities format*/));
      for(size_t j = 0; j < _device.sppCapabilities[i].uuidList.size(); j++)
      {
         ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): sppCapabilities[%u]: uuidList[%u]: uuid=%s",
                  getMessageTraceId(),
                  ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
                  i,
                  j,
                  _device.sppCapabilities[i].uuidList[j].c_str() /*DONE: _sppCapabilities format*/));
      }
   }

#else

   BTSBDAddressTrace traceAddr(_device.address);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X rssi=%d majorServiceClass=0x%04X majorDeviceClass=%s minorDeviceClass=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.rssi,
            _device.majorServiceClass.getData(),
            getMajorDeviceClass2String(_device.majorDeviceClass),
            getMinorDeviceClass2String(_device.minorDeviceClass)));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): services=0x%04X sppCapabilities.size()=%u name=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _device.services.getData(),
            _device.sppCapabilities.size(),
            _device.name.c_str()));
   for(size_t i = 0; i < _device.sppCapabilities.size(); i++)
   {
      BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): sppCapabilities[%u]: uuidList.size()=%u serviceName=%s",
               getMessageTraceId(),
               getGenIpc2BtsOpcode2String(opcode),
               i,
               _device.sppCapabilities[i].uuidList.size(),
               _device.sppCapabilities[i].serviceName.c_str() /*DONE: _sppCapabilities format*/));
      for(size_t j = 0; j < _device.sppCapabilities[i].uuidList.size(); j++)
      {
         BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): sppCapabilities[%u]: uuidList[%u]: uuid=%s",
                  getMessageTraceId(),
                  getGenIpc2BtsOpcode2String(opcode),
                  i,
                  j,
                  _device.sppCapabilities[i].uuidList[j].c_str() /*DONE: _sppCapabilities format*/));
      }
   }

#endif
}

void Ipc2Bts_AdapterAddressUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower()));

#endif
}

void Ipc2Bts_AdapterClassUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d cod=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _cod));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s cod=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _cod));

#endif
}

void Ipc2Bts_AdapterAliasUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d alias=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _alias.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s alias=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _alias.c_str()));

#endif
}

void Ipc2Bts_AdapterPoweredUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d powered=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _powered)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s powered=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_powered)));

#endif
}

void Ipc2Bts_AdapterDiscoverableUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d discoverable=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _discoverable)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s discoverable=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_discoverable)));

#endif
}

void Ipc2Bts_AdapterDiscoverabletimeoutUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d discoverableTimeout=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _discoverableTimeout));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s discoverableTimeout=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _discoverableTimeout));

#endif
}

void Ipc2Bts_AdapterPairableUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d pairable=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _pairable)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s pairable=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_pairable)));

#endif
}

void Ipc2Bts_AdapterPairabletimeoutUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d pairableTimeout=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _pairableTimeout));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s pairableTimeout=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _pairableTimeout));

#endif
}

void Ipc2Bts_AdapterDiscoveringUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d discovering=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _discovering)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s discovering=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_discovering)));

#endif
}

void Ipc2Bts_AdapterUuidsUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d uuids.size()=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _uuids.size() /*DONE: format*/));
   for(size_t i = 0; i < _uuids.size(); i++)
   {
      ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): uuids[%u]=%s",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
               i,
               _uuids[i].c_str() /*DONE: _uuids format*/));
   }

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s uuids.size()=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _uuids.size() /*DONE: format*/));
   for(size_t i = 0; i < _uuids.size(); i++)
   {
      BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): uuids[%u]=%s",
               getMessageTraceId(),
               getGenIpc2BtsOpcode2String(opcode),
               i,
               _uuids[i].c_str() /*DONE: _uuids format*/));
   }

#endif
}

void Ipc2Bts_AdapterWbsModeUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d wbsMode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _wbsMode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s wbsMode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_wbsMode)));

#endif
}

void Ipc2Bts_AdapterConnectableUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d connectable=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _connectable)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s connectable=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_connectable)));

#endif
}

void Ipc2Bts_AdapterHcimodeUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d hciMode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _hciMode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s hciMode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_hciMode)));

#endif
}

void Ipc2Bts_AdapterVersionUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d appCode=0x%04X geniviVersion=0x%04X evolutionVersion=0x%04X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _stackVersion.appCode,
            _stackVersion.geniviVersion,
            _stackVersion.evolutionVersion /*DONE: _stackVersion format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s appCode=0x%04X geniviVersion=0x%04X evolutionVersion=0x%04X",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _stackVersion.appCode,
            _stackVersion.geniviVersion,
            _stackVersion.evolutionVersion /*DONE: _stackVersion format*/));

#endif
}

void Ipc2Bts_AdapterTraceLevelUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d traceLevel=0x%08X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _traceLevel.getData()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s traceLevel=0x%08X",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _traceLevel.getData()));

#endif
}

void Ipc2Bts_AdapterTraceCategoryUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_traceCategory.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_traceCategory.getData()      ) & 0x00000000FFFFFFFF);
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d traceCategory=0x%08X%08X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            u1, u2));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_traceCategory.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_traceCategory.getData()      ) & 0x00000000FFFFFFFF);
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s traceCategory=0x%08X%08X",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            u1, u2));

#endif
}

void Ipc2Bts_AdapterAvailableSppServersUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d availableSppServers.size()=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _availableSppServers.size() /*DONE: format*/));
   for(size_t i = 0; i < _availableSppServers.size(); i++)
   {
      ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): availableSppServers[%u]: uuid=%s",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
               i,
               _availableSppServers[i].c_str() /*DONE: _availableSppServers format*/));
   }

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s availableSppServers.size()=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _availableSppServers.size() /*DONE: format*/));
   for(size_t i = 0; i < _availableSppServers.size(); i++)
   {
      BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): availableSppServers[%u]: uuid=%s",
               getMessageTraceId(),
               getGenIpc2BtsOpcode2String(opcode),
               i,
               _availableSppServers[i].c_str() /*DONE: _availableSppServers format*/));
   }

#endif
}

void Ipc2Bts_RegisterAgentResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_UnregisterAgentResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_RequestDefaultAgentResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_DeviceConnectResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_DeviceDisconnectResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_PairResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_CancelPairingResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_DiscoverServicesResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_CancelDiscoveryResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_ReportLinkkey::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): linkKey=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _linkKey.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): linkKey=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _linkKey.c_str()));

#endif
}

void Ipc2Bts_IncomingPairingComp::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d status=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSStatusCode, _status),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s status=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getStatusCode2String(_status),
            _device.c_str()));

#endif
}

void Ipc2Bts_SDPProcessReport::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d active=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _active),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s active=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_active),
            _device.c_str()));

#endif
}

void Ipc2Bts_DeviceAddressUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d address=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _deviceAddress.c_str()));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%d address=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _deviceAddress.c_str()));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _device.c_str()));

#endif
}

void Ipc2Bts_DeviceClassUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d cod=%u device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _cod,
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s cod=%u device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _cod,
            _device.c_str()));

#endif
}

void Ipc2Bts_DeviceIconUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d icon=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _icon.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _device.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s icon=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _icon.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _device.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_DeviceUuidsUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d supportedServices=0x%04X (HFP=%d A2DP=%d AVRCP=%d PBAP=%d MAP=%d) device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _supportedServices.getData() /*DONE: format*/,
            _supportedServices.getBit(BTS_SUPP_SRV_HFP),
            _supportedServices.getBit(BTS_SUPP_SRV_A2DP),
            _supportedServices.getBit(BTS_SUPP_SRV_AVRCP),
            _supportedServices.getBit(BTS_SUPP_SRV_PBAP),
            _supportedServices.getBit(BTS_SUPP_SRV_MAP),
            _device.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): (DUN=%d PAN=%d SPP=%d PNP=%d DID=%d) uuidList.size()=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _supportedServices.getBit(BTS_SUPP_SRV_DUN),
            _supportedServices.getBit(BTS_SUPP_SRV_PAN),
            _supportedServices.getBit(BTS_SUPP_SRV_SPP),
            _supportedServices.getBit(BTS_SUPP_SRV_PNP),
            _supportedServices.getBit(BTS_SUPP_SRV_DID),
            _uuidList.size() /*DONE: format*/));
   for(size_t i = 0; i < _uuidList.size(); i++)
   {
      ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): uuidList[%u]=%s",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
               i,
               _uuidList[i].c_str() /*DONE: uuidList format*/));
   }


#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s supportedServices=0x%04X (HFP=%d A2DP=%d AVRCP=%d PBAP=%d MAP=%d) device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _supportedServices.getData() /*DONE: format*/,
            _supportedServices.getBit(BTS_SUPP_SRV_HFP),
            _supportedServices.getBit(BTS_SUPP_SRV_A2DP),
            _supportedServices.getBit(BTS_SUPP_SRV_AVRCP),
            _supportedServices.getBit(BTS_SUPP_SRV_PBAP),
            _supportedServices.getBit(BTS_SUPP_SRV_MAP),
            _device.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): (DUN=%d PAN=%d SPP=%d PNP=%d DID=%d) uuidList.size()=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _supportedServices.getBit(BTS_SUPP_SRV_DUN),
            _supportedServices.getBit(BTS_SUPP_SRV_PAN),
            _supportedServices.getBit(BTS_SUPP_SRV_SPP),
            _supportedServices.getBit(BTS_SUPP_SRV_PNP),
            _supportedServices.getBit(BTS_SUPP_SRV_DID),
            _uuidList.size() /*DONE: format*/));
   for(size_t i = 0; i < _uuidList.size(); i++)
   {
      BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): uuidList[%u]=%s",
               getMessageTraceId(),
               getGenIpc2BtsOpcode2String(opcode),
               i,
               _uuidList[i].c_str() /*DONE: uuidList format*/));
   }

#endif
}

void Ipc2Bts_DevicePairedUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d paired=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _paired),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s paired=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_paired),
            _device.c_str()));

#endif
}

void Ipc2Bts_DeviceConnectedUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d connected=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _connected),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s connected=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_connected),
            _device.c_str()));

#endif
}

void Ipc2Bts_DeviceTrustedUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d trusted=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _trusted),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s trusted=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_trusted),
            _device.c_str()));

#endif
}

void Ipc2Bts_DeviceBlockedUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d blocked=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _blocked),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s blocked=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_blocked),
            _device.c_str()));

#endif
}

void Ipc2Bts_DeviceAliasUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d alias=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _alias.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _device.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s alias=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _alias.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _device.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_DeviceAdapterUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d adapter=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _adapter.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _device.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s adapter=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _adapter.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _device.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_DeviceModaliasUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d modalias=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _modalias.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _device.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s modalias=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _modalias.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _device.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_DeviceRssiUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d rssi=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _rssi,
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s rssi=%d device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _rssi,
            _device.c_str()));

#endif
}

void Ipc2Bts_DeviceVendorIdUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d vendorId=0x%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _vendorId,
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s vendorId=0x%04X device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _vendorId,
            _device.c_str()));

#endif
}

void Ipc2Bts_DeviceVendorIdSourceUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d vendorIdSource=0x%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _vendorIdSource,
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s vendorIdSource=0x%04X device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _vendorIdSource,
            _device.c_str()));

#endif
}

void Ipc2Bts_DeviceProductIdUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d productId=0x%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _productId,
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s productId=0x%04X device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _productId,
            _device.c_str()));

#endif
}

void Ipc2Bts_DeviceMapInfoUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d valid=%d mapInfoList.size()=%u device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _valid),
            _mapInfoList.size() /*DONE: format*/,
            _device.c_str() /*DONE: format*/));
   for(size_t i = 0; i < _mapInfoList.size(); i++)
   {
      const BTSMapSdpRecord& record = _mapInfoList[i];
      ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): mapInfoList[%u]: instanceId=%u messageTypes=0x%02X (EMAIL=%d SMS_GSM=%d SMS_CDMA=%d MMS=%d IM=%d)",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
               i,
               record.instanceId,
               record.messageTypes.getData(),
               record.messageTypes.getBit(BTS_MAP_MSG_TYPE_EMAIL),
               record.messageTypes.getBit(BTS_MAP_MSG_TYPE_SMS_GSM),
               record.messageTypes.getBit(BTS_MAP_MSG_TYPE_SMS_CDMA),
               record.messageTypes.getBit(BTS_MAP_MSG_TYPE_MMS),
               record.messageTypes.getBit(BTS_MAP_MSG_TYPE_IM)));
      ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): mapInfoList[%u]: version=0x%04X features=0x%08X name=%s",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
               i,
               record.version,
               record.features.getData(),
               record.name.c_str()));
   }

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s valid=%s mapInfoList.size()=%u device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_valid),
            _mapInfoList.size() /*DONE: format*/,
            _device.c_str() /*DONE: format*/));
   for(size_t i = 0; i < _mapInfoList.size(); i++)
   {
      const BTSMapSdpRecord& record = _mapInfoList[i];
      BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): mapInfoList[%u]: instanceId=%u messageTypes=0x%02X (EMAIL=%d SMS_GSM=%d SMS_CDMA=%d MMS=%d IM=%d)",
               getMessageTraceId(),
               getGenIpc2BtsOpcode2String(opcode),
               i,
               record.instanceId,
               record.messageTypes.getData(),
               record.messageTypes.getBit(BTS_MAP_MSG_TYPE_EMAIL),
               record.messageTypes.getBit(BTS_MAP_MSG_TYPE_SMS_GSM),
               record.messageTypes.getBit(BTS_MAP_MSG_TYPE_SMS_CDMA),
               record.messageTypes.getBit(BTS_MAP_MSG_TYPE_MMS),
               record.messageTypes.getBit(BTS_MAP_MSG_TYPE_IM)));
      BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): mapInfoList[%u]: version=0x%04X features=0x%08X name=%s",
               getMessageTraceId(),
               getGenIpc2BtsOpcode2String(opcode),
               i,
               record.version,
               record.features.getData(),
               record.name.c_str()));
   }

#endif
}

void Ipc2Bts_DeviceConnectedServiceUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_connectedService.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_connectedService.getData()      ) & 0x00000000FFFFFFFF);
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d connectedService=0x%08X%08X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            u1, u2 /*DONE: format*/,
            _device.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): SPP=%d HFP=%d A2DP=%d AVRCP=%d PIM_HFP=%d PIM_SPP=%d PIM_PBAP=%d PIM_SYNCML=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _connectedService.getBit(BTS_SRV_FUNC_SPP),
            _connectedService.getBit(BTS_SRV_FUNC_HFP),
            _connectedService.getBit(BTS_SRV_FUNC_A2DP),
            _connectedService.getBit(BTS_SRV_FUNC_AVRCP),
            _connectedService.getBit(BTS_SRV_FUNC_PIM_HFP),
            _connectedService.getBit(BTS_SRV_FUNC_PIM_SPP),
            _connectedService.getBit(BTS_SRV_FUNC_PIM_PBAP),
            _connectedService.getBit(BTS_SRV_FUNC_PIM_SYNCML)));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): MSG_HFP=%d MSG_MAP=%d MSG_SPP=%d DUN=%d OPPS=%d OPPC=%d BIPR=%d BIPI=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _connectedService.getBit(BTS_SRV_FUNC_MSG_HFP),
            _connectedService.getBit(BTS_SRV_FUNC_MSG_MAP),
            _connectedService.getBit(BTS_SRV_FUNC_MSG_SPP),
            _connectedService.getBit(BTS_SRV_FUNC_DUN),
            _connectedService.getBit(BTS_SRV_FUNC_OPPS),
            _connectedService.getBit(BTS_SRV_FUNC_OPPC),
            _connectedService.getBit(BTS_SRV_FUNC_BIPR),
            _connectedService.getBit(BTS_SRV_FUNC_BIPI)));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): PANU=%d DID=%d SAP=%d CTN1=%d CTN2=%d SYNCML=%d AVP_COVERART=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _connectedService.getBit(BTS_SRV_FUNC_PANU),
            _connectedService.getBit(BTS_SRV_FUNC_DID),
            _connectedService.getBit(BTS_SRV_FUNC_SAP),
            _connectedService.getBit(BTS_SRV_FUNC_CTN1),
            _connectedService.getBit(BTS_SRV_FUNC_CTN2),
            _connectedService.getBit(BTS_SRV_FUNC_SYNCML),
            _connectedService.getBit(BTS_SRV_FUNC_AVP_COVERART)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_connectedService.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_connectedService.getData()      ) & 0x00000000FFFFFFFF);
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s connectedService=0x%08X%08X device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            u1, u2 /*DONE: format*/,
            _device.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): SPP=%d HFP=%d A2DP=%d AVRCP=%d PIM_HFP=%d PIM_SPP=%d PIM_PBAP=%d PIM_SYNCML=%d",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _connectedService.getBit(BTS_SRV_FUNC_SPP),
            _connectedService.getBit(BTS_SRV_FUNC_HFP),
            _connectedService.getBit(BTS_SRV_FUNC_A2DP),
            _connectedService.getBit(BTS_SRV_FUNC_AVRCP),
            _connectedService.getBit(BTS_SRV_FUNC_PIM_HFP),
            _connectedService.getBit(BTS_SRV_FUNC_PIM_SPP),
            _connectedService.getBit(BTS_SRV_FUNC_PIM_PBAP),
            _connectedService.getBit(BTS_SRV_FUNC_PIM_SYNCML)));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): MSG_HFP=%d MSG_MAP=%d MSG_SPP=%d DUN=%d OPPS=%d OPPC=%d BIPR=%d BIPI=%d",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _connectedService.getBit(BTS_SRV_FUNC_MSG_HFP),
            _connectedService.getBit(BTS_SRV_FUNC_MSG_MAP),
            _connectedService.getBit(BTS_SRV_FUNC_MSG_SPP),
            _connectedService.getBit(BTS_SRV_FUNC_DUN),
            _connectedService.getBit(BTS_SRV_FUNC_OPPS),
            _connectedService.getBit(BTS_SRV_FUNC_OPPC),
            _connectedService.getBit(BTS_SRV_FUNC_BIPR),
            _connectedService.getBit(BTS_SRV_FUNC_BIPI)));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): PANU=%d DID=%d SAP=%d CTN1=%d CTN2=%d SYNCML=%d AVP_COVERART=%d",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _connectedService.getBit(BTS_SRV_FUNC_PANU),
            _connectedService.getBit(BTS_SRV_FUNC_DID),
            _connectedService.getBit(BTS_SRV_FUNC_SAP),
            _connectedService.getBit(BTS_SRV_FUNC_CTN1),
            _connectedService.getBit(BTS_SRV_FUNC_CTN2),
            _connectedService.getBit(BTS_SRV_FUNC_SYNCML),
            _connectedService.getBit(BTS_SRV_FUNC_AVP_COVERART)));

#endif
}

void Ipc2Bts_DeviceVersionUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d version=0x%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _version,
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s version=0x%04X device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _version,
            _device.c_str()));

#endif
}

void Ipc2Bts_DevicePbapInfoUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d valid=%d version=0x%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _valid),
            _pbapInfo.version,
            _device.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): repositories=0x%02X (LOCAL_PHONEBOOK=%d SIM_CARD=%d SPEED_DIAL=%d FAVORITES=%d) features=0x%08X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _pbapInfo.repositories.getData(),
            _pbapInfo.repositories.getBit(BTS_PBAP_REPO_LOCAL_PHONEBOOK),
            _pbapInfo.repositories.getBit(BTS_PBAP_REPO_SIM_CARD),
            _pbapInfo.repositories.getBit(BTS_PBAP_REPO_SPEED_DIAL),
            _pbapInfo.repositories.getBit(BTS_PBAP_REPO_FAVORITES),
            _pbapInfo.features.getData()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s valid=%s version=0x%04X device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_valid),
            _pbapInfo.version,
            _device.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): repositories=0x%02X (LOCAL_PHONEBOOK=%d SIM_CARD=%d SPEED_DIAL=%d FAVORITES=%d) features=0x%08X",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _pbapInfo.repositories.getData(),
            _pbapInfo.repositories.getBit(BTS_PBAP_REPO_LOCAL_PHONEBOOK),
            _pbapInfo.repositories.getBit(BTS_PBAP_REPO_SIM_CARD),
            _pbapInfo.repositories.getBit(BTS_PBAP_REPO_SPEED_DIAL),
            _pbapInfo.repositories.getBit(BTS_PBAP_REPO_FAVORITES),
            _pbapInfo.features.getData()));

#endif
}

void Ipc2Bts_DeviceModeUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d mode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBtPowerMode, _mode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s mode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBtPowerMode2String(_mode),
            _device.c_str()));

#endif
}

void Ipc2Bts_DeviceRoleUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d master=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _master),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s master=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_master),
            _device.c_str()));

#endif
}

void Ipc2Bts_DeviceEnableAvpPauseUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d enableAvpPause=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _enableAvpPause),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s enableAvpPause=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_enableAvpPause),
            _device.c_str()));

#endif
}

void Ipc2Bts_BtApplDeviceConnectReqResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_serviceFunction.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_serviceFunction.getData()      ) & 0x00000000FFFFFFFF);
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X serviceFunction=0x%08X%08X instance=%u opId=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            u1, u2 /*DONE: format*/,
            _instance,
            _opId));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_serviceFunction.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_serviceFunction.getData()      ) & 0x00000000FFFFFFFF);
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X serviceFunction=0x%08X%08X instance=%u opId=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            u1, u2 /*DONE: format*/,
            _instance,
            _opId));

#endif
}

void Ipc2Bts_BtApplDeviceDisconnectReqResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_serviceFunction.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_serviceFunction.getData()      ) & 0x00000000FFFFFFFF);
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X serviceFunction=0x%08X%08X instance=%u opId=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            u1, u2 /*DONE: format*/,
            _instance,
            _opId));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_serviceFunction.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_serviceFunction.getData()      ) & 0x00000000FFFFFFFF);
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X serviceFunction=0x%08X%08X instance=%u opId=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            u1, u2 /*DONE: format*/,
            _instance,
            _opId));

#endif
}

void Ipc2Bts_BtApplCancelConnectReqResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_BtApplGetHwVersionReqResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_BtApplSetConfigurationReqResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d instance=%u configId=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _instance,
            ETG_ENUM(TRC_BTSEvoConfigIdEnum, _configId)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s instance=%u configId=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _instance,
            getEvoConfigIdEnum2String(_configId)));

#endif
}

void Ipc2Bts_BtApplReadConfigurationReqResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d instance=%u configId=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _instance,
            ETG_ENUM(TRC_BTSEvoConfigIdEnum, _configId)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s instance=%u configId=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _instance,
            getEvoConfigIdEnum2String(_configId)));

#endif
}

void Ipc2Bts_BtApplTestModeReqResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_BtApplTestModeLinkQualityReqResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_BtApplDeviceConnectCfm::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d status=%d opId=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSStatusCode, _status),
            _opId));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s status=%s opId=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getStatusCode2String(_status),
            _opId));

#endif
}

void Ipc2Bts_BtApplDeviceConnectInd::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_serviceFunction.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_serviceFunction.getData()      ) & 0x00000000FFFFFFFF);
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X status=%d serviceFunction=0x%08X%08X instance=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSStatusCode, _status),
            u1, u2 /*DONE: format*/,
            _instance));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_serviceFunction.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_serviceFunction.getData()      ) & 0x00000000FFFFFFFF);
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X status=%s serviceFunction=0x%08X%08X instance=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            getStatusCode2String(_status),
            u1, u2 /*DONE: format*/,
            _instance));

#endif
}

void Ipc2Bts_BtApplDeviceDisconnectCfm::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d status=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSStatusCode, _status)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s status=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getStatusCode2String(_status)));

#endif
}

void Ipc2Bts_BtApplDeviceDisconnectInd::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_serviceFunction.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_serviceFunction.getData()      ) & 0x00000000FFFFFFFF);
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X status=%d reason=%d serviceFunction=0x%08X%08X instance=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSStatusCode, _status),
            ETG_ENUM(TRC_BTSBtApplDisconnectReason, _reason),
            u1, u2 /*DONE: format*/,
            _instance));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_serviceFunction.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_serviceFunction.getData()      ) & 0x00000000FFFFFFFF);
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X status=%s reason=%s serviceFunction=0x%08X%08X instance=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            getStatusCode2String(_status),
            getBtApplDisconnectReason2String(_reason),
            u1, u2 /*DONE: format*/,
            _instance));

#endif
}

void Ipc2Bts_BtApplDeviceDisconnectCompInd::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d opId=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _opId));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s opId=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _opId));

#endif
}

void Ipc2Bts_BtApplDeviceConnectCompInd::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d opId=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _opId));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s opId=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _opId));

#endif
}

void Ipc2Bts_BtApplCancelConnectCfm::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d status=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSStatusCode, _status)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s status=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getStatusCode2String(_status)));

#endif
}

void Ipc2Bts_BtApplStatusNotificationInd::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d deviceName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _deviceName.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s deviceName=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _deviceName.c_str()));

#endif
}

void Ipc2Bts_BtApplGetHwVersionCfm::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int binSize = (unsigned int)_extInfo.size();
   unsigned char* binData = &(_extInfo[0]);
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d status=%d btCoreVersion=%d manufacturer=0x%04X hwSubVersion=0x%04X extInfo=%02X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSStatusCode, _status),
            ETG_ENUM(TRC_BTSBtCoreVersion, _btCoreVersion),
            _manufacturer,
            _hwSubVersion,
            ETG_LIST(binSize, ETG_T8, binData) /*DONE: _extInfo format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int binSize = (unsigned int)_extInfo.size();
   unsigned char* binData = &(_extInfo[0]);
   char tmpBuffer[1024] = { 0 };
   for(unsigned int i = 0; (i < binSize) && (i < ((sizeof(tmpBuffer) - 1) / 3)); i++)
   {
      sprintf(&tmpBuffer[3 * i], "%02X ", binData[i]);
   }
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s status=%s btCoreVersion=%s manufacturer=0x%04X hwSubVersion=0x%04X extInfo=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getStatusCode2String(_status),
            getBtCoreVersion2String(_btCoreVersion),
            _manufacturer,
            _hwSubVersion,
            tmpBuffer /*DONE: _extInfo format*/));

#endif
}

void Ipc2Bts_BtApplSetConfigurationCfm::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d status=%d instance=%u configId=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSStatusCode, _status),
            _instance,
            ETG_ENUM(TRC_BTSEvoConfigIdEnum, _configId)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s status=%s instance=%u configId=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getStatusCode2String(_status),
            _instance,
            getEvoConfigIdEnum2String(_configId)));

#endif
}

void Ipc2Bts_BtApplReadConfigurationCfm::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int binSize = (unsigned int)_data.size();
   unsigned char* binData = &(_data[0]);
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d status=%d instance=%u configId=%d configLength=%u data=%02X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSStatusCode, _status),
            _instance,
            ETG_ENUM(TRC_BTSEvoConfigIdEnum, _configId),
            binSize,
            ETG_LIST(binSize, ETG_T8, binData) /*DONE: _data format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int binSize = (unsigned int)_data.size();
   unsigned char* binData = &(_data[0]);
   char tmpBuffer[1024] = { 0 };
   for(unsigned int i = 0; (i < binSize) && (i < ((sizeof(tmpBuffer) - 1) / 3)); i++)
   {
      sprintf(&tmpBuffer[3 * i], "%02X ", binData[i]);
   }
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s status=%s instance=%u configId=%s configLength=%u data=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getStatusCode2String(_status),
            _instance,
            getEvoConfigIdEnum2String(_configId),
            binSize,
            tmpBuffer /*DONE: _data format*/));

#endif
}

void Ipc2Bts_BtApplTestModeCfm::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d status=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSStatusCode, _status)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s status=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getStatusCode2String(_status)));

#endif
}

void Ipc2Bts_BtApplTestModeCompInd::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d enable=%d status=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _enable),
            ETG_ENUM(TRC_BTSStatusCode, _status)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s enable=%s status=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_enable),
            getStatusCode2String(_status)));

#endif
}

void Ipc2Bts_BtApplTestModeLinkQualityCfm::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d status=%d rssi=%d linkQuality=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSStatusCode, _status),
            _rssi,
            _linkQuality));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s status=%s rssi=%d linkQuality=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getStatusCode2String(_status),
            _rssi,
            _linkQuality));

#endif
}

void Ipc2Bts_ConnectResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d service=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _service.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s service=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _service.c_str()));

#endif
}

void Ipc2Bts_DisconnectResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d service=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _service.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s service=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _service.c_str()));

#endif
}

void Ipc2Bts_ServiceRemoteUuidUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d protocol=%d uuid=%40s service=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSProtocolId, _protocol),
            _uuid.c_str() /*DONE: format*/,
            _service.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s protocol=%s uuid=%40s service=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getProtocolId2String(_protocol),
            _uuid.c_str() /*DONE: format*/,
            _service.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_ServiceStateUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d serviceState=%d service=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSServiceState, _serviceState),
            _service.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s serviceState=%s service=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getServiceState2String(_serviceState),
            _service.c_str()));

#endif
}

void Ipc2Bts_ServiceDeviceUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): service=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _service.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): service=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _service.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_ServiceVersionUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d version=0x%04X service=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _version,
            _service.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s version=0x%04X service=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _version,
            _service.c_str()));

#endif
}

void Ipc2Bts_DebugTraceConfigResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_ProtocolTraceConfigResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_ReportDebugTrace::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int binSize = (unsigned int)_traceData.size();
   unsigned char* binData = &(_traceData[0]);
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d traceLocation=%u traceLevel=%u traceCategory=%d errorCode=0x%04X traceLength=%u traceData=%02X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _traceLocation,
            _traceLevel,
            ETG_ENUM(TRC_BTSDebugTraceCategory, _traceCategory),
            _errorCode,
            binSize,
            ETG_LIST(binSize, ETG_T8, binData) /*DONE: _traceData format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int binSize = (unsigned int)_traceData.size();
   unsigned char* binData = &(_traceData[0]);
   char tmpBuffer[1024] = { 0 };
   for(unsigned int i = 0; (i < binSize) && (i < ((sizeof(tmpBuffer) - 1) / 3)); i++)
   {
      sprintf(&tmpBuffer[3 * i], "%02X ", binData[i]);
   }
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s traceLocation=%u traceLevel=%u traceCategory=%s errorCode=0x%04X traceLength=%u traceData=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _traceLocation,
            _traceLevel,
            getDebugTraceCategory2String(_traceCategory),
            _errorCode,
            binSize,
            tmpBuffer /*DONE: _traceData format*/));

#endif
}

void Ipc2Bts_ReportProtocolTrace::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int binSize = (unsigned int)_traceData.size();
   unsigned char* binData = &(_traceData[0]);
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d traceType=0x%04X traceDirection=%d traceLength=%u traceData=%02X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _traceType,
            ETG_ENUM(TRC_BTSDebugTraceDirection, _traceDirection),
            binSize,
            ETG_LIST(binSize, ETG_T8, binData) /*DONE: _traceData format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int binSize = (unsigned int)_traceData.size();
   unsigned char* binData = &(_traceData[0]);
   char tmpBuffer[1024] = { 0 };
   for(unsigned int i = 0; (i < binSize) && (i < ((sizeof(tmpBuffer) - 1) / 3)); i++)
   {
      sprintf(&tmpBuffer[3 * i], "%02X ", binData[i]);
   }
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s traceType=0x%04X traceDirection=%s traceLength=%u traceData=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _traceType,
            getDebugTraceDirection2String(_traceDirection),
            binSize,
            tmpBuffer /*DONE: _traceData format*/));

#endif
}

void Ipc2Bts_FatalError::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d errLocation=%u errType=0x%08X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _errLocation,
            _errType));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s errLocation=%u errType=0x%08X",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _errLocation,
            _errType));

#endif
}

void Ipc2Bts_GeneralError::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d status=%d binaryStatus=0x%04X opcode=0x%04X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSStatusCode, _status),
            _binaryStatus,
            _opcode));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s status=%s binaryStatus=0x%04X opcode=0x%04X",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getStatusCode2String(_status),
            _binaryStatus,
            _opcode));

#endif
}

void Ipc2Bts_ReportGeniviDebugTrace::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int binSize = (unsigned int)_data.size();
   unsigned char* binData = &(_data[0]);
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d errFile=%u errLine=%u level=%u category=%u errorCode=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _errFile,
            _errLine,
            _level,
            _category,
            _errorCode));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): length=%u data=%02X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            binSize,
            ETG_LIST(binSize, ETG_T8, binData) /*DONE: _data format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int binSize = (unsigned int)_data.size();
   unsigned char* binData = &(_data[0]);
   char tmpBuffer[1024] = { 0 };
   for(unsigned int i = 0; (i < binSize) && (i < ((sizeof(tmpBuffer) - 1) / 3)); i++)
   {
      sprintf(&tmpBuffer[3 * i], "%02X ", binData[i]);
   }
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s errFile=%u errLine=%u level=%u category=%u errorCode=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _errFile,
            _errLine,
            _level,
            _category,
            _errorCode));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): length=%u data=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            binSize,
            tmpBuffer /*DONE: _data format*/));

#endif
}

void Ipc2Bts_GeniviFatalError::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d errFile=%u errLine=%u errCode=%u fmt=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _errFile,
            _errLine,
            _errCode,
            _fmt.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errValue=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _errValue.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s errFile=%u errLine=%u errCode=%u fmt=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _errFile,
            _errLine,
            _errCode,
            _fmt.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errValue=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _errValue.c_str()));

#endif
}

void Ipc2Bts_BtApplDidSetReqResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_BtApplDidSetCfm::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d status=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSStatusCode, _status)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s status=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getStatusCode2String(_status)));

#endif
}

void Ipc2Bts_DunDeviceFilePathUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d deviceName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _deviceName.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s deviceName=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _deviceName.c_str()));

#endif
}

void Ipc2Bts_BtApplSppSetUuidReqResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_BtApplSppSetUuidCfm::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d instance=%u status=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _instance,
            ETG_ENUM(TRC_BTSStatusCode, _status)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s instance=%u status=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _instance,
            getStatusCode2String(_status)));

#endif
}

void Ipc2Bts_BtApplSppVirtualDeviceInd::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d instance=%u deviceName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _instance,
            _deviceName.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s instance=%u deviceName=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _instance,
            _deviceName.c_str()));

#endif
}

void Ipc2Bts_CreateSessionResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X protocolId=%d instanceId=%u session=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSProtocolId, _protocolId),
            _instanceId,
            _session.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X protocolId=%s instanceId=%u session=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            getProtocolId2String(_protocolId),
            _instanceId,
            _session.c_str()));

#endif
}

void Ipc2Bts_RemoveSessionResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X protocolId=%d instanceId=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSProtocolId, _protocolId),
            _instanceId));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X protocolId=%s instanceId=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            getProtocolId2String(_protocolId),
            _instanceId));

#endif
}

void Ipc2Bts_CancelSessionResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X protocolId=%d instanceId=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSProtocolId, _protocolId),
            _instanceId));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X protocolId=%s instanceId=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            getProtocolId2String(_protocolId),
            _instanceId));

#endif
}

void Ipc2Bts_SessionCreated::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X instanceId=%u protocol=%d session=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _instanceId,
            ETG_ENUM(TRC_BTSProtocolId, _protocol),
            _session.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X instanceId=%u protocol=%s session=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _instanceId,
            getProtocolId2String(_protocol),
            _session.c_str()));

#endif
}

void Ipc2Bts_SessionRemoved::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X instanceId=%u protocol=%d session=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _instanceId,
            ETG_ENUM(TRC_BTSProtocolId, _protocol),
            _session.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X instanceId=%u protocol=%s session=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _instanceId,
            getProtocolId2String(_protocol),
            _session.c_str()));

#endif
}

void Ipc2Bts_AddTechnologyObjectPath::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_DelTechnologyObjectPath::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_AddNetworkObjectPathMapping::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _objPath.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_DelNetworkObjectPathMapping::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_GetConnManManagerPropertiesResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d state=%d offlineMode=%d sessionMode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_CCDBUSIFConnManManagerStateSuppValues, _properties.state),
            ETG_ENUM(TRC_BTSBool, _properties.offlineMode),
            ETG_ENUM(TRC_BTSBool, _properties.sessionMode)));

#else

   ::ccdbusif::connman::ConnManDbusParser parser;
   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s state=%s offlineMode=%s sessionMode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            parser.getManagerState2String(_properties.state).c_str(),
            getBool2String(_properties.offlineMode),
            getBool2String(_properties.sessionMode)));

#endif
}

void Ipc2Bts_GetConnManTechnologiesResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d technologies.size()=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _technologies.size()));
   for(size_t i = 0; i < _technologies.size(); i++)
   {
      const ConnManTechnologyItem& item = _technologies[i];
      ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): technologies[%u]: powered=%d connected=%d type=%d tethering=%d name=%20s objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            i,
            ETG_ENUM(TRC_BTSBool, item.technology.powered),
            ETG_ENUM(TRC_BTSBool, item.technology.connected),
            ETG_ENUM(TRC_CCDBUSIFConnManServiceTypeSuppValues, item.technology.type),
            ETG_ENUM(TRC_BTSBool, item.technology.tethering),
            item.technology.name.c_str(),
            item.objPath.c_str()));
   }

#else

   ::ccdbusif::connman::ConnManDbusParser parser;
   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s technologies.size()=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _technologies.size()));
   for(size_t i = 0; i < _technologies.size(); i++)
   {
      const ConnManTechnologyItem& item = _technologies[i];
      BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): technologies[%u]: powered=%s connected=%s type=%s tethering=%s name=%20s objPath=%s",
               getMessageTraceId(),
               getGenIpc2BtsOpcode2String(opcode),
               i,
               getBool2String(item.technology.powered),
               getBool2String(item.technology.connected),
               parser.getServiceType2String(item.technology.type).c_str(),
               getBool2String(item.technology.tethering),
               item.technology.name.c_str(),
               item.objPath.c_str()));
   }

#endif
}

void Ipc2Bts_GetConnManServicesResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d services.size()=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _services.size()));
   for(size_t i = 0; i < _services.size(); i++)
   {
      const ConnManServiceItem& item = _services[i];
      ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): services[%u]: type=%d state=%d autoConnect=%d name=%20s interface=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            i,
            ETG_ENUM(TRC_CCDBUSIFConnManServiceTypeSuppValues, item.service.type),
            ETG_ENUM(TRC_CCDBUSIFConnManServiceStateSuppValues, item.service.state),
            ETG_ENUM(TRC_BTSBool, item.service.autoConnect),
            item.service.name.c_str(),
            item.service.interface.c_str()));
      ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): services[%u]: objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            i,
            item.objPath.c_str()));
   }

#else

   ::ccdbusif::connman::ConnManDbusParser parser;
   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s services.size()=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _services.size()));
   for(size_t i = 0; i < _services.size(); i++)
   {
      const ConnManServiceItem& item = _services[i];
      BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): services[%u]: type=%s state=%s autoConnect=%s name=%20s interface=%s",
               getMessageTraceId(),
               getGenIpc2BtsOpcode2String(opcode),
               i,
               parser.getServiceType2String(item.service.type).c_str(),
               parser.getServiceState2String(item.service.state).c_str(),
               getBool2String(item.service.autoConnect),
               item.service.name.c_str(),
               item.service.interface.c_str()));
      BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): services[%u]: objPath=%s",
               getMessageTraceId(),
               getGenIpc2BtsOpcode2String(opcode),
               i,
               item.objPath.c_str()));
   }

#endif
}

void Ipc2Bts_ConnManTechnologyAdded::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d path=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _path.c_str()));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): powered=%d connected=%d type=%d tethering=%d name=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSBool, _properties.powered),
            ETG_ENUM(TRC_BTSBool, _properties.connected),
            ETG_ENUM(TRC_CCDBUSIFConnManServiceTypeSuppValues, _properties.type),
            ETG_ENUM(TRC_BTSBool, _properties.tethering),
            _properties.name.c_str()));

#else

   ::ccdbusif::connman::ConnManDbusParser parser;
   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s path=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _path.c_str()));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): powered=%s connected=%s type=%s tethering=%s name=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getBool2String(_properties.powered),
            getBool2String(_properties.connected),
            parser.getServiceType2String(_properties.type).c_str(),
            getBool2String(_properties.tethering),
            _properties.name.c_str()));

#endif
}

void Ipc2Bts_ConnManTechnologyRemoved::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d path=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _path.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s path=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _path.c_str()));

#endif
}

void Ipc2Bts_ConnManServicesChanged::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d services.size()=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _services.size()));
   for(size_t i = 0; i < _services.size(); i++)
   {
      const ConnManServiceItem& item = _services[i];
      ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): services[%u]: type=%d state=%d autoConnect=%d name=%20s interface=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            i,
            ETG_ENUM(TRC_CCDBUSIFConnManServiceTypeSuppValues, item.service.type),
            ETG_ENUM(TRC_CCDBUSIFConnManServiceStateSuppValues, item.service.state),
            ETG_ENUM(TRC_BTSBool, item.service.autoConnect),
            item.service.name.c_str(),
            item.service.interface.c_str()));
      ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): services[%u]: objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            i,
            item.objPath.c_str()));
   }

#else

   ::ccdbusif::connman::ConnManDbusParser parser;
   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s services.size()=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _services.size()));
   for(size_t i = 0; i < _services.size(); i++)
   {
      const ConnManServiceItem& item = _services[i];
      BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): services[%u]: type=%s state=%s autoConnect=%s name=%20s interface=%s",
               getMessageTraceId(),
               getGenIpc2BtsOpcode2String(opcode),
               i,
               parser.getServiceType2String(item.service.type).c_str(),
               parser.getServiceState2String(item.service.state).c_str(),
               getBool2String(item.service.autoConnect),
               item.service.name.c_str(),
               item.service.interface.c_str()));
      BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): services[%u]: objPath=%s",
               getMessageTraceId(),
               getGenIpc2BtsOpcode2String(opcode),
               i,
               item.objPath.c_str()));
   }

#endif
}

void Ipc2Bts_ConnManServicesRemoved::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d services.size()=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _services.size()));
   for(size_t i = 0; i < _services.size(); i++)
   {
      ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): services[%u]=%s",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
               i,
               _services[i].c_str()));
   }

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s services.size()=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _services.size()));
   for(size_t i = 0; i < _services.size(); i++)
   {
      BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): services[%u]=%s",
               getMessageTraceId(),
               getGenIpc2BtsOpcode2String(opcode),
               i,
               _services[i].c_str()));
   }

#endif
}

void Ipc2Bts_GetConnManServicePropertiesResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d type=%d state=%d autoConnect=%d name=%20s interface=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_CCDBUSIFConnManServiceTypeSuppValues, _properties.type),
            ETG_ENUM(TRC_CCDBUSIFConnManServiceStateSuppValues, _properties.state),
            ETG_ENUM(TRC_BTSBool, _properties.autoConnect),
            _properties.name.c_str(),
            _properties.interface.c_str()));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _device.c_str()));

#else

   ::ccdbusif::connman::ConnManDbusParser parser;
   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s type=%s state=%s autoConnect=%s name=%20s interface=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            parser.getServiceType2String(_properties.type).c_str(),
            parser.getServiceState2String(_properties.state).c_str(),
            getBool2String(_properties.autoConnect),
            _properties.name.c_str(),
            _properties.interface.c_str()));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _device.c_str()));

#endif
}

void Ipc2Bts_ConnManServiceConnectResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_ConnManServiceDisconnectResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_ConnManServiceStateUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d value=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_CCDBUSIFConnManServiceStateSuppValues, _value),
            _device.c_str()));

#else

   ::ccdbusif::connman::ConnManDbusParser parser;
   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s value=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            parser.getServiceState2String(_value).c_str(),
            _device.c_str()));

#endif
}

void Ipc2Bts_ConnManServiceInterfaceUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d interface=%20s device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _interface.c_str() /*DONE: format*/,
            _device.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s interface=%20s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _interface.c_str() /*DONE: format*/,
            _device.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_SerialDeviceFilePath::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d deviceName=%50s service=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _deviceName.c_str() /*DONE: format*/,
            _service.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s deviceName=%50s service=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _deviceName.c_str() /*DONE: format*/,
            _service.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_CharacterDeviceAddedRemoved::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d added=%d deviceName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _added),
            _deviceName.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s added=%s deviceName=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_added),
            _deviceName.c_str()));

#endif
}

void Ipc2Bts_CreateLeDeviceResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_RemoveLeDeviceResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_LeAdapterInitialization::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_LeDeviceCreated::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_LeDeviceRemoved::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_LeAdapterAddressTypeUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d addressType=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBLEDeviceAddressType, _addressType)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s addressType=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBLEDeviceAddressType2String(_addressType)));

#endif
}

void Ipc2Bts_LeAdapterAddressUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower()));

#endif
}

void Ipc2Bts_LeAdapterAliasUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d alias=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _alias.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s alias=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _alias.c_str()));

#endif
}

void Ipc2Bts_LeAdapterAdvertisingTimeoutUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d advertisingTimeout=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _advertisingTimeout));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s advertisingTimeout=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _advertisingTimeout));

#endif
}

void Ipc2Bts_LeAdapterDiscoverableModeUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d discoverableMode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBLEDiscoverableMode, _discoverableMode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s discoverableMode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBLEDiscoverableMode2String(_discoverableMode)));

#endif
}

void Ipc2Bts_LeAdapterAdvertisingDataUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int binSize = (unsigned int)_advertisingData.size();
   unsigned char* binData = &(_advertisingData[0]);
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d length=%u data=%02X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            binSize,
            ETG_LIST(binSize, ETG_T8, binData) /*DONE: _advertisingData format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int binSize = (unsigned int)_advertisingData.size();
   unsigned char* binData = &(_advertisingData[0]);
   char tmpBuffer[1024] = { 0 };
   for(unsigned int i = 0; (i < binSize) && (i < ((sizeof(tmpBuffer) - 1) / 3)); i++)
   {
      sprintf(&tmpBuffer[3 * i], "%02X ", binData[i]);
   }
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s length=%u data=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            binSize,
            tmpBuffer /*DONE: _advertisingData format*/));

#endif
}

void Ipc2Bts_LeAdapterAdvertiseEnableUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d advertiseEnable=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _advertiseEnable)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s advertiseEnable=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_advertiseEnable)));

#endif
}

void Ipc2Bts_AddLeDeviceObjectPathMapping::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _objPath.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_DelLeDeviceObjectPathMapping::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_LeDeviceDisconnectResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_LeDisconnectionComp::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d status=%d reason=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBLEStatus, _status),
            ETG_ENUM(TRC_BTSBtApplDisconnectReason, _reason),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s status=%s reason=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBLEStatus2String(_status),
            getBtApplDisconnectReason2String(_reason),
            _device.c_str()));

#endif
}

void Ipc2Bts_LeDeviceAddressTypeUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d addressType=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBLEDeviceAddressType, _addressType),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s addressType=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBLEDeviceAddressType2String(_addressType),
            _device.c_str()));

#endif
}

void Ipc2Bts_LeDeviceAddressUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Ipc2Bts_LeDeviceConnectedUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d connected=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _connected),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s connected=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_connected),
            _device.c_str()));

#endif
}

void Ipc2Bts_SendSeamlessPairingErrorCodeResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_SeamlessPairingAllowAccessResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_SeamlessPairingErrorCodeReceived::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d status=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBLEStatus, _status),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s status=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBLEStatus2String(_status),
            _device.c_str()));

#endif
}

void Ipc2Bts_SeamlessPairingAccessRequest::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _device.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _device.c_str()));

#endif
}

void Ipc2Bts_SeamlessPairingPairingDataUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int binSize = (unsigned int)_pairingData.size();
   unsigned char* binData = &(_pairingData[0]);
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d length=%u data=%02X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            binSize,
            ETG_LIST(binSize, ETG_T8, binData) /*DONE: _pairingData format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _device.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   unsigned int binSize = (unsigned int)_pairingData.size();
   unsigned char* binData = &(_pairingData[0]);
   char tmpBuffer[1024] = { 0 };
   for(unsigned int i = 0; (i < binSize) && (i < ((sizeof(tmpBuffer) - 1) / 3)); i++)
   {
      sprintf(&tmpBuffer[3 * i], "%02X ", binData[i]);
   }
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s length=%u data=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            binSize,
            tmpBuffer /*DONE: _pairingData format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): device=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _device.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_ServiceAvailabilityTelephony::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d interface=%d availabilityEvent=%d busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSGenDbusServiceInterface, _interface),
            ETG_ENUM(TRC_BTSDbusServiceAvailability, _availabilityEvent),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s interface=%s availabilityEvent=%s busType=%d busName=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getGenDbusServiceInterface2String(_interface),
            getDbusServiceAvailability2String(_availabilityEvent),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_AddHfpInterfaceObjectPathMapping::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d interface=%d objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_CCDBUSIFInterface, _interface),
            _objPath.c_str()));

#else

   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s interface=%s objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            parser.getInterface2StringTrace(_interface).c_str(),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_DelHfpInterfaceObjectPathMapping::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d interface=%d objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_CCDBUSIFInterface, _interface),
            _objPath.c_str()));

#else

   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s interface=%s objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            parser.getInterface2StringTrace(_interface).c_str(),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_GetHandsfreeProperties::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d modem=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _modem.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s modem=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _modem.c_str()));

#endif
}

void Ipc2Bts_GetHandsfreePropertiesExt::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d properties.size()=%u modem=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _properties.size() /*TODO: format*/,
            _modem.c_str() /*TODO: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s properties.size()=%u modem=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _properties.size() /*TODO: format*/,
            _modem.c_str() /*TODO: format*/));

#endif
}

void Ipc2Bts_HandsfreeFeaturesUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d features=0x%08X (VOICE_RECOGNITION=%d ATTACH_VOICE_TAG=%d 3WAY=%d ECNR=%d) modem=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _features.getData() /*DONE: format*/,
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_VOICE_RECOGNITION),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_ATTACH_VOICE_TAG),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_3WAY),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_ECNR),
            _modem.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): (INBAND_RING=%d REJECT_INCOMING_CALL=%d ENHANCED_CALL_STATUS=%d ENHANCED_CALL_CONTROL=%d EXTENDED_ERROR_CODES=%d CODEC_NEGOTIATION=%d)",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_INBAND_RING),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_REJECT_INCOMING_CALL),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_ENHANCED_CALL_STATUS),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_ENHANCED_CALL_CONTROL),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_EXTENDED_ERROR_CODES),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_CODEC_NEGOTIATION)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s features=0x%08X (VOICE_RECOGNITION=%d ATTACH_VOICE_TAG=%d 3WAY=%d ECNR=%d) modem=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _features.getData() /*DONE: format*/,
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_VOICE_RECOGNITION),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_ATTACH_VOICE_TAG),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_3WAY),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_ECNR),
            _modem.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): (INBAND_RING=%d REJECT_INCOMING_CALL=%d ENHANCED_CALL_STATUS=%d ENHANCED_CALL_CONTROL=%d EXTENDED_ERROR_CODES=%d CODEC_NEGOTIATION=%d)",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_INBAND_RING),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_REJECT_INCOMING_CALL),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_ENHANCED_CALL_STATUS),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_ENHANCED_CALL_CONTROL),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_EXTENDED_ERROR_CODES),
            _features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_CODEC_NEGOTIATION)));

#endif
}

void Ipc2Bts_HandsfreeInbandRingingUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d inbandRingingEnabled=%d modem=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSBool, _inbandRingingEnabled),
            _modem.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s inbandRingingEnabled=%s modem=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getBool2String(_inbandRingingEnabled),
            _modem.c_str()));

#endif
}

void Ipc2Bts_GetModems::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode)));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode)));

#endif
}

void Ipc2Bts_GetModemsExt::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d modems.size()=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _modems.size() /*TODO: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s modems.size()=%u",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _modems.size() /*TODO: format*/));

#endif
}

void Ipc2Bts_ModemAdded::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X modem=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _modem.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X modem=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _modem.c_str()));

#endif
}

void Ipc2Bts_ModemAddedExt::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X properties.size()=%u modem=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _properties.size() /*TODO: format*/,
            _modem.c_str() /*TODO: format*/));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X properties.size()=%u modem=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _properties.size() /*TODO: format*/,
            _modem.c_str() /*TODO: format*/));

#endif
}

void Ipc2Bts_ModemRemoved::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d modem=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _modem.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s modem=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _modem.c_str()));

#endif
}

void Ipc2Bts_GetModemProperties::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d modem=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _modem.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s modem=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _modem.c_str()));

#endif
}

void Ipc2Bts_GetModemPropertiesExt::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d properties.size()=%u modem=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _properties.size() /*TODO: format*/,
            _modem.c_str() /*TODO: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s properties.size()=%u modem=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _properties.size() /*TODO: format*/,
            _modem.c_str() /*TODO: format*/));

#endif
}

void Ipc2Bts_ModemInterfacesUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d interfaces.size()=%u modem=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _interfaces.size() /*DONE: format*/,
            _modem.c_str() /*DONE: format*/));
   for(size_t i = 0; i < _interfaces.size(); i++)
   {
      ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): interfaces[%u]=%d",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
               i,
               ETG_ENUM(TRC_CCDBUSIFInterface, _interfaces[i]) /*DONE: _interfaces format*/));
   }

#else

   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s interfaces.size()=%u modem=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _interfaces.size() /*DONE: format*/,
            _modem.c_str() /*DONE: format*/));
   for(size_t i = 0; i < _interfaces.size(); i++)
   {
      BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): interfaces[%u]=%s",
               getMessageTraceId(),
               getGenIpc2BtsOpcode2String(opcode),
               i,
               parser.getInterface2StringTrace(_interfaces[i]).c_str() /*DONE: _interfaces format*/));
   }

#endif
}

void Ipc2Bts_ServiceAvailabilityPhonebook::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d interface=%d availabilityEvent=%d busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSGenDbusServiceInterface, _interface),
            ETG_ENUM(TRC_BTSDbusServiceAvailability, _availabilityEvent),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s interface=%s availabilityEvent=%s busType=%d busName=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getGenDbusServiceInterface2String(_interface),
            getDbusServiceAvailability2String(_availabilityEvent),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_PimSessionSourceUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _objPath.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_PimSessionDestinationUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _objPath.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_PimSessionChannelUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d channel=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _channel,
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s channel=%u objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _channel,
            _objPath.c_str()));

#endif
}

void Ipc2Bts_PimSessionTargetUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d target=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _target.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s target=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _target.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_PimTransferCancelResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_PimTransferStatusUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d status=%d objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSTransferStatus, _status),
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s status=%s objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getTransferStatus2String(_status),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_PimTransferSessionUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d session=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _session.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s session=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _session.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_PimTransferNameUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d name=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _name.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s name=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _name.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_PimTransferTypeUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d type=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _type.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s type=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _type.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_PimTransferTimeUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d time=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _time,
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s time=%u objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _time,
            _objPath.c_str()));

#endif
}

void Ipc2Bts_PimTransferSizeUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d size=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _size,
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s size=%u objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _size,
            _objPath.c_str()));

#endif
}

void Ipc2Bts_PimTransferTransferredUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d transferred=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _transferred,
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s transferred=%u objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _transferred,
            _objPath.c_str()));

#endif
}

void Ipc2Bts_PimTransferFilenameUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d filename=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _filename.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s filename=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _filename.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_ServiceAvailabilityMessaging::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d interface=%d availabilityEvent=%d busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSGenDbusServiceInterface, _interface),
            ETG_ENUM(TRC_BTSDbusServiceAvailability, _availabilityEvent),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s interface=%s availabilityEvent=%s busType=%d busName=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getGenDbusServiceInterface2String(_interface),
            getDbusServiceAvailability2String(_availabilityEvent),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_MsgSessionSourceUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _objPath.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_MsgSessionDestinationUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d btAddr=%08X%04X objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _objPath.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s btAddr=%08X%04X objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            traceAddr.getUpper(), traceAddr.getLower(),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_MsgSessionChannelUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d channel=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _channel,
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s channel=%u objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _channel,
            _objPath.c_str()));

#endif
}

void Ipc2Bts_MsgSessionTargetUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d target=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _target.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s target=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _target.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_MsgTransferCancelResult::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_MsgTransferStatusUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d status=%d objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSTransferStatus, _status),
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s status=%s objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getTransferStatus2String(_status),
            _objPath.c_str()));

#endif
}

void Ipc2Bts_MsgTransferSessionUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d session=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _session.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s session=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _session.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_MsgTransferNameUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d name=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _name.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s name=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _name.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_MsgTransferTypeUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d type=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _type.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s type=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _type.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_MsgTransferTimeUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d time=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _time,
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s time=%u objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _time,
            _objPath.c_str()));

#endif
}

void Ipc2Bts_MsgTransferSizeUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d size=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _size,
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s size=%u objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _size,
            _objPath.c_str()));

#endif
}

void Ipc2Bts_MsgTransferTransferredUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d transferred=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _transferred,
            _objPath.c_str()));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s transferred=%u objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _transferred,
            _objPath.c_str()));

#endif
}

void Ipc2Bts_MsgTransferFilenameUpdate::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d filename=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            _filename.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s filename=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            _filename.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Ipc2Bts_ServiceAvailabilityMediaPlayer::doInputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): errCode=%d interface=%d availabilityEvent=%d busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            ETG_ENUM(TRC_BTSIpcCommonErrorCode, _commonErrorCode),
            ETG_ENUM(TRC_BTSGenDbusServiceInterface, _interface),
            ETG_ENUM(TRC_BTSDbusServiceAvailability, _availabilityEvent),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Ipc2Bts: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenIpc2BtsOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)_opCode;
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): errCode=%s interface=%s availabilityEvent=%s busType=%d busName=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            getIpcCommonErrorCode2String(_commonErrorCode),
            getGenDbusServiceInterface2String(_interface),
            getDbusServiceAvailability2String(_availabilityEvent),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Ipc2Bts: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenIpc2BtsOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

} //genivi
} //btstackif
