/**
 * @file EvolutionGeniviDbusLEDeviceCallbackIf.cpp
 *
 * @par SW-Component
 * BtStackIf
 *
 * @brief EvolutionGenivi DBUS Callback Interface for LEDevice.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS Callback Interface for LEDevice.
 */

#include "EvolutionGeniviDbusLEDeviceCallbackIf.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "BtsUtils.h"
#include "EvolutionGeniviUtils.h"
#include "FwAssert.h"

using namespace ::org::bluez::LE_Device1;
using namespace ::asf::core;

namespace btstackif {
namespace genivi {

EvolutionGeniviDbusLEDeviceCallbackIf::EvolutionGeniviDbusLEDeviceCallbackIf()
{
}

EvolutionGeniviDbusLEDeviceCallbackIf::EvolutionGeniviDbusLEDeviceCallbackIf(IDbusRecHandler* recHandler) : DbusCallbackIf(recHandler)
{
}

EvolutionGeniviDbusLEDeviceCallbackIf::~EvolutionGeniviDbusLEDeviceCallbackIf()
{
}

// "ServiceAvailableIF" implementation --- start
void EvolutionGeniviDbusLEDeviceCallbackIf::onProxyAvailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(0 != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_BLE_DEVICE);
      ptrMsg->setObjPath(objPath);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}

void EvolutionGeniviDbusLEDeviceCallbackIf::onProxyUnavailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(0 != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_BLE_DEVICE);
      ptrMsg->setObjPath(objPath);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}
// "ServiceAvailableIF" implementation --- end

// org/bluez/LE_Device1Proxy implementation --- start
void EvolutionGeniviDbusLEDeviceCallbackIf::onDisconnectError(const ::boost::shared_ptr< LE_Device1Proxy >& proxy, const ::boost::shared_ptr< DisconnectError >& error)
{
   Ipc2Bts_LeDeviceDisconnectResult* ptrResultMsg = ptrNew_Ipc2Bts_LeDeviceDisconnectResult();
   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusLEDeviceCallbackIf::onDisconnectResponse(const ::boost::shared_ptr< LE_Device1Proxy >& proxy, const ::boost::shared_ptr< DisconnectResponse >& response)
{
   Ipc2Bts_LeDeviceDisconnectResult* ptrResultMsg = ptrNew_Ipc2Bts_LeDeviceDisconnectResult();
   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onResponse(ptrResultMsg, response);
}

void EvolutionGeniviDbusLEDeviceCallbackIf::onDisconnectionCompError(const ::boost::shared_ptr< LE_Device1Proxy >& proxy, const ::boost::shared_ptr< DisconnectionCompError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusLEDeviceCallbackIf::onDisconnectionCompSignal(const ::boost::shared_ptr< LE_Device1Proxy >& proxy, const ::boost::shared_ptr< DisconnectionCompSignal >& signal)
{
   Ipc2Bts_LeDisconnectionComp* ptrStatusMsg = ptrNew_Ipc2Bts_LeDisconnectionComp();
   if(0 != ptrStatusMsg)
   {
      ptrStatusMsg->setDevice(proxy->getDBusObjectPath());
      ptrStatusMsg->setStatus((BTSBLEStatus)signal->getStatus()); // cast OK because enum and Evolution value match
      ptrStatusMsg->setReason(convertBtApplDisconnectReason2InternalValue(signal->getReason())); // TODO: double check with ALPS
   }
   onSignal(ptrStatusMsg, signal->getAct());
}

void EvolutionGeniviDbusLEDeviceCallbackIf::onAddressTypeError(const ::boost::shared_ptr< LE_Device1Proxy >& proxy, const ::boost::shared_ptr< AddressTypeError >& error)
{
   Ipc2Bts_LeDeviceAddressTypeUpdate* ptrStatusMsg = ptrNew_Ipc2Bts_LeDeviceAddressTypeUpdate();
   if(0 != ptrStatusMsg)
   {
      ptrStatusMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrStatusMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusLEDeviceCallbackIf::onAddressTypeUpdate(const ::boost::shared_ptr< LE_Device1Proxy >& proxy, const ::boost::shared_ptr< AddressTypeUpdate >& update)
{
   Ipc2Bts_LeDeviceAddressTypeUpdate* ptrStatusMsg = ptrNew_Ipc2Bts_LeDeviceAddressTypeUpdate();
   if(0 != ptrStatusMsg)
   {
      ptrStatusMsg->setDevice(proxy->getDBusObjectPath());
      ptrStatusMsg->setAddressType((BTSBLEDeviceAddressType)update->getAddressType()); // cast OK because enum and Evolution value match
   }
   onUpdate(ptrStatusMsg, update->getAct());
}

void EvolutionGeniviDbusLEDeviceCallbackIf::onAddressError(const ::boost::shared_ptr< LE_Device1Proxy >& proxy, const ::boost::shared_ptr< AddressError >& error)
{
   Ipc2Bts_LeDeviceAddressUpdate* ptrStatusMsg = ptrNew_Ipc2Bts_LeDeviceAddressUpdate();
   if(0 != ptrStatusMsg)
   {
      ptrStatusMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrStatusMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusLEDeviceCallbackIf::onAddressUpdate(const ::boost::shared_ptr< LE_Device1Proxy >& proxy, const ::boost::shared_ptr< AddressUpdate >& update)
{
   Ipc2Bts_LeDeviceAddressUpdate* ptrStatusMsg = ptrNew_Ipc2Bts_LeDeviceAddressUpdate();
   if(0 != ptrStatusMsg)
   {
      ptrStatusMsg->setDevice(proxy->getDBusObjectPath());
      ptrStatusMsg->setBDAddress(update->getAddress());
   }
   onUpdate(ptrStatusMsg, update->getAct());
}

void EvolutionGeniviDbusLEDeviceCallbackIf::onConnectedError(const ::boost::shared_ptr< LE_Device1Proxy >& proxy, const ::boost::shared_ptr< ConnectedError >& error)
{
   Ipc2Bts_LeDeviceConnectedUpdate* ptrStatusMsg = ptrNew_Ipc2Bts_LeDeviceConnectedUpdate();
   if(0 != ptrStatusMsg)
   {
      ptrStatusMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrStatusMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusLEDeviceCallbackIf::onConnectedUpdate(const ::boost::shared_ptr< LE_Device1Proxy >& proxy, const ::boost::shared_ptr< ConnectedUpdate >& update)
{
   Ipc2Bts_LeDeviceConnectedUpdate* ptrStatusMsg = ptrNew_Ipc2Bts_LeDeviceConnectedUpdate();
   if(0 != ptrStatusMsg)
   {
      ptrStatusMsg->setDevice(proxy->getDBusObjectPath());
      ptrStatusMsg->setConnected(update->getConnected());
   }
   onUpdate(ptrStatusMsg, update->getAct());
}
// org/bluez/LE_Device1Proxy implementation --- end

} //genivi
} //btstackif
