/**
 * @file EvolutionGeniviDbusLEAdapterCallbackIf.cpp
 *
 * @par SW-Component
 * BtStackIf
 *
 * @brief EvolutionGenivi DBUS Callback Interface for LEAdapter.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS Callback Interface for LEAdapter.
 */

#include "EvolutionGeniviDbusLEAdapterCallbackIf.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "BtsUtils.h"
#include "EvolutionGeniviUtils.h"
#include "FwAssert.h"

using namespace ::org::bluez::LE_Adapter1;
using namespace ::asf::core;

namespace btstackif {
namespace genivi {

EvolutionGeniviDbusLEAdapterCallbackIf::EvolutionGeniviDbusLEAdapterCallbackIf()
{
}

EvolutionGeniviDbusLEAdapterCallbackIf::EvolutionGeniviDbusLEAdapterCallbackIf(IDbusRecHandler* recHandler) : DbusCallbackIf(recHandler)
{
}

EvolutionGeniviDbusLEAdapterCallbackIf::~EvolutionGeniviDbusLEAdapterCallbackIf()
{
}

// "ServiceAvailableIF" implementation --- start
void EvolutionGeniviDbusLEAdapterCallbackIf::onProxyAvailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);
   (void)(objPath);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(0 != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_BLE_ADAPTER);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onProxyUnavailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);
   (void)(objPath);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(0 != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_BLE_ADAPTER);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}
// "ServiceAvailableIF" implementation --- end

// org/bluez/LE_Adapter1Proxy implementation --- start
void EvolutionGeniviDbusLEAdapterCallbackIf::onCreateLeDeviceError(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< CreateLeDeviceError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_CreateLeDeviceResult(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onCreateLeDeviceResponse(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< CreateLeDeviceResponse >& response)
{
   (void)(proxy);
   Ipc2Bts_CreateLeDeviceResult* ptrResultMsg = ptrNew_Ipc2Bts_CreateLeDeviceResult();
   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setDevice(response->getDevice());
      onResponse(ptrResultMsg, response->getAct());
   }
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onRemoveLeDeviceError(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< RemoveLeDeviceError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_RemoveLeDeviceResult(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onRemoveLeDeviceResponse(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< RemoveLeDeviceResponse >& response)
{
   (void)(proxy);
   onResponse(ptrNew_Ipc2Bts_RemoveLeDeviceResult(), response);
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onLeAdapterInitializationError(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< LeAdapterInitializationError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onLeAdapterInitializationSignal(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< LeAdapterInitializationSignal >& signal)
{
   (void)(proxy);
   onSignal(ptrNew_Ipc2Bts_LeAdapterInitialization(), signal);
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onLeDeviceCreatedError(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< LeDeviceCreatedError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onLeDeviceCreatedSignal(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< LeDeviceCreatedSignal >& signal)
{
   (void)(proxy);
   Ipc2Bts_LeDeviceCreated* ptrStatusMsg = ptrNew_Ipc2Bts_LeDeviceCreated();
   if(0 != ptrStatusMsg)
   {
      ptrStatusMsg->setDevice(signal->getDevice());
      onSignal(ptrStatusMsg, signal->getAct());
   }
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onLeDeviceRemovedError(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< LeDeviceRemovedError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onLeDeviceRemovedSignal(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< LeDeviceRemovedSignal >& signal)
{
   (void)(proxy);
   Ipc2Bts_LeDeviceRemoved* ptrStatusMsg = ptrNew_Ipc2Bts_LeDeviceRemoved();
   if(0 != ptrStatusMsg)
   {
      ptrStatusMsg->setDevice(signal->getDevice());
      onSignal(ptrStatusMsg, signal->getAct());
   }
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onAddressTypeError(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< AddressTypeError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_LeAdapterAddressTypeUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onAddressTypeUpdate(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< AddressTypeUpdate >& update)
{
   (void)(proxy);
   Ipc2Bts_LeAdapterAddressTypeUpdate* ptrStatusMsg = ptrNew_Ipc2Bts_LeAdapterAddressTypeUpdate();
   if(0 != ptrStatusMsg)
   {
      ptrStatusMsg->setAddressType(BTSBLEDeviceAddressType(update->getAddressType())); // cast OK because enum and Evolution value match
      onUpdate(ptrStatusMsg, update->getAct());
   }
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onAddressError(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< AddressError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_LeAdapterAddressUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onAddressUpdate(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< AddressUpdate >& update)
{
   (void)(proxy);
   Ipc2Bts_LeAdapterAddressUpdate* ptrStatusMsg = ptrNew_Ipc2Bts_LeAdapterAddressUpdate();
   if(0 != ptrStatusMsg)
   {
      ptrStatusMsg->setBDAddress(update->getAddress());
      onUpdate(ptrStatusMsg, update->getAct());
   }
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onAliasError(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< AliasError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_LeAdapterAliasUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onAliasUpdate(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< AliasUpdate >& update)
{
   (void)(proxy);
   Ipc2Bts_LeAdapterAliasUpdate* ptrStatusMsg = ptrNew_Ipc2Bts_LeAdapterAliasUpdate();
   if(0 != ptrStatusMsg)
   {
      ptrStatusMsg->setAlias(update->getAlias());
      onUpdate(ptrStatusMsg, update->getAct());
   }
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onAdvertisingTimeoutError(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< AdvertisingTimeoutError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_LeAdapterAdvertisingTimeoutUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onAdvertisingTimeoutUpdate(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< AdvertisingTimeoutUpdate >& update)
{
   (void)(proxy);
   Ipc2Bts_LeAdapterAdvertisingTimeoutUpdate* ptrStatusMsg = ptrNew_Ipc2Bts_LeAdapterAdvertisingTimeoutUpdate();
   if(0 != ptrStatusMsg)
   {
      ptrStatusMsg->setAdvertisingTimeout(update->getAdvertisingTimeout());
      onUpdate(ptrStatusMsg, update->getAct());
   }
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onDiscoverableModeError(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< DiscoverableModeError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_LeAdapterDiscoverableModeUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onDiscoverableModeUpdate(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< DiscoverableModeUpdate >& update)
{
   (void)(proxy);
   Ipc2Bts_LeAdapterDiscoverableModeUpdate* ptrStatusMsg = ptrNew_Ipc2Bts_LeAdapterDiscoverableModeUpdate();
   if(0 != ptrStatusMsg)
   {
      ptrStatusMsg->setDiscoverableMode((BTSBLEDiscoverableMode)update->getDiscoverableMode()); // cast OK because enum and Evolution value match
      onUpdate(ptrStatusMsg, update->getAct());
   }
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onAdvertisingDataError(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< AdvertisingDataError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_LeAdapterAdvertisingDataUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onAdvertisingDataUpdate(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< AdvertisingDataUpdate >& update)
{
   (void)(proxy);
   Ipc2Bts_LeAdapterAdvertisingDataUpdate* ptrStatusMsg = ptrNew_Ipc2Bts_LeAdapterAdvertisingDataUpdate();
   if(0 != ptrStatusMsg)
   {
      ptrStatusMsg->setAdvertisingData(update->getAdvertisingData());
      onUpdate(ptrStatusMsg, update->getAct());
   }
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onAdvertiseEnableError(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< AdvertiseEnableError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_LeAdapterAdvertiseEnableUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusLEAdapterCallbackIf::onAdvertiseEnableUpdate(const ::boost::shared_ptr< LE_Adapter1Proxy >& proxy, const ::boost::shared_ptr< AdvertiseEnableUpdate >& update)
{
   (void)(proxy);
   Ipc2Bts_LeAdapterAdvertiseEnableUpdate* ptrStatusMsg = ptrNew_Ipc2Bts_LeAdapterAdvertiseEnableUpdate();
   if(0 != ptrStatusMsg)
   {
      ptrStatusMsg->setAdvertiseEnable(update->getAdvertiseEnable());
      onUpdate(ptrStatusMsg, update->getAct());
   }
}
// org/bluez/LE_Adapter1Proxy implementation --- end

} //genivi
} //btstackif
