/**
 * @file EvolutionGeniviDbusHfpModemCallbackIf.cpp
 *
 * @par SW-Component
 * BtStackIf
 *
 * @brief EvolutionGenivi DBUS Callback Interface for HfpModem.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS Callback Interface for HfpModem.
 */

#include "EvolutionGeniviDbusHfpModemCallbackIf.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "BtsUtils.h"
#include "EvolutionGeniviUtils.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"
#include "cc_dbus_if/IEvolutionGeniviDbusHfpHandsfreeSendRequestIf.h"

#include "FwAssert.h"

using namespace ::org::ofono::Modem;
using namespace ::asf::core;

namespace btstackif {
namespace genivi {

EvolutionGeniviDbusHfpModemCallbackIf::EvolutionGeniviDbusHfpModemCallbackIf()
{
   _hfpHandsfreeIf = NULL;
}

EvolutionGeniviDbusHfpModemCallbackIf::EvolutionGeniviDbusHfpModemCallbackIf(IDbusRecHandler* recHandler) : DbusCallbackIf(recHandler)
{
   _hfpHandsfreeIf = NULL;
}

EvolutionGeniviDbusHfpModemCallbackIf::~EvolutionGeniviDbusHfpModemCallbackIf()
{
   _hfpHandsfreeIf = NULL;
}

// "ServiceAvailableIF" implementation --- start
void EvolutionGeniviDbusHfpModemCallbackIf::onProxyAvailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   /*--- BTS_FB_CONNECTION start ----------------------------------------------------------------*/
   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_HFP_MODEM);
      ptrMsg->setObjPath(objPath);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
   /*--- BTS_FB_CONNECTION end ------------------------------------------------------------------*/
}

void EvolutionGeniviDbusHfpModemCallbackIf::onProxyUnavailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   /*--- BTS_FB_CONNECTION start ----------------------------------------------------------------*/
   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_HFP_MODEM);
      ptrMsg->setObjPath(objPath);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
   /*--- BTS_FB_CONNECTION end ------------------------------------------------------------------*/
}
// "ServiceAvailableIF" implementation --- end

// org/ofono/ModemProxy implementation --- start
void EvolutionGeniviDbusHfpModemCallbackIf::onGetPropertiesError(const ::boost::shared_ptr< ModemProxy >& proxy, const ::boost::shared_ptr< GetPropertiesError >& error)
{
   /*--- BTS_FB_CONNECTION start ----------------------------------------------------------------*/
   BTSFunctionBlock destFuncBlock = BTS_FB_CONNECTION;

   Ipc2Bts_GetModemProperties* ptrResultMsg = ptrNew_Ipc2Bts_GetModemProperties();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setBtsDestinationFunctionBlock(destFuncBlock);
      ptrResultMsg->setModem(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
   /*--- BTS_FB_CONNECTION end ------------------------------------------------------------------*/
}

void EvolutionGeniviDbusHfpModemCallbackIf::onGetPropertiesResponse(const ::boost::shared_ptr< ModemProxy >& proxy, const ::boost::shared_ptr< GetPropertiesResponse >& response)
{
   /*--- BTS_FB_CONNECTION start ----------------------------------------------------------------*/
   BTSFunctionBlock destFuncBlock = BTS_FB_CONNECTION;

   Ipc2Bts_GetModemProperties* ptrResultMsg = ptrNew_Ipc2Bts_GetModemProperties();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setBtsDestinationFunctionBlock(destFuncBlock);
      ptrResultMsg->setModem(proxy->getDBusObjectPath());

      // check received properties for Interfaces
      ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
      parser.setTraces(true);
      BTSDbusPropertyList outPropertyList;

      if(true == parser.findAndParseProperty(outPropertyList, response->getPropertiesMutable(), parser.getHfpModemProperty2String(::ccdbusif::evolution::HFP_MODEM_INTERFACES), (int)::ccdbusif::evolution::IF_MODEM))
      {
         if(1 == outPropertyList.size())
         {
            const ::ccdbusif::DbusVariantProperty& outProperty = outPropertyList[0];

            handleInterfaces(destFuncBlock, proxy->getDBusObjectPath(), outProperty, ::ccdbusif::DEFAULT_ACT);
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }
      }
   }
   onResponse(ptrResultMsg, response->getAct());
   /*--- BTS_FB_CONNECTION end ------------------------------------------------------------------*/
}

void EvolutionGeniviDbusHfpModemCallbackIf::onSetPropertyError(const ::boost::shared_ptr< ModemProxy >& proxy, const ::boost::shared_ptr< SetPropertyError >& error)
{
   // TODO: implement
   (void)(proxy);
   (void)(error);
}

void EvolutionGeniviDbusHfpModemCallbackIf::onSetPropertyResponse(const ::boost::shared_ptr< ModemProxy >& proxy, const ::boost::shared_ptr< SetPropertyResponse >& response)
{
   // TODO: implement
   (void)(proxy);
   (void)(response);
}

void EvolutionGeniviDbusHfpModemCallbackIf::onAcceptSCOConnectError(const ::boost::shared_ptr< ModemProxy >& proxy, const ::boost::shared_ptr< AcceptSCOConnectError >& error)
{
   // TODO: implement
   (void)(proxy);
   (void)(error);
}

void EvolutionGeniviDbusHfpModemCallbackIf::onAcceptSCOConnectResponse(const ::boost::shared_ptr< ModemProxy >& proxy, const ::boost::shared_ptr< AcceptSCOConnectResponse >& response)
{
   // TODO: implement
   (void)(proxy);
   (void)(response);
}

void EvolutionGeniviDbusHfpModemCallbackIf::onPropertyChangedError(const ::boost::shared_ptr< ModemProxy >& proxy, const ::boost::shared_ptr< PropertyChangedError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusHfpModemCallbackIf::onPropertyChangedSignal(const ::boost::shared_ptr< ModemProxy >& proxy, const ::boost::shared_ptr< PropertyChangedSignal >& signal)
{
   /*--- BTS_FB_CONNECTION start ----------------------------------------------------------------*/
   BTSFunctionBlock destFuncBlock = BTS_FB_CONNECTION;

   // check received property for Interfaces
   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
   parser.setTraces(true);
   ::ccdbusif::evolution::HfpModemProperty propEnum = parser.getHfpModemProperty2Enum(signal->getName());

   if(::ccdbusif::evolution::HFP_MODEM_INTERFACES == propEnum)
   {
      BTSDbusPropertyList outPropertyList;

      parser.parseProperty(outPropertyList, signal->getValueMutable(), signal->getName(), (int)::ccdbusif::evolution::IF_MODEM);

      if(1 == outPropertyList.size())
      {
         const ::ccdbusif::DbusVariantProperty& outProperty = outPropertyList[0];

         handleInterfaces(destFuncBlock, proxy->getDBusObjectPath(), outProperty, signal->getAct());
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
   /*--- BTS_FB_CONNECTION end ------------------------------------------------------------------*/
}

void EvolutionGeniviDbusHfpModemCallbackIf::onSCOConnectRequestError(const ::boost::shared_ptr< ModemProxy >& proxy, const ::boost::shared_ptr< SCOConnectRequestError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusHfpModemCallbackIf::onSCOConnectRequestSignal(const ::boost::shared_ptr< ModemProxy >& proxy, const ::boost::shared_ptr< SCOConnectRequestSignal >& signal)
{
   // TODO: implement
   (void)(proxy);
   (void)(signal);
}
// org/ofono/ModemProxy implementation --- end

void EvolutionGeniviDbusHfpModemCallbackIf::handleInterfaces(const BTSFunctionBlock destFuncBlock, const ::std::string& modemObjPath, const ::ccdbusif::DbusVariantProperty& outProperty, const act_t token)
{
   /*--- BTS_FB_CONNECTION start ----------------------------------------------------------------*/
   if(BTS_FB_CONNECTION == destFuncBlock)
   {
      Ipc2Bts_ModemInterfacesUpdate* ptrInterfacesUpd = ptrNew_Ipc2Bts_ModemInterfacesUpdate();
      if(NULL != ptrInterfacesUpd)
      {
         ptrInterfacesUpd->setBtsDestinationFunctionBlock(destFuncBlock);
         ptrInterfacesUpd->setModem(modemObjPath);
         BTSDbusInterfaceList& interfaces = ptrInterfacesUpd->getInterfacesMutable();
         bool handsfreeFound = false;

         if(false == outProperty.propData.getNone())
         {
            const ::std::vector<std::string>& strArray = outProperty.propData.getStringArray();
            interfaces.reserve(strArray.size());
            ::ccdbusif::evolution::Interface item;
            ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;

            for(size_t j = 0; j < strArray.size(); j++)
            {
               item = parser.getInterface2Enum(strArray[j]);
               interfaces.push_back(item);

               if(::ccdbusif::evolution::IF_HANDSFREE == item)
               {
                  handsfreeFound = true;
               }
            }
         }

         if(true == handsfreeFound)
         {
            // send mapping
            Ipc2Bts_AddHfpInterfaceObjectPathMapping* ptrMapping = ptrNew_Ipc2Bts_AddHfpInterfaceObjectPathMapping();
            if(NULL != ptrMapping)
            {
               ptrMapping->setBtsDestinationFunctionBlock(destFuncBlock);
               ptrMapping->setInterface(::ccdbusif::evolution::IF_HANDSFREE);
               ptrMapping->setObjPath(modemObjPath);
            }
            onSignal(ptrMapping, token, true);

            // create proxy
            if(NULL != _hfpHandsfreeIf)
            {
               _hfpHandsfreeIf->deviceCreated(modemObjPath);
            }
         }
         else
         {
            // destroy proxy
            if(NULL != _hfpHandsfreeIf)
            {
               _hfpHandsfreeIf->deviceRemoved(modemObjPath, true);
            }

            // send removed mapping
            Ipc2Bts_DelHfpInterfaceObjectPathMapping* ptrMapping = ptrNew_Ipc2Bts_DelHfpInterfaceObjectPathMapping();
            if(NULL != ptrMapping)
            {
               ptrMapping->setBtsDestinationFunctionBlock(destFuncBlock);
               ptrMapping->setInterface(::ccdbusif::evolution::IF_HANDSFREE);
               ptrMapping->setObjPath(modemObjPath);
               ptrMapping->setObjectId(modemObjPath);
            }
            onSignal(ptrMapping, token);
         }
      }
      onUpdate(ptrInterfacesUpd, token);
   }
   /*--- BTS_FB_CONNECTION end ------------------------------------------------------------------*/
}

} //genivi
} //btstackif
