/**
 * @file EvolutionGeniviDbusHfpManagerCallbackIf.cpp
 *
 * @par SW-Component
 * BtStackIf
 *
 * @brief EvolutionGenivi DBUS Callback Interface for HfpManager.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS Callback Interface for HfpManager.
 */

#include "EvolutionGeniviDbusHfpManagerCallbackIf.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "BtsUtils.h"
#include "EvolutionGeniviUtils.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"
#include "cc_dbus_if/IEvolutionGeniviDbusHfpHandsfreeSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusHfpModemSendRequestIf.h"

#include "FwAssert.h"

using namespace ::org::ofono::Manager;
using namespace ::asf::core;

namespace btstackif {
namespace genivi {

EvolutionGeniviDbusHfpManagerCallbackIf::EvolutionGeniviDbusHfpManagerCallbackIf()
{
   _hfpHandsfreeIf = NULL;
   _hfpModemIf = NULL;
}

EvolutionGeniviDbusHfpManagerCallbackIf::EvolutionGeniviDbusHfpManagerCallbackIf(IDbusRecHandler* recHandler) : DbusCallbackIf(recHandler)
{
   _hfpHandsfreeIf = NULL;
   _hfpModemIf = NULL;
}

EvolutionGeniviDbusHfpManagerCallbackIf::~EvolutionGeniviDbusHfpManagerCallbackIf()
{
   _hfpHandsfreeIf = NULL;
   _hfpModemIf = NULL;
}

// "ServiceAvailableIF" implementation --- start
void EvolutionGeniviDbusHfpManagerCallbackIf::onProxyAvailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);
   (void)(objPath);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   /*--- BTS_FB_CONNECTION start ----------------------------------------------------------------*/
   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_HFP_MANAGER);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
   /*--- BTS_FB_CONNECTION end ------------------------------------------------------------------*/
}

void EvolutionGeniviDbusHfpManagerCallbackIf::onProxyUnavailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);
   (void)(objPath);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   /*--- BTS_FB_CONNECTION start ----------------------------------------------------------------*/
   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_HFP_MANAGER);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
   /*--- BTS_FB_CONNECTION end ------------------------------------------------------------------*/
}
// "ServiceAvailableIF" implementation --- end

// org/ofono/ManagerProxy implementation --- start
void EvolutionGeniviDbusHfpManagerCallbackIf::onGetModemsError(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< GetModemsError >& error)
{
   (void)(proxy);
   /*--- BTS_FB_CONNECTION start ----------------------------------------------------------------*/
   BTSFunctionBlock destFuncBlock = BTS_FB_CONNECTION;

   Ipc2Bts_GetModems* ptrResultMsg = ptrNew_Ipc2Bts_GetModems();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setBtsDestinationFunctionBlock(destFuncBlock);
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
   /*--- BTS_FB_CONNECTION end ------------------------------------------------------------------*/
}

void EvolutionGeniviDbusHfpManagerCallbackIf::onGetModemsResponse(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< GetModemsResponse >& response)
{
   ::std::vector< GetModemsResponseModemsStruct >& modems = response->getModemsMutable();
   for(::std::vector< GetModemsResponseModemsStruct >::iterator it = modems.begin(); it != modems.end(); ++it)
   {
      // const ::std::string& getElem1(): object path
      // ::std::map< ::std::string, ::asf::dbus::DBusVariant >& getElem2Mutable(): properties
      /*--- BTS_FB_CONNECTION start ----------------------------------------------------------------*/
      BTSFunctionBlock destFuncBlock = BTS_FB_CONNECTION;
      handleModemAdded(destFuncBlock, it->getElem1(), it->getElem2Mutable(), ::ccdbusif::BUS_TYPE_SYSTEM, proxy->getDBusBusName(), proxy->getDBusObjectPath(), proxy->getInterfaceName(), ::ccdbusif::DEFAULT_ACT);
      /*--- BTS_FB_CONNECTION end ------------------------------------------------------------------*/
   }

   /*--- BTS_FB_CONNECTION start ----------------------------------------------------------------*/
   BTSFunctionBlock destFuncBlock = BTS_FB_CONNECTION;
   Ipc2Bts_GetModems* ptrResultMsg = ptrNew_Ipc2Bts_GetModems();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setBtsDestinationFunctionBlock(destFuncBlock);
   }
   onResponse(ptrResultMsg, response->getAct());
   /*--- BTS_FB_CONNECTION end ------------------------------------------------------------------*/
}

void EvolutionGeniviDbusHfpManagerCallbackIf::onModemAddedError(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< ModemAddedError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusHfpManagerCallbackIf::onModemAddedSignal(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< ModemAddedSignal >& signal)
{
   // HINT: proxy object delivers following data:
   //    const std::string& getDBusBusName() const;
   //    const std::string& getDBusObjectPath() const;
   //    const std::string& getInterfaceName() const;
   //    DBus type: ::ccdbusif::BUS_TYPE_SYSTEM

   /*--- BTS_FB_CONNECTION start ----------------------------------------------------------------*/
   BTSFunctionBlock destFuncBlock = BTS_FB_CONNECTION;
   handleModemAdded(destFuncBlock, signal->getPath(), signal->getPropertiesMutable(), ::ccdbusif::BUS_TYPE_SYSTEM, proxy->getDBusBusName(), proxy->getDBusObjectPath(), proxy->getInterfaceName(), signal->getAct());
   /*--- BTS_FB_CONNECTION end ------------------------------------------------------------------*/
}

void EvolutionGeniviDbusHfpManagerCallbackIf::onModemRemovedError(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< ModemRemovedError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusHfpManagerCallbackIf::onModemRemovedSignal(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< ModemRemovedSignal >& signal)
{
   (void)(proxy);

   /*--- BTS_FB_CONNECTION start ----------------------------------------------------------------*/
   BTSFunctionBlock destFuncBlock = BTS_FB_CONNECTION;

   // destroy proxy
   if(NULL != _hfpModemIf)
   {
      _hfpModemIf->modemRemoved(signal->getPath(), true);
   }

   // send removed mapping
   Ipc2Bts_ModemRemoved* ptrMsg = ptrNew_Ipc2Bts_ModemRemoved();
   if(NULL != ptrMsg)
   {
      ptrMsg->setBtsDestinationFunctionBlock(destFuncBlock);
      ptrMsg->setModem(signal->getPath());
      ptrMsg->setObjectId(signal->getPath());
   }
   onSignal(ptrMsg, signal->getAct());

   // destroy proxy
   if(NULL != _hfpHandsfreeIf)
   {
      _hfpHandsfreeIf->deviceRemoved(signal->getPath(), true);
   }

   // send removed mapping
   Ipc2Bts_DelHfpInterfaceObjectPathMapping* ptrMapping = ptrNew_Ipc2Bts_DelHfpInterfaceObjectPathMapping();
   if(NULL != ptrMapping)
   {
      ptrMapping->setBtsDestinationFunctionBlock(destFuncBlock);
      ptrMapping->setInterface(::ccdbusif::evolution::IF_HANDSFREE);
      ptrMapping->setObjPath(signal->getPath());
      ptrMapping->setObjectId(signal->getPath());
   }
   onSignal(ptrMapping, signal->getAct());
   /*--- BTS_FB_CONNECTION end ------------------------------------------------------------------*/
}
// org/ofono/ManagerProxy implementation --- end

void EvolutionGeniviDbusHfpManagerCallbackIf::handleModemAdded(const BTSFunctionBlock destFuncBlock, const ::std::string& modemObjPath, ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties,
      const ::ccdbusif::DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName, const act_t token)
{
   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
   parser.setTraces(true);

   BTSDbusPropertyList ifProperties;
   ::std::vector<int> matchingInterfaces;
   matchingInterfaces.push_back((int)::ccdbusif::evolution::IF_MODEM);

   parser.parseProperties(ifProperties, matchingInterfaces, inProperties, busType, busName, objPath, interfaceName);

   /*--- BTS_FB_CONNECTION start ----------------------------------------------------------------*/
   if(BTS_FB_CONNECTION == destFuncBlock)
   {
      // send mapping
      BTSBDAddress address;
      for(size_t i = 0; i < ifProperties.size(); i++)
      {
         if(::ccdbusif::evolution::HFP_MODEM_SERIAL == (::ccdbusif::evolution::HfpModemProperty)ifProperties[i].propEnum)
         {
            convertBdAddress2InternalValue(address, ifProperties[i].propData.getString());
            break;
         }
      }

      Ipc2Bts_ModemAdded* ptrMsg = ptrNew_Ipc2Bts_ModemAdded();
      if(NULL != ptrMsg)
      {
         ptrMsg->setBtsDestinationFunctionBlock(destFuncBlock);
         ptrMsg->setModem(modemObjPath);
         ptrMsg->setBDAddress(address);
      }
      onSignal(ptrMsg, token, true);

      // create proxy
      if(NULL != _hfpModemIf)
      {
         _hfpModemIf->modemCreated(modemObjPath);
      }

      // send data
      Ipc2Bts_ModemInterfacesUpdate* ptrInterfacesUpd = ptrNew_Ipc2Bts_ModemInterfacesUpdate();
      if(NULL != ptrInterfacesUpd)
      {
         ptrInterfacesUpd->setBtsDestinationFunctionBlock(destFuncBlock);
         ptrInterfacesUpd->setModem(modemObjPath);
         BTSDbusInterfaceList& interfaces = ptrInterfacesUpd->getInterfacesMutable();

         for(size_t i = 0; i < ifProperties.size(); i++)
         {
            if(::ccdbusif::evolution::HFP_MODEM_INTERFACES == (::ccdbusif::evolution::HfpModemProperty)ifProperties[i].propEnum)
            {
               if(false == ifProperties[i].propData.getNone())
               {
                  const ::std::vector<std::string>& strArray = ifProperties[i].propData.getStringArray();
                  interfaces.reserve(strArray.size());
                  ::ccdbusif::evolution::Interface item;

                  for(size_t j = 0; j < strArray.size(); j++)
                  {
                     item = parser.getInterface2Enum(strArray[j]);
                     interfaces.push_back(item);

                     if(::ccdbusif::evolution::IF_HANDSFREE == item)
                     {
                        // send mapping
                        Ipc2Bts_AddHfpInterfaceObjectPathMapping* ptrMapping = ptrNew_Ipc2Bts_AddHfpInterfaceObjectPathMapping();
                        if(NULL != ptrMapping)
                        {
                           ptrMapping->setBtsDestinationFunctionBlock(destFuncBlock);
                           ptrMapping->setInterface(::ccdbusif::evolution::IF_HANDSFREE);
                           ptrMapping->setObjPath(modemObjPath);
                        }
                        onSignal(ptrMapping, token, true);

                        // create proxy
                        if(NULL != _hfpHandsfreeIf)
                        {
                           _hfpHandsfreeIf->deviceCreated(modemObjPath);
                        }
                     }
                  }
               }

               break;
            }
         }
      }
      onSignal(ptrInterfacesUpd, token);
   }
   /*--- BTS_FB_CONNECTION end ------------------------------------------------------------------*/
}

} //genivi
} //btstackif
