/**
 * @file EvolutionGeniviDbusDeviceCallbackIf.cpp
 *
 * @par SW-Component
 * BtStackIf
 *
 * @brief EvolutionGenivi DBUS Callback Interface for Device.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS Callback Interface for Device.
 */

#include "EvolutionGeniviDbusDeviceCallbackIf.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "BtsUtils.h"
#include "EvolutionGeniviUtils.h"
#include "cc_dbus_if/EvolutionGeniviUtility.h"
#include "FwAssert.h"

using namespace ::org::bluez::Device1;
using namespace ::asf::core;

namespace btstackif {
namespace genivi {

EvolutionGeniviDbusDeviceCallbackIf::EvolutionGeniviDbusDeviceCallbackIf()
{
}

EvolutionGeniviDbusDeviceCallbackIf::EvolutionGeniviDbusDeviceCallbackIf(IDbusRecHandler* recHandler) : DbusCallbackIf(recHandler)
{
}

EvolutionGeniviDbusDeviceCallbackIf::~EvolutionGeniviDbusDeviceCallbackIf()
{
}

// "ServiceAvailableIF" implementation --- start
void EvolutionGeniviDbusDeviceCallbackIf::onProxyAvailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_DEVICE);
      ptrMsg->setObjPath(objPath);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}

void EvolutionGeniviDbusDeviceCallbackIf::onProxyUnavailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_DEVICE);
      ptrMsg->setObjPath(objPath);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}
// "ServiceAvailableIF" implementation --- end

// org/bluez/Device1Proxy implementation --- start
void EvolutionGeniviDbusDeviceCallbackIf::onConnectError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< ConnectError >& error)
{
   Ipc2Bts_DeviceConnectResult* ptrResultMsg = ptrNew_Ipc2Bts_DeviceConnectResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onConnectResponse(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< ConnectResponse >& response)
{
   Ipc2Bts_DeviceConnectResult* ptrResultMsg = ptrNew_Ipc2Bts_DeviceConnectResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onResponse(ptrResultMsg, response);
}

void EvolutionGeniviDbusDeviceCallbackIf::onDisconnectError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< DisconnectError >& error)
{
   Ipc2Bts_DeviceDisconnectResult* ptrResultMsg = ptrNew_Ipc2Bts_DeviceDisconnectResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onDisconnectResponse(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< DisconnectResponse >& response)
{
   Ipc2Bts_DeviceDisconnectResult* ptrResultMsg = ptrNew_Ipc2Bts_DeviceDisconnectResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onResponse(ptrResultMsg, response);
}

void EvolutionGeniviDbusDeviceCallbackIf::onPairError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< PairError >& error)
{
   Ipc2Bts_PairResult* ptrResultMsg = ptrNew_Ipc2Bts_PairResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onPairResponse(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< PairResponse >& response)
{
   Ipc2Bts_PairResult* ptrResultMsg = ptrNew_Ipc2Bts_PairResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onResponse(ptrResultMsg, response);
}

void EvolutionGeniviDbusDeviceCallbackIf::onCancelPairingError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< CancelPairingError >& error)
{
   Ipc2Bts_CancelPairingResult* ptrResultMsg = ptrNew_Ipc2Bts_CancelPairingResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onCancelPairingResponse(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< CancelPairingResponse >& response)
{
   Ipc2Bts_CancelPairingResult* ptrResultMsg = ptrNew_Ipc2Bts_CancelPairingResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onResponse(ptrResultMsg, response);
}

void EvolutionGeniviDbusDeviceCallbackIf::onDiscoverServicesError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< DiscoverServicesError >& error)
{
   Ipc2Bts_DiscoverServicesResult* ptrResultMsg = ptrNew_Ipc2Bts_DiscoverServicesResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onDiscoverServicesResponse(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< DiscoverServicesResponse >& response)
{
   Ipc2Bts_DiscoverServicesResult* ptrResultMsg = ptrNew_Ipc2Bts_DiscoverServicesResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onResponse(ptrResultMsg, response);
}

void EvolutionGeniviDbusDeviceCallbackIf::onCancelDiscoveryError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< CancelDiscoveryError >& error)
{
   Ipc2Bts_CancelDiscoveryResult* ptrResultMsg = ptrNew_Ipc2Bts_CancelDiscoveryResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onCancelDiscoveryResponse(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< CancelDiscoveryResponse >& response)
{
   Ipc2Bts_CancelDiscoveryResult* ptrResultMsg = ptrNew_Ipc2Bts_CancelDiscoveryResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onResponse(ptrResultMsg, response);
}

void EvolutionGeniviDbusDeviceCallbackIf::onReportLinkkeyError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< ReportLinkkeyError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDeviceCallbackIf::onReportLinkkeySignal(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< ReportLinkkeySignal >& signal)
{
   Ipc2Bts_ReportLinkkey* ptrResultMsg = ptrNew_Ipc2Bts_ReportLinkkey();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      BTSLinkKey linkKey;
      convertLinkKeyBinary2String(linkKey, signal->getLinkkey(), true); // link key can be empty
      ptrResultMsg->setLinkKey(linkKey);
   }
   onSignal(ptrResultMsg, signal->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onIncomingPairingCompError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< IncomingPairingCompError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDeviceCallbackIf::onIncomingPairingCompSignal(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< IncomingPairingCompSignal >& signal)
{
   Ipc2Bts_IncomingPairingComp* ptrResultMsg = ptrNew_Ipc2Bts_IncomingPairingComp();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setStatus(convertPairingStatusCode2InternalValue(signal->getStatus()));
   }
   onSignal(ptrResultMsg, signal->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onAddressError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< AddressError >& error)
{
   Ipc2Bts_DeviceAddressUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceAddressUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onAddressUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< AddressUpdate >& update)
{
   Ipc2Bts_DeviceAddressUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceAddressUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setBDAddress(update->getAddress());
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onClassError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< ClassError >& error)
{
   Ipc2Bts_DeviceClassUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceClassUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onClassUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< ClassUpdate >& update)
{
   Ipc2Bts_DeviceClassUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceClassUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setCod(update->getClass());
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onIconError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< IconError >& error)
{
   Ipc2Bts_DeviceIconUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceIconUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onIconUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< IconUpdate >& update)
{
   Ipc2Bts_DeviceIconUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceIconUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setIcon(update->getIcon());
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onUUIDsError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< UUIDsError >& error)
{
   Ipc2Bts_DeviceUuidsUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceUuidsUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onUUIDsUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< UUIDsUpdate >& update)
{
   Ipc2Bts_DeviceUuidsUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceUuidsUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());

      BTSSupportedServices& serviceList = ptrResultMsg->getSupportedServicesMutable();
      BTSUuidList& uuidList = ptrResultMsg->getUuidListMutable();
      const ::std::vector< ::std::string >& uuids = update->getUUIDs();
      ::std::string convertedUuid;

      for(size_t i = 0; i < uuids.size(); i++)
      {
         BTSSupportedServicesBit supportedServicesBit(convertUuid2SupportedService(convertedUuid, uuids[i]));

         if(BTS_SUPP_SRV_LAST != supportedServicesBit)
         {
            serviceList.setBit(supportedServicesBit);

            if(BTS_SUPP_SRV_SPP == supportedServicesBit)
            {
               uuidList.push_back(convertedUuid);
            }
         }

         convertedUuid.clear();
      }
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onPairedError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< PairedError >& error)
{
   Ipc2Bts_DevicePairedUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DevicePairedUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onPairedUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< PairedUpdate >& update)
{
   Ipc2Bts_DevicePairedUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DevicePairedUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setPaired(update->getPaired());
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onConnectedError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< ConnectedError >& error)
{
   Ipc2Bts_DeviceConnectedUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceConnectedUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onConnectedUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< ConnectedUpdate >& update)
{
   Ipc2Bts_DeviceConnectedUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceConnectedUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setConnected(update->getConnected());
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onTrustedError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< TrustedError >& error)
{
   Ipc2Bts_DeviceTrustedUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceTrustedUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onTrustedUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< TrustedUpdate >& update)
{
   Ipc2Bts_DeviceTrustedUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceTrustedUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setTrusted(update->getTrusted());
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onBlockedError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< BlockedError >& error)
{
   Ipc2Bts_DeviceBlockedUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceBlockedUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onBlockedUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< BlockedUpdate >& update)
{
   Ipc2Bts_DeviceBlockedUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceBlockedUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setBlocked(update->getBlocked());
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onAliasError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< AliasError >& error)
{
   Ipc2Bts_DeviceAliasUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceAliasUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onAliasUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< AliasUpdate >& update)
{
   Ipc2Bts_DeviceAliasUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceAliasUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setAlias(update->getAlias());
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onAdapterError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< AdapterError >& error)
{
   Ipc2Bts_DeviceAdapterUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceAdapterUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onAdapterUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< AdapterUpdate >& update)
{
   Ipc2Bts_DeviceAdapterUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceAdapterUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setAdapter(update->getAdapter());
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onModaliasError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< ModaliasError >& error)
{
   Ipc2Bts_DeviceModaliasUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceModaliasUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onModaliasUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< ModaliasUpdate >& update)
{
   Ipc2Bts_DeviceModaliasUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceModaliasUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setModalias(update->getModalias());
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onRSSIError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< RSSIError >& error)
{
   Ipc2Bts_DeviceRssiUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceRssiUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onRSSIUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< RSSIUpdate >& update)
{
   Ipc2Bts_DeviceRssiUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceRssiUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setRssi((BTSRssi)update->getRSSI());
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onVendorError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< VendorError >& error)
{
   Ipc2Bts_DeviceVendorIdUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceVendorIdUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onVendorUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< VendorUpdate >& update)
{
   Ipc2Bts_DeviceVendorIdUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceVendorIdUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setVendorId(update->getVendor());
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onVendorSourceError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< VendorSourceError >& error)
{
   Ipc2Bts_DeviceVendorIdSourceUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceVendorIdSourceUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onVendorSourceUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< VendorSourceUpdate >& update)
{
   Ipc2Bts_DeviceVendorIdSourceUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceVendorIdSourceUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setVendorIdSource(update->getVendorSource());
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onProductError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< ProductError >& error)
{
   Ipc2Bts_DeviceProductIdUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceProductIdUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onProductUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< ProductUpdate >& update)
{
   Ipc2Bts_DeviceProductIdUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceProductIdUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setProductId(update->getProduct());
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onMapInfoError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< MapInfoError >& error)
{
   Ipc2Bts_DeviceMapInfoUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceMapInfoUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onMapInfoUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< MapInfoUpdate >& update)
{
   /*
      dict entry(
         string "MapInfo"
         variant             array [
               string "(remote_instance:0x0,version:0x100,support_type:0x2,name:Android SMS,support_feature:0xffffffff,reserved:0x0)"
            ]
      )
      dict entry(
         string "MapInfo"
         variant             array [
               string "(remote_instance:0x0,version:0x102,support_type:0xe,name:RPI MAS Instance No 0,support_feature:0x7f,reserved:0x0)"
               string "(remote_instance:0x1,version:0x102,support_type:0x1,name:RPI MAS Instance No 1,support_feature:0x7f,reserved:0x0)"
               string "(remote_instance:0x2,version:0x102,support_type:0xf,name:RPI MAS Instance No 2,support_feature:0x7f,reserved:0x0)"
            ]
      )
    */

   Ipc2Bts_DeviceMapInfoUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceMapInfoUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      BTSMapSdpRecordList& recordList = ptrResultMsg->getMapInfoListMutable();
      const ::std::vector< ::std::string >& mapInfo = update->getMapInfo();
      bool valid = false;
      BTSMapSdpRecord record;
      unsigned char types;
      unsigned int features;
      for(size_t i = 0; i < mapInfo.size(); i++)
      {
         record.version = 0;
         record.instanceId = 0;
         record.name.clear();
         types = 0;
         features = 0;
         if(true == ::ccdbusif::evolution_genivi_utility::extractMapData(record.version, record.instanceId, record.name, types, features, mapInfo[i]))
         {
            valid = true;
            if(0xFFFFFFFF == features)
            {
               // features are unknown
               features = 0;
            }
            record.messageTypes.setData(types);
            record.features.setData(features);
            recordList.push_back(record);
         }
      }
      ptrResultMsg->setValid(valid);
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onConnectedServiceError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< ConnectedServiceError >& error)
{
   Ipc2Bts_DeviceConnectedServiceUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceConnectedServiceUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onConnectedServiceUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< ConnectedServiceUpdate >& update)
{
   Ipc2Bts_DeviceConnectedServiceUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceConnectedServiceUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      BTSServiceFunction serviceFunction;
      serviceFunction.setData(update->getConnectedService());
      ptrResultMsg->setConnectedService(serviceFunction);
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onVersionError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< VersionError >& error)
{
   Ipc2Bts_DeviceVersionUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceVersionUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onVersionUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< VersionUpdate >& update)
{
   Ipc2Bts_DeviceVersionUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceVersionUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setVersion(update->getVersion());
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onPBAPInfoError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< PBAPInfoError >& error)
{
   Ipc2Bts_DevicePbapInfoUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DevicePbapInfoUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onPBAPInfoUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< PBAPInfoUpdate >& update)
{
   /*
      dict entry(
         string "PBAPInfo"
         variant             string "version:0x102,support_repositories:0xb,support_feature:0x3ff"
      )
    */

   Ipc2Bts_DevicePbapInfoUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DevicePbapInfoUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      BTSPbapSdpRecord& record = ptrResultMsg->getPbapInfoMutable();
      unsigned char repositories = 0;
      unsigned int features = 0;
      bool valid = ::ccdbusif::evolution_genivi_utility::extractPbapData(record.version, repositories, features, update->getPBAPInfo());
      if(0xFFFFFFFF == features)
      {
         // features are unknown
         features = 0;
      }
      record.repositories.setData(repositories);
      record.features.setData(features);
      ptrResultMsg->setValid(valid);
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onModeError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< ModeError >& error)
{
   Ipc2Bts_DeviceModeUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceModeUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onModeUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< ModeUpdate >& update)
{
   Ipc2Bts_DeviceModeUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceModeUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setMode((BTSBtPowerMode)update->getMode()); // cast OK because enum and Evolution value match
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onRoleError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< RoleError >& error)
{
   Ipc2Bts_DeviceRoleUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceRoleUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onRoleUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< RoleUpdate >& update)
{
   Ipc2Bts_DeviceRoleUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceRoleUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setMaster((0x00 == update->getRole()));
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onSDPProcessReportError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< SDPProcessReportError >& error)
{
   Ipc2Bts_SDPProcessReport* ptrResultMsg = ptrNew_Ipc2Bts_SDPProcessReport();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onSDPProcessReportSignal(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< SDPProcessReportSignal >& signal)
{
   Ipc2Bts_SDPProcessReport* ptrResultMsg = ptrNew_Ipc2Bts_SDPProcessReport();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setActive((0x01 == signal->getStatus()));
   }
   onSignal(ptrResultMsg, signal->getAct());
}

void EvolutionGeniviDbusDeviceCallbackIf::onEnableAvpPauseError(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< EnableAvpPauseError >& error)
{
   Ipc2Bts_DeviceEnableAvpPauseUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceEnableAvpPauseUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDeviceCallbackIf::onEnableAvpPauseUpdate(const ::boost::shared_ptr< Device1Proxy >& proxy, const ::boost::shared_ptr< EnableAvpPauseUpdate >& update)
{
   Ipc2Bts_DeviceEnableAvpPauseUpdate* ptrResultMsg = ptrNew_Ipc2Bts_DeviceEnableAvpPauseUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
      ptrResultMsg->setEnableAvpPause(update->getEnableAvpPause());
   }
   onUpdate(ptrResultMsg, update->getAct());
}
// org/bluez/Device1Proxy implementation --- end

} //genivi
} //btstackif
